from xml.etree import ElementTree as et

from sandbox.projects.ott.backend.common import utils

class TeamCityClient(object):
    TEAMCITY_URL = 'https://teamcity.yandex-team.ru'
    TEAMCITY_REST_URL = '{}/app/rest'.format(TEAMCITY_URL)

    def __init__(self, auth=None):
        self._session = utils.createSession()
        self._session.headers['Authorization'] = 'OAuth {}'.format(auth)

    def trigger_build(self, build_conf, params=None):
        build_el = et.Element('build')
        et.SubElement(build_el, 'buildType').set('id', build_conf)
        if params:
            properties_el = et.SubElement(build_el, 'properties')
            for param in params:
                property_el = et.SubElement(properties_el, 'property')
                property_el.set('name', param)
                property_el.set('value', str(params[param]))

        resp = self._session.post(
            '{}/buildQueue'.format(self.TEAMCITY_REST_URL),
            data=et.tostring(build_el),
            headers={
                'Content-Type': 'application/xml',
                'Origin': 'http://teamcity.yandex-team.ru:443',
            })
        resp.raise_for_status()
        return et.fromstring(resp.text.encode('utf-8'))

    def build_details(self, build_id):
        resp = self._session.get('{}/builds/id:{}'.format(self.TEAMCITY_REST_URL, build_id))
        resp.raise_for_status()
        tree = et.fromstring(resp.text.encode('utf-8'))
        return tree

    def build_url(self, build_id):
        return '{}/viewLog.html?buildId={}'.format(self.TEAMCITY_URL, build_id)

    def cancel_queue(self, build_id, task_id):
        cancel_el = et.Element('buildCancelRequest')
        cancel_el.set('comment', 'SB task {} is stopped'.format(task_id))
        cancel_el.set('readdIntoQueue', 'false')

        response = self._session.post(
            '{}/buildQueue/id:{}'.format(self.TEAMCITY_REST_URL, build_id),
            data=et.tostring(cancel_el),
            headers={
                'Content-Type': 'application/xml',
                'Origin': 'http://teamcity.yandex-team.ru:443',
            })
        response.raise_for_status()

    def cancel_build(self, build_id, task_id):
        cancel_el = et.Element('buildCancelRequest')
        cancel_el.set('comment', 'SB task {} is stopped'.format(task_id))
        cancel_el.set('readdIntoQueue', 'false')

        response = self._session.post(
            '{}/builds/id:{}'.format(self.TEAMCITY_REST_URL, build_id),
            data=et.tostring(cancel_el),
            headers={
                'Content-Type': 'application/xml',
                'Origin': 'http://teamcity.yandex-team.ru:443',
            })
        response.raise_for_status()

    def build_list(self, build_id):
        url = '{}/builds?locator=snapshotDependency:(to:(id:{}),includeInitial:true),defaultFilter:false'.format(self.TEAMCITY_REST_URL, build_id)
        resp = self._session.get(url)
        resp.raise_for_status()
        tree = et.fromstring(resp.text.encode('utf-8'))
        return tree
