import logging
import sys
from datetime import datetime

import click
from ott.drm.library.python.cms.clients import VhAdminClient, OttContentApiClient
from ott.drm.library.python.packager_task.clients import PackagerTasksApiClient
from sandbox.projects.ott.packager_management_system.lib import (
    graph_creator,
    launcher,
    watcher,
    importer,
    exporter,
    publisher,
    notifier
)

from sandbox.projects.ott.packager_management_system.lib.graph_creator.ott_packager import (
    OttPackager,
    DummyOttPackagerRepository,
    OttPackagerReleaseStatus,
    OttPackagerResourceAttrs
)

services = {'EXPORTER', 'GRAPH_CREATOR', 'IMPORTER', 'LAUNCHER', 'WATCHER', 'PUBLISHER', 'NOTIFIER'}


@click.command('run_service')
@click.option('--service', type=click.Choice(services), help='Service to launch', required=True)
@click.option('--packager-tasks-api-url', type=click.STRING, help='ott-packager-tasks-api url')
@click.option('--nirvana-oauth-token', type=click.STRING, help='Nirvana oauth-token')
@click.option('--ott-packager-bin', type=click.STRING, help='Path to ott-packager-bin')
@click.option('--vh-admin-url', type=click.STRING, help='vh-admin url')
@click.option('--vod-provider-id', type=click.STRING, help='VodProviderId')
@click.option('--vh-admin-timeout-sec', type=click.FLOAT, help='vh-admin timeout in seconds', default=40)
@click.option('--max_running_graphs', type=click.INT, default=5)
@click.option('--max_not_launched_graphs', type=click.INT, default=5)
@click.option('--max_workers', type=click.INT, default=1)
@click.option('--u_duty_project', type=click.STRING, default='ott-encoder')
@click.option('--vh_frontend_url', type=click.STRING, default='https://vh.yandex-team.ru')
@click.option('--nirvana_quota', type=click.STRING, default='ott-encoder-empty')
@click.option('--vod-providers', type=click.STRING, multiple=True)
@click.option('--ott-content-api-url', type=click.STRING, help='ott-content-api url')
@click.option('--ott-content-api-timeout-sec', type=click.FLOAT, help='ott-content-api timeout in secs', default=5)
@click.option('--s3-creds-nirvana-secret-name', type=click.STRING, default='robot-ott-packager-s3-creds')
@click.option('--s3-creds', type=click.STRING)
@click.option('--parallel-graph-launch-delay-sec', type=click.INT, default=60)
def run_service(service, packager_tasks_api_url, nirvana_oauth_token, ott_packager_bin, vh_admin_url,
                vh_admin_timeout_sec, vod_provider_id, max_running_graphs, max_not_launched_graphs,
                max_workers, u_duty_project, vh_frontend_url, nirvana_quota, vod_providers,
                ott_content_api_url, ott_content_api_timeout_sec, s3_creds_nirvana_secret_name, s3_creds,
                parallel_graph_launch_delay_sec):
    if service == 'GRAPH_CREATOR':
        ott_packager = OttPackager(
            ott_packager_bin,
            OttPackagerResourceAttrs(datetime.now(), 'unstable', OttPackagerReleaseStatus.UNSTABLE)
        )

        graph_creator.GraphCreator(
            tasks_client=PackagerTasksApiClient(packager_tasks_api_url),
            ott_packager_repository=DummyOttPackagerRepository(ott_packager),
            max_graph_creating_workers=max_workers,
            max_not_launched_graphs=max_not_launched_graphs,
            nirvana_oauth_token=nirvana_oauth_token,
            nirvana_quota=nirvana_quota,
            vod_providers=vod_providers,
            s3_creds_nirvana_secret_name=s3_creds_nirvana_secret_name,
            s3_creds=s3_creds
        ).run()
    elif service == 'LAUNCHER':
        launcher.Launcher(packager_tasks_api_url, nirvana_oauth_token, max_running_graphs,
                          nirvana_quota, vod_providers, parallel_graph_launch_delay_sec).run()
    elif service == 'WATCHER':
        watcher.Watcher(packager_tasks_api_url, nirvana_oauth_token).run()
    elif service == 'IMPORTER':
        importer.Importer([vod_provider_id], packager_tasks_api_url, vh_admin_url, vh_admin_timeout_sec,
                          default_nirvana_quota=nirvana_quota).run()
    elif service == 'EXPORTER':
        exporter.Exporter(packager_tasks_api_url, vh_admin_url, vh_admin_timeout_sec, max_workers).run()
    elif service == 'PUBLISHER':
        publisher.Publisher(
            packager_tasks_api_url,
            VhAdminClient(vh_admin_url, vh_admin_timeout_sec),
            OttContentApiClient(ott_content_api_url, ott_content_api_timeout_sec),
            max_workers=max_workers
        ).run()
    elif service == 'NOTIFIER':
        notifier.Notifier(packager_tasks_api_url, vh_admin_url, vh_admin_timeout_sec, u_duty_project,
                          vh_frontend_url, nirvana_oauth_token, max_workers).run()


@click.group()
def cli():
    pass


def main():
    _setup_logger()
    cli.add_command(run_service)
    cli()


def _setup_logger():
    root = logging.getLogger()
    root.setLevel(logging.INFO)

    handler = logging.StreamHandler(sys.stdout)
    handler.setLevel(logging.INFO)
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    handler.setFormatter(formatter)
    root.addHandler(handler)


if __name__ == '__main__':
    main()
