import logging
import os
from time import sleep

from ydb import DriverConfig, Driver, ExportClient, ExportProgress, ExportToYTSettings


class YdbExporter:
    def __init__(self, database: str, ydb_endpoint: str, ydb_oauth: str, ydb_timeout: float, yt_dir: str, yt_token: str,
                 yt_proxy: str, yt_port: int, export_retries: int, export_timeout: float):
        self.database = database
        self.driver_config = DriverConfig(endpoint=ydb_endpoint, database=database, auth_token=ydb_oauth)
        self.ydb_timeout = ydb_timeout
        self.yt_dir = yt_dir
        self.yt_token = yt_token
        self.yt_proxy = yt_proxy
        self.yt_port = yt_port
        self.export_retries = export_retries
        self.export_timeout = export_timeout

    def export_to_yt(self):
        with Driver(driver_config=self.driver_config) as driver:
            logging.info('Wait driver to connect')
            driver.wait(timeout=self.ydb_timeout)
            logging.info('Driver was successfully connected to ydb')

            export_client = ExportClient(driver)
            export_to_yt_settings = (
                ExportToYTSettings()
                .with_token(token=self.yt_token)
                .with_host(host=self.yt_proxy)
                .with_port(port=self.yt_port)
                .with_number_of_retries(self.export_retries)
                .with_timeout(self.export_timeout)
            )

            logging.info('Retrieve tables')
            tables = []
            for children in driver.scheme_client.list_directory(self.database).children:
                if not children.is_table():
                    logging.info(f'Skipping {children.name} because it is {children.type}')
                    continue
                tables.append(children.name)

            logging.info(f'Tables to be exported from ydb to yt: {tables}')
            for table in tables:
                export_to_yt_settings.with_source_and_destination(
                    source_path=os.path.join(self.database, table),
                    destination_path=os.path.join(os.path.join(self.yt_dir, table))
                )

            logging.info('Trying to start export..')
            operation = export_client.export_to_yt(export_to_yt_settings)
            logging.info(f'Export started with operation id {operation.id}')

            try:
                while (operation := operation.get()).progress not in (ExportProgress.DONE, ExportProgress.CANCELLED):
                    sleep(5)
                    logging.info(f'Export is alive, in {operation.progress.name} status')
            except Exception:
                logging.exception('Export failed, trying to cancel it')
                operation.cancel()
                raise
            finally:
                logging.info(f'Export is finished with status {operation.progress.name}, try to forget it')
                operation.forget()
