import logging
from concurrent.futures.thread import ThreadPoolExecutor
from functools import partial

from ott.drm.library.python.cms.clients import VhAdminClient, VhAdminException, VhAdminNotFoundException
from ott.drm.library.python.cms.models import InputStream, InputStreamStatus
from ott.drm.library.python.packager_task.clients import PackagerTasksApiClient, PackagerTasksApiException
from ott.drm.library.python.packager_task.models import (
    TaskStatus,
    PackagerTask,
    TaskExecutionState,
    Notification,
    NotificationEvent,
    PackagerOutputStatus,
    PackagerOutput,
    TaskResult,
    PackagerOutputResult
)
from yweb.video.faas.outputs.result import OutputLabel

from sandbox.projects.ott.packager_management_system.lib.exporter.callback import CallbackClient, CallbackException


class Exporter:
    _OUTPUT_LABEL_TO_NOTIFICATION_EVENT = {
        OutputLabel.AVC: NotificationEvent.AVC_PACKAGER_OUTPUT_PUBLISHED,
        OutputLabel.ALL: NotificationEvent.ALL_PACKAGER_OUTPUT_PUBLISHED
    }

    _TASK_RESULT_TO_NOTIFICATION_EVENT = {
        TaskResult.PACKAGE_FAILED: NotificationEvent.PACKAGE_FAILED,
        TaskResult.PACKAGE_CANCELLED: NotificationEvent.PACKAGE_CANCELLED,
        TaskResult.CREATE_GRAPH_FAILED: NotificationEvent.CREATE_GRAPH_FAILED
    }

    def __init__(self, packager_tasks_api_url, vh_admin_url, vh_admin_timeout_sec, max_workers: int,
                 sandbox_task_id=None, **_):
        self.vh_admin_client = VhAdminClient(vh_admin_url, vh_admin_timeout_sec)
        self.tasks_client = PackagerTasksApiClient(packager_tasks_api_url)
        self.sandbox_task_id = sandbox_task_id
        self._callback_client = CallbackClient()
        self._max_workers = max_workers

    def run(self):
        with ThreadPoolExecutor(max_workers=self._max_workers) as workers_pool:
            published_packager_outputs = self.tasks_client.get_packager_outputs(PackagerOutputStatus.PUBLISHED)
            logging.debug(f'PUBLISHED packager_outputs: {published_packager_outputs}')
            workers_pool.map(self._notify_and_finish_packager_output_safely, published_packager_outputs)

            cancelled_packager_outputs = self.tasks_client.get_packager_outputs(PackagerOutputStatus.CANCELLED)
            logging.debug(f'CANCELLED packager_outputs: {cancelled_packager_outputs}')
            workers_pool.map(
                partial(self._finish_packager_output_safely, result=PackagerOutputResult.CANCELLED),
                cancelled_packager_outputs
            )

            packaged_tasks = self.tasks_client.get_tasks(TaskStatus.PACKAGED)
            logging.debug(f'PACKAGED tasks: {packaged_tasks}')
            for task in packaged_tasks:
                if all(output.result == PackagerOutputResult.SUCCESS for output in task.packager_outputs):
                    workers_pool.submit(
                        self._export_and_finish_task_safely,
                        task,
                        TaskResult.SUCCESS,
                        InputStreamStatus.CONVERTED
                    )
                elif any(output.result == PackagerOutputResult.CANCELLED for output in task.packager_outputs):
                    workers_pool.submit(self._finish_task_safely, task, TaskResult.CANCELLED)
                else:
                    logging.info(f'{task.task_id} - Some outputs are not published: {task.packager_outputs}')

            for task_status, task_result in [(TaskStatus.PACKAGE_FAILED, TaskResult.PACKAGE_FAILED),
                                             (TaskStatus.PACKAGE_CANCELLED, TaskResult.PACKAGE_CANCELLED),
                                             (TaskStatus.CREATE_GRAPH_FAILED, TaskResult.CREATE_GRAPH_FAILED)]:
                failed_tasks = self.tasks_client.get_tasks(task_status)
                logging.info(f'{task_status.value} tasks: {failed_tasks}')
                workers_pool.map(
                    partial(
                        self._export_and_finish_task_safely,
                        task_result=task_result,
                        input_stream_result_status=InputStreamStatus.FAILED
                    ),
                    failed_tasks
                )

            cancelled_tasks = self.tasks_client.get_tasks(TaskStatus.CANCELLED)
            logging.debug(f'CANCELLED tasks: {cancelled_tasks}')
            workers_pool.map(
                partial(self._finish_task_safely, task_result=TaskResult.CANCELLED),
                cancelled_tasks
            )

    def _notify_and_finish_packager_output_safely(self, packager_output: PackagerOutput):
        notification_event = self._OUTPUT_LABEL_TO_NOTIFICATION_EVENT[packager_output.label]
        notification = Notification(packager_output.task_id, notification_event)

        try:
            self._notify(notification)
            self._finish_packager_output_safely(packager_output, PackagerOutputResult.SUCCESS)
        except PackagerTasksApiException:
            logging.exception('Failed to notify packager output published')

    def _export_and_finish_task_safely(self, task: PackagerTask, task_result: TaskResult,
                                       input_stream_result_status: InputStreamStatus):
        try:
            self.vh_admin_client.update_input_stream(
                task.input_stream_id,
                InputStream(status=input_stream_result_status)
            )
        except VhAdminNotFoundException:
            logging.exception(f'InputStream {task.input_stream_id} was not found')

            # override any task_result
            task_result = TaskResult.CANCELLED
        except VhAdminException:
            logging.exception(f'Failed to update {task.input_stream_id} InputStream')
            return

        if task_result in [TaskResult.PACKAGE_FAILED, TaskResult.PACKAGE_CANCELLED, TaskResult.CREATE_GRAPH_FAILED]:
            try:
                notification_event = self._TASK_RESULT_TO_NOTIFICATION_EVENT[task_result]
                self._notify(Notification(task.task_id, notification_event))
            except PackagerTasksApiException:
                logging.exception('Failed to add notification to queue')
                return

        if task.input_params.get('callback_url'):
            callback_url = task.input_params['callback_url']
            try:
                self._callback_client.send_finished_task_callback(
                    callback_url,
                    task.input_params['ott_content_uuid'],
                    input_stream_result_status
                )
                logging.info(f'Sent callback to {callback_url} for task {task.task_id}')
            except CallbackException:
                logging.exception(f'Failed to send callback to {callback_url} for task {task.task_id}')
                return

        self._finish_task_safely(task, task_result)

    def _notify(self, notification):
        self.tasks_client.add_notification_to_queue(notification)
        logging.info(f'Notification {notification} was added to queue')

    def _finish_task_safely(self, task: PackagerTask, task_result: TaskResult):
        task.status = TaskStatus.FINISHED
        task.result = task_result

        for packager_output in task.packager_outputs:
            if packager_output.status == PackagerOutputStatus.NOT_READY:
                try:
                    self._finish_packager_output(packager_output, PackagerOutputResult.UNCOMPLETED)
                except PackagerTasksApiException:
                    logging.exception(f'Failed to finish packager output: {packager_output}')
                    return

        try:
            self.tasks_client.update(task, TaskExecutionState(self.sandbox_task_id, 'EXPORTER'))
            logging.info(f'FINISHED task: {task}')
        except PackagerTasksApiException:
            logging.exception(f'Failed to update {task.task_id} PackagerTask')

    def _finish_packager_output(self, packager_output: PackagerOutput, result: PackagerOutputResult):
        packager_output.status = PackagerOutputStatus.FINISHED
        packager_output.result = result
        self.tasks_client.update_packager_output(packager_output)

    def _finish_packager_output_safely(self, packager_output: PackagerOutput, result: PackagerOutputResult):
        try:
            self._finish_packager_output(packager_output, result)
        except PackagerTasksApiException:
            logging.exception(f'Failed to finish packager output: {packager_output}')
