import json
import logging
import os
import subprocess as sp
from abc import ABC, abstractmethod
from dataclasses import dataclass
from datetime import datetime
from enum import Enum
from typing import Optional

from yweb.video.faas.outputs.error import load_error, DEFAULT_ERROR_FILENAME
from yweb.video.faas.proto.common.trace_pb2 import TError

from sandbox.projects.ott.packager_management_system.lib.graph_creator.cache import SyncedCache


@dataclass
class OttPackagerGraphCreationResult:
    code: int
    error: TError
    graph_meta: dict


class OttPackagerReleaseStatus(Enum):
    STABLE = 'stable'
    TESTING = 'testing'
    UNSTABLE = 'unstable'


class OttPackagerStage(Enum):
    STABLE = 'stable'
    TESTING = 'testing'


@dataclass
class OttPackagerResourceAttrs:
    create_time: datetime
    arc_revision: str
    release_status: OttPackagerReleaseStatus


class OttPackager:
    _RETRYABLE_FAAS_CODES = [TError.EEC_BAD_NONE, TError.EEC_INTERNAL_ERROR]

    def __init__(self, path: str, attrs: OttPackagerResourceAttrs, **_):
        self.path = path
        self.attrs = attrs

    def create_graph(self, task_id: str, nirvana_quota: str, nirvana_oauth_token: str,
                     nirvana_project: str, retries: int, s3_creds_nirvana_secret_name: str,
                     s3_creds: str, options: dict) -> OttPackagerGraphCreationResult:
        graph_meta_output_filename = f'graph_meta_output_{task_id}.json'
        open(graph_meta_output_filename, 'w').close()

        task_working_dir = os.path.abspath(task_id)
        logging.info(f'{task_id} working dir - {task_working_dir}')

        options = options.copy()
        options['no-start'] = True
        options['no-wait'] = True
        options['disable_graph_lifecycle_telegram_notifications'] = True
        options['num-threads'] = 5
        options['graph_meta_output'] = os.path.abspath(graph_meta_output_filename)
        options['oauth-token'] = nirvana_oauth_token
        options['nirvana_project'] = nirvana_project
        options['id'] = task_id
        options['quota'] = nirvana_quota
        options['s3_access_keys_secret_name'] = s3_creds_nirvana_secret_name
        options['working_dir'] = task_working_dir
        options['nirvana_data_ttl'] = 365

        secrets_filename = f'secrets_{task_id}.json'
        with open(secrets_filename, 'w') as secrets_file:
            secrets = {s3_creds_nirvana_secret_name: s3_creds}
            json.dump(secrets, secrets_file)

        options['secrets'] = os.path.abspath(secrets_filename)

        options_filename = f'ott_packager_options_{task_id}.json'
        with open(options_filename, 'w') as ott_packager_options_file:
            json.dump(options, ott_packager_options_file)

        command = [
            self.path,
            '--task', os.path.abspath(options_filename)
        ]

        # TODO: remove after https://st.yandex-team.ru/SANDBOX-8219 release
        env = os.environ.copy()
        if 'Y_PYTHON_ENTRY_POINT' in env:
            env.pop('Y_PYTHON_ENTRY_POINT')

        process_logger = logging.getLogger(f'subprocess-{task_id}')
        process = None
        error = None

        for attempt in range(retries):
            logging.info(f'{task_id} - run {command} (attempt - {attempt + 1})')
            process = sp.run(command, capture_output=True, env=env)

            _log_subprocess_stream(process_logger, process.stdout)
            _log_subprocess_stream(process_logger, process.stderr)

            if process.returncode == 0:
                logging.info(f'{task_id} - graph created')
                error = None
                break
            else:
                error = load_error(os.path.join(task_working_dir, DEFAULT_ERROR_FILENAME))
                logging.warning(f'{task_id} - failed to create graph (attempt - {attempt + 1}), '
                                f'exception: {error.Exception}, message: {error.Message}, faas_code: {error.Code}')

                if error.Code and error.Code not in self._RETRYABLE_FAAS_CODES:
                    logging.info(f'{task_id} - faas_code={error.Code} is not retryable. Stop trying')
                    break

        with open(graph_meta_output_filename, 'r') as graph_meta_output_file:
            raw_graph_meta = graph_meta_output_file.read()

        graph_meta = json.loads(raw_graph_meta) if len(raw_graph_meta) > 0 else {}
        graph_meta['arc_revision'] = self.attrs.arc_revision

        return OttPackagerGraphCreationResult(process.returncode, error, graph_meta)


def _log_subprocess_stream(process_logger, stream):
    for line in stream.decode('utf-8').split('\n'):
        if line:
            process_logger.info(f'{line}')


class OttPackagerRepository(ABC):
    @abstractmethod
    def fetch(self, arc_revision: str) -> Optional[OttPackager]:
        """
        Returns latest OttPackager by attrs.create_time with selected arc_revision
        """
        pass

    @abstractmethod
    def fetch_resource_attrs(self, release_status: OttPackagerReleaseStatus) -> OttPackagerResourceAttrs:
        """
        Returns latest OttPackagerResourceAttrs by create_time with selected release_status
        """
        pass


class DummyOttPackagerRepository(OttPackagerRepository):
    def __init__(self, ott_packager: OttPackager):
        self.ott_packager = ott_packager

    def fetch(self, arc_revision: str) -> Optional[OttPackager]:
        return self.ott_packager

    def fetch_resource_attrs(self, release_status: OttPackagerReleaseStatus) -> OttPackagerResourceAttrs:
        return self.ott_packager.attrs


class CachedOttPackagerRepository(OttPackagerRepository):
    def __init__(self, ott_packager_repository: OttPackagerRepository):
        self._ott_packager_repository = ott_packager_repository
        self._ott_packagers_cache = SyncedCache()
        self._ott_packager_resource_attrs_cache = SyncedCache()

    def fetch(self, arc_revision: str) -> Optional[OttPackager]:
        return self._ott_packagers_cache.get(arc_revision, self._ott_packager_repository.fetch)

    def fetch_resource_attrs(self, release_status: OttPackagerReleaseStatus) -> OttPackagerResourceAttrs:
        return self._ott_packager_resource_attrs_cache.get(
            release_status,
            self._ott_packager_repository.fetch_resource_attrs
        )


class OttPackagerService:
    def __init__(self, ott_packager_repository: OttPackagerRepository):
        self._ott_packager_repository = ott_packager_repository

    def find_by_arc_revision(self, arc_revision: str) -> Optional[OttPackager]:
        return self._ott_packager_repository.fetch(arc_revision=arc_revision)

    def find_by_stage(self, stage: OttPackagerStage) -> OttPackager:
        stable_ott_packager_resource_attrs = self._ott_packager_repository.fetch_resource_attrs(
            OttPackagerReleaseStatus.STABLE)

        if stage == OttPackagerStage.TESTING:
            testing_ott_packager_resource_attrs = self._ott_packager_repository.fetch_resource_attrs(
                OttPackagerReleaseStatus.TESTING)

            if testing_ott_packager_resource_attrs.create_time < stable_ott_packager_resource_attrs.create_time:
                return self._ott_packager_repository.fetch(stable_ott_packager_resource_attrs.arc_revision)

            return self._ott_packager_repository.fetch(testing_ott_packager_resource_attrs.arc_revision)

        return self._ott_packager_repository.fetch(stable_ott_packager_resource_attrs.arc_revision)
