import datetime
import logging
from typing import List

from ott.drm.library.python.cms.clients import VhAdminClient, VhAdminException, VhAdminNotFoundException
from ott.drm.library.python.cms.models import InputStreamStatus, InputStream, VodProvider, FaasAnswer
from ott.drm.library.python.packager_task.clients import PackagerTasksApiClient, PackagerTasksApiException
from ott.drm.library.python.packager_task.models import PackagerTask, TaskStatus, TaskExecutionState

from sandbox.projects.ott.packager_management_system.lib.importer.vod_provider_params import VOD_PROVIDER_PARAMS_MAP

INPUT_STREAMS_DAYS_IN_PAST = 7
INPUT_STREAMS_REQUEST_LIMIT = 100


class Importer:
    def __init__(self, vod_provider_ids, packager_tasks_api_url, vh_admin_url, vh_admin_timeout_sec,
                 default_nirvana_quota, sandbox_task_id=None, **_):
        self.vh_admin_client = VhAdminClient(vh_admin_url, vh_admin_timeout_sec)
        self.tasks_client = PackagerTasksApiClient(packager_tasks_api_url)
        self.vod_providers = [self.vh_admin_client.get_vod_provider(id) for id in vod_provider_ids]
        self.default_nirvana_quota = default_nirvana_quota
        self.sandbox_task_id = sandbox_task_id

    def run(self):
        update_time_gt = datetime.datetime.now() - datetime.timedelta(days=INPUT_STREAMS_DAYS_IN_PAST)
        for vod_provider in self.vod_providers:
            logging.info(f'Get input streams for vod_provider: {vod_provider}')

            new_input_streams = []
            total = -1
            while total != len(new_input_streams):
                new_input_streams, total = self.vh_admin_client.get_input_streams(
                    InputStreamStatus.FAAS_WAIT_FOR_ADD,
                    vod_provider.id,
                    update_time_gt.date(),
                    INPUT_STREAMS_REQUEST_LIMIT
                )
                logging.info(f'New input streams: {new_input_streams}')

                draft_tasks = self._create_draft_tasks(new_input_streams, vod_provider)
                self._handle_draft_tasks(draft_tasks)

    def _create_draft_tasks(self, new_input_streams: List[InputStream], vod_provider: VodProvider
                            ) -> List[PackagerTask]:
        draft_tasks_by_input_stream_id = self._get_draft_tasks_by_input_stream_id()
        for input_stream in new_input_streams:
            if input_stream.input_stream_id not in draft_tasks_by_input_stream_id:
                try:
                    task = self._create_task(input_stream, vod_provider)
                    draft_tasks_by_input_stream_id[input_stream.input_stream_id] = task
                    logging.info(f'DRAFT task: {task}')
                except PackagerTasksApiException:
                    logging.exception(f'Failed to create task for input_stream: {input_stream}')
                    continue

        return list(draft_tasks_by_input_stream_id.values())

    def _get_draft_tasks_by_input_stream_id(self):
        draft_tasks = self.tasks_client.get_tasks(TaskStatus.DRAFT)

        logging.debug(f'Suspended DRAFT tasks: {draft_tasks}')

        draft_tasks_by_input_stream_id = {}
        for task in draft_tasks:
            draft_tasks_by_input_stream_id[task.input_stream_id] = task

        return draft_tasks_by_input_stream_id

    def _create_task(self, input_stream: InputStream, vod_provider: VodProvider):
        input_params = input_stream.faas_custom_parameters.copy()
        input_params['content_version_id'] = input_stream.content_version_id
        input_params['input_video_url'] = input_stream.data
        input_params['s3_bucket'] = vod_provider.s3_bucket
        input_params['s3_dir'] = vod_provider.s3_dir
        input_params['vh_admin_url'] = self.vh_admin_client.url
        input_params['vh_admin_timeout_sec'] = self.vh_admin_client.timeout
        input_params['callback_url'] = vod_provider.callback_url

        input_params.update(VOD_PROVIDER_PARAMS_MAP.get(vod_provider.name, {}))

        task = PackagerTask(
            input_stream_id=input_stream.input_stream_id,
            vod_provider=vod_provider.name,
            status=TaskStatus.DRAFT,
            author=input_stream.author,
            priority=input_stream.priority,
            input_params=input_params,
            nirvana_quota=input_params.get('quota', self.default_nirvana_quota)
        )

        task_id = self.tasks_client.create(task)
        task.task_id = task_id

        return task

    def _handle_draft_tasks(self, tasks: List[PackagerTask]):
        for task in tasks:
            try:
                self._handle_draft_task(task)
            except (PackagerTasksApiException, VhAdminException):
                logging.exception(f'Failed to handle draft task: {task}')

    def _handle_draft_task(self, task: PackagerTask):
        try:
            self.vh_admin_client.update_input_stream(
                task.input_stream_id,
                InputStream(
                    status=InputStreamStatus.FAAS_SENT,
                    faas_answer=FaasAnswer(ott_packager_task_id=task.task_id),
                    faas_custom_parameters=task.input_params
            ))
            task.status = TaskStatus.NEW
        except VhAdminNotFoundException:
            logging.exception(f'InputStream {task.input_stream_id} was not found. '
                              f'Task {task.task_id} will be cancelled')
            task.status = TaskStatus.CANCELLED

        self.tasks_client.update(task, TaskExecutionState(self.sandbox_task_id, 'IMPORTER'))

        logging.info(f'{task.status.value} task: {task}')
