import logging
from typing import List

from nirvana_api import NirvanaApi, RPCException
from ott.drm.library.python.packager_task.clients import PackagerTasksApiClient
from ott.drm.library.python.packager_task.models import TaskStatus, TaskExecutionState

from sandbox.projects.ott.packager_management_system.lib.launcher.quota_manager import QuotaManager

QUOTA_EXCEEDED_NIRVANA_ERROR_CODE = 70


class Launcher:
    def __init__(self, packager_tasks_api_url, nirvana_oauth_token, max_running_graphs,
                 nirvana_quota, vod_providers: List[str], parallel_graph_launch_delay_sec, sandbox_task_id=None, **_):
        self.nirvana = NirvanaApi(nirvana_oauth_token)
        self.tasks_client = PackagerTasksApiClient(packager_tasks_api_url)
        self.quota_manager = QuotaManager(
            self.tasks_client,
            max_running_graphs,
            nirvana_quota,
            vod_providers,
            parallel_graph_launch_delay_sec
        )
        self.sandbox_task_id = sandbox_task_id
        self.nirvana_quota = nirvana_quota
        self.vod_providers = vod_providers

    def run(self):
        self._handle_graph_created_tasks()
        self._handle_enqueued_tasks()

    def _handle_graph_created_tasks(self):
        graph_created_tasks = self.tasks_client.get_tasks(
            TaskStatus.GRAPH_CREATED,
            nirvana_quota=self.nirvana_quota,
            vod_providers=self.vod_providers
        )

        logging.debug(f'GRAPH_CREATED tasks: {graph_created_tasks}')

        for task in graph_created_tasks:
            task.status = TaskStatus.ENQUEUED_FOR_PACKAGE
            self.tasks_client.update(task, TaskExecutionState(self.sandbox_task_id, 'LAUNCHER'))

    def _handle_enqueued_tasks(self):
        enqueued_tasks = self.tasks_client.get_tasks(
            TaskStatus.ENQUEUED_FOR_PACKAGE,
            nirvana_quota=self.nirvana_quota,
            vod_providers=self.vod_providers
        )

        logging.debug(f'ENQUEUED tasks: {enqueued_tasks}')

        while len(enqueued_tasks) > 0 and self.quota_manager.can_launch_graph(enqueued_tasks[0]):
            task = enqueued_tasks.pop(0)

            if task.is_parallel_encoding() and not self.quota_manager.can_launch_parallel_graph():
                logging.info(f'Parallel encoding quota limit is reached! '
                             f'Task {task} will be launched later')
                continue

            logging.info(f'Launching task: {task.task_id}')
            logging.info(f'Task details: {task}')

            execution_state = self.nirvana.get_execution_state(
                workflow_id=task.graph_meta['nirvana_workflow_id'],
                workflow_instance_id=task.graph_meta['nirvana_workflow_instance_id']
            )

            logging.info(f'Graph execution state: {execution_state}')

            # Otherwise graph was already launched but status was not changed
            if execution_state.status == 'undefined':
                try:
                    self.nirvana.start_workflow(
                        task.graph_meta['nirvana_workflow_id'],
                        task.graph_meta['nirvana_workflow_instance_id']
                    )
                except RPCException as ex:
                    error_code = ex.args[0]
                    error_message = ex.args[1]
                    if error_code == QUOTA_EXCEEDED_NIRVANA_ERROR_CODE:
                        logging.info(f'{error_message}. Task {task} will be launched later')
                        break

                    raise ex

            task.status = TaskStatus.PACKAGING
            self.tasks_client.update(task, TaskExecutionState(self.sandbox_task_id, 'LAUNCHER'))

        if len(enqueued_tasks) > 0:
            logging.info('Quota limit is reached!')
            logging.debug(f'Tasks: {enqueued_tasks} will be launched later')
