import logging
from concurrent.futures import ThreadPoolExecutor
from typing import List, Optional

from google.protobuf.json_format import ParseDict
from nirvana_api import NirvanaApi
from ott.drm.library.python.cms.clients import VhAdminClient
from ott.drm.library.python.packager_task.clients import PackagerTasksApiClient
from ott.drm.library.python.packager_task.models import (
    NotificationEvent,
    PackagingTaskError,
    PackagerOutput,
    Notification,
    GraphCreatingError
)
from ott.drm.library.python.telegram.clients import TelegramProxyClient, TelegramProxyNotFoundException
from yweb.video.faas.graphs.ott.common import get_chat_labels
from yweb.video.faas.lib.nirvana import get_graph_execution_duration
from yweb.video.faas.outputs.error import NirvanaBlock
from yweb.video.faas.outputs.result import OutputLabel
from yweb.video.faas.proto.common.output_links_pb2 import TOutputLinks
from yweb.video.faas.services.ott.telegram.message_factory import TelegramMessageFactory


class Notifier:
    def __init__(self, packager_tasks_api_url, vh_admin_url, vh_admin_timeout_sec, u_duty_project, vh_frontend_url,
                 nirvana_oauth_token, max_workers: int):
        self.vh_admin_client = VhAdminClient(vh_admin_url, vh_admin_timeout_sec)
        self.tasks_client = PackagerTasksApiClient(packager_tasks_api_url)
        self.telegram_proxy_client = TelegramProxyClient()
        self.telegram_message_factory = TelegramMessageFactory(vh_frontend_url=vh_frontend_url)
        self.u_duty_project = u_duty_project
        self.nirvana = NirvanaApi(nirvana_oauth_token)
        self._max_workers = max_workers

    def run(self):
        with ThreadPoolExecutor(max_workers=self._max_workers) as workers_pool:
            notifications = self.tasks_client.get_notifications()
            logging.info(f'Notifications: {notifications}')
            workers_pool.map(self._handle_notification_safely, notifications)

    def _handle_notification_safely(self, notification: Notification):
        try:
            self._handle_notification(notification)
        except Exception:
            logging.exception(f'Failed to handle notification: {notification}')

    def _handle_notification(self, notification: Notification):
        task = self.tasks_client.get_task(notification.task_id)
        content_group = self.vh_admin_client.get_content_group(task.input_params['ott_content_uuid'])
        parallel = task.input_params.get('parallel_encoding')
        ladder = task.input_params.get('ladder')
        embedded = task.input_params.get('embedded')

        if notification.event == NotificationEvent.PACKAGE_FAILED:
            message = self.telegram_message_factory.create_packaging_error_message(
                content_group,
                task.input_params['input_video_url'],
                task.graph_meta['nirvana_workflow_id'],
                task.graph_meta['nirvana_workflow_instance_id'],
                self._get_packaging_task_error_nirvana_block(task.errors),
                task.graph_meta['main_log_s3_url']
            )
        elif notification.event == NotificationEvent.PACKAGE_CANCELLED:
            message = self.telegram_message_factory.create_package_cancelled_message(
                content_group,
                task.input_params['input_video_url'],
                task.graph_meta['nirvana_workflow_id'],
                task.graph_meta['nirvana_workflow_instance_id']
            )
        elif notification.event == NotificationEvent.CREATE_GRAPH_FAILED:
            graph_creating_error = self._get_graph_creating_error(task.errors)
            message = self.telegram_message_factory.create_graph_creating_error_message(
                content_group,
                task.input_params['input_video_url'],
                graph_creating_error.sandbox_task_id,
                graph_creating_error.ex_info
            )
        elif notification.event == NotificationEvent.AVC_PACKAGER_OUTPUT_PUBLISHED:
            packager_output = self._get_packager_output_by_label(task.packager_outputs, OutputLabel.AVC)
            message = self.telegram_message_factory.create_streams_encoded_message(
                content_group,
                task.input_params['input_video_url'],
                ParseDict(packager_output.data, TOutputLinks()).OutputForVhLink,
                task.graph_meta['nirvana_workflow_id'],
                task.graph_meta['nirvana_workflow_instance_id'],
                parallel,
                embedded,
                ladder,
                self._get_duration(
                    task.graph_meta['nirvana_workflow_id'],
                    task.graph_meta['nirvana_workflow_instance_id'],
                    packager_output.block_meta.block_guid
                ),
                OutputLabel.AVC.value
            )
        elif notification.event == NotificationEvent.ALL_PACKAGER_OUTPUT_PUBLISHED:
            packager_output = self._get_packager_output_by_label(task.packager_outputs, OutputLabel.ALL)
            message = self.telegram_message_factory.create_streams_encoded_message(
                content_group,
                task.input_params['input_video_url'],
                ParseDict(packager_output.data, TOutputLinks()).OutputForVhLink,
                task.graph_meta['nirvana_workflow_id'],
                task.graph_meta['nirvana_workflow_instance_id'],
                parallel,
                embedded,
                ladder,
                self._get_duration(
                    task.graph_meta['nirvana_workflow_id'],
                    task.graph_meta['nirvana_workflow_instance_id'],
                    packager_output.block_meta.block_guid
                )
            )
        else:
            raise RuntimeError(f'Unsupported notification event: {notification.event}')

        for label in get_chat_labels(task.input_params.get('notification_chat_label'), task.input_params.get('author')):
            try:
                self.telegram_proxy_client.send(self.u_duty_project, label, message)
                logging.info(f'{notification} was sent to {label} chat')
            except TelegramProxyNotFoundException:
                logging.exception(f'{notification} was not sent due to wrong notification params')

        self.tasks_client.remove_notification_from_queue(notification)

    def _get_duration(self, workflow_id: str, workflow_instance_id: str, block_guid: str):
        state = self.nirvana.get_execution_state(
            workflow_id=workflow_id,
            workflow_instance_id=workflow_instance_id,
            block_patterns=[{'guid': block_guid}]
        )
        if state['blocks'][0]['status'] != 'completed':
            raise RuntimeError(f'Block with guid {block_guid} is not completed. Workflow id: {workflow_id}')
        return get_graph_execution_duration(state['started'], state['blocks'][0]['completed'])

    @staticmethod
    def _get_packaging_task_error_nirvana_block(task_errors) -> Optional[NirvanaBlock]:
        for task_error in task_errors:
            if isinstance(task_error, PackagingTaskError) and task_error.nirvana_blocks:
                return task_error.nirvana_blocks[0]

    @staticmethod
    def _get_graph_creating_error(task_errors) -> GraphCreatingError:
        for task_error in task_errors:
            if isinstance(task_error, GraphCreatingError):
                return task_error

    @staticmethod
    def _get_packager_output_by_label(packager_outputs: List[PackagerOutput], label: OutputLabel):
        for packager_output in packager_outputs:
            if packager_output.label == label:
                return packager_output
