import json
import logging
import uuid
from concurrent.futures import ThreadPoolExecutor
from typing import List, Optional

from google.protobuf.json_format import ParseDict
from ott.drm.library.python.cms.clients import (
    VhAdminException,
    VhAdminClient,
    VhAdminNotFoundException,
    OttContentApiClient,
    OttContentApiNotFoundException,
    OttContentApiException
)
from ott.drm.library.python.cms.models import (
    OutputStream,
    ContentGroup,
    ContentVersionUpdateParams,
    InputStream,
    ContentVersion
)
from ott.drm.library.python.packager_task.clients import PackagerTasksApiClient, PackagerTasksApiException
from ott.drm.library.python.packager_task.models import PackagerOutputStatus
from retrying import retry
from yweb.video.faas.outputs.utils import download
from yweb.video.faas.proto.common.output_links_pb2 import TOutputLinks
from yweb.video.faas.proto.common.outputs_pb2 import TOutput

from sandbox.projects.ott.packager_management_system.lib.publisher.faas_answer_factory import create_faas_answer
from sandbox.projects.ott.packager_management_system.lib.publisher.output_stream_factory import create_output_stream


def _retry_on_update_content_group_exception(exception):
    return isinstance(exception, UpdateContentGroupException) or isinstance(exception, UpdateContentVersionException)


class Publisher:
    def __init__(self, packager_tasks_api_url, vh_admin_client: VhAdminClient,
                 ott_content_api_client: OttContentApiClient, max_workers: int):
        self.vh_admin_client = vh_admin_client
        self.tasks_client = PackagerTasksApiClient(packager_tasks_api_url)
        self.ott_content_api_client = ott_content_api_client
        self._max_workers = max_workers

    def run(self):
        with ThreadPoolExecutor(max_workers=self._max_workers) as workers_pool:
            ready_packager_outputs = self.tasks_client.get_packager_outputs(PackagerOutputStatus.READY)
            logging.debug(f'Ready packager outputs: {ready_packager_outputs}')

            workers_pool.map(self._handle_ready_packager_output, ready_packager_outputs)

    def _handle_ready_packager_output(self, packager_output):
        output_links = ParseDict(packager_output.data, TOutputLinks())
        output_json = json.loads(download(output_links.OutputLink))
        output = ParseDict(output_json, TOutput(), ignore_unknown_fields=True)

        try:
            self.publish_packager_output(
                packager_output.task_id,
                packager_output.content_group_uuid,
                packager_output.content_version_id,
                output
            )
            packager_output.status = PackagerOutputStatus.PUBLISHED
        except ContentVersionNotFoundException:
            packager_output.status = PackagerOutputStatus.CANCELLED
            logging.info(f'Output is cancelled for taskId={packager_output.task_id}, label={packager_output.label}')
        except PublishException:
            logging.exception(f'{packager_output.label} packager output not published '
                              f'for task_id={packager_output.task_id} ')
            return

        if packager_output.activate_content_version:
            try:
                self.ott_content_api_client.activate_content_version(
                    packager_output.content_group_uuid,
                    packager_output.content_version_id
                )
            except OttContentApiNotFoundException as e:
                packager_output.status = PackagerOutputStatus.CANCELLED
                logging.info(f'No content version or content group for taskId={packager_output.task_id}: {str(e)}')
            except OttContentApiException:
                logging.exception(f'Content version is not activated for taskId={packager_output.task_id}')
                return

        try:
            self.tasks_client.update_packager_output(packager_output)
        except PackagerTasksApiException:
            logging.exception(f'Update packager output failed for taskId={packager_output.task_id}, '
                              f'label={packager_output.label}')

    @retry(stop_max_attempt_number=3, retry_on_exception=_retry_on_update_content_group_exception)
    def publish_packager_output(self, task_id: uuid.UUID, content_group_uuid: str, content_version_id: int,
                                packager_output_data: TOutput):
        try:
            content_version = self.vh_admin_client.get_content_version(content_version_id)
            logging.info(f'Received content version: {content_version}')
        except VhAdminNotFoundException:
            msg = f'Not found content version {content_version_id} of {content_group_uuid} content group'
            raise ContentVersionNotFoundException(msg)
        except VhAdminException as e:
            msg = f'Failed to receive content version {content_version_id} of {content_group_uuid} content group'
            raise RetrieveContentVersionException(msg) from e

        new_streams = [create_output_stream(stream) for stream in packager_output_data.Streams]
        merged_streams = self._merge_streams(content_version.output_streams, new_streams)
        logging.info(f'Merged output streams: {merged_streams}')

        try:
            self.vh_admin_client.update_content_version(content_version_id, ContentVersionUpdateParams(merged_streams))
        except VhAdminException as e:
            msg = f'Failed to update content version {content_version_id} of {content_group_uuid} content group'
            raise UpdateContentVersionException(msg) from e

        content_group_update_params = ContentGroup(
            duration=int(round(packager_output_data.Duration)),
            video_screenshots=[thumb for thumb in packager_output_data.Thumbnails]
        )
        logging.info(f'ContentGroupUpdateParams for uuid={content_group_uuid}: {content_group_update_params}')

        try:
            self.vh_admin_client.update_content_group(content_group_uuid, content_group_update_params)
        except VhAdminException as e:
            msg = f'Failed to update content group with uuid={content_group_uuid}'
            raise UpdateContentGroupException(msg) from e

        task_input_stream = self._get_task_input_stream(task_id, content_version)
        logging.info(f'{task_id} - task input stream: {task_input_stream}')
        if task_input_stream:
            input_stream_update_params = InputStream(
                input_stream_id=task_input_stream.input_stream_id,
                faas_answer=(create_faas_answer(task_id, packager_output_data.InputVideo))
            )
            try:
                self.vh_admin_client.update_input_stream(
                    task_input_stream.input_stream_id,
                    input_stream_update_params
                )
                logging.info(f'Updated input stream by params {input_stream_update_params}')
            except VhAdminException as e:
                msg = f'Failed to update input stream by params {input_stream_update_params}'
                raise UpdateInputStreamException(msg) from e

    @staticmethod
    def _merge_streams(old_streams: List[OutputStream], new_streams: List[OutputStream]) -> List[OutputStream]:
        merged_output_streams = []
        for stream in old_streams + new_streams:
            is_add_output_stream = True
            for added_stream in merged_output_streams:
                if stream == added_stream:
                    logging.info(f'Skip {stream} cause it is already in {merged_output_streams}')
                    is_add_output_stream = False

            if is_add_output_stream:
                merged_output_streams.append(stream)

        return merged_output_streams

    @staticmethod
    def _get_task_input_stream(task_id: uuid.UUID, content_version: ContentVersion) -> Optional[InputStream]:
        return next(filter(lambda s: s.ott_packager_task_id == task_id, content_version.input_streams), None)


class PublishException(Exception):
    pass


class ContentVersionNotFoundException(PublishException):
    pass


class UpdateContentGroupException(PublishException):
    pass


class UpdateContentVersionException(PublishException):
    pass


class RetrieveContentVersionException(PublishException):
    pass


class UpdateInputStreamException(PublishException):
    pass
