import json
import logging
from datetime import datetime

from nirvana_api import NirvanaApi
from nirvana_api.json_rpc import DefaultEncoder

from yweb.video.faas.outputs.error import get_failed_operations
from yweb.video.faas.outputs.result import download_result, ResultMeta
import humanize
from ott.drm.library.python.packager_task.clients import PackagerTasksApiClient
from ott.drm.library.python.packager_task.models import (
    TaskStatus,
    TaskExecutionState,
    TaskErrorType,
    PackagerOutputStatus,
    PackagingTaskError
)


class Watcher:
    def __init__(self, packager_tasks_api_url, nirvana_oauth_token, sandbox_task_id=None, **_):
        self.nirvana = NirvanaApi(nirvana_oauth_token)
        self.tasks_client = PackagerTasksApiClient(packager_tasks_api_url)
        self.sandbox_task_id = sandbox_task_id

    def run(self):
        packaging_tasks = self.tasks_client.get_tasks(TaskStatus.PACKAGING)

        logging.debug(f'Packaging tasks: {packaging_tasks}')
        for task in packaging_tasks:
            try:
                self._handle_packaging_task(task)
            except Exception:
                logging.exception(f'{task.task_id} - Failed to handle packaging task')

    def _handle_packaging_task(self, task):
        current_time = datetime.now()
        graph_meta = task.graph_meta
        nirvana_workflow_instance_id = graph_meta['nirvana_workflow_instance_id']

        not_ready_packager_outputs = [
            packager_output
            for packager_output in task.packager_outputs
            if packager_output.status == PackagerOutputStatus.NOT_READY
        ]

        execution_state = self.nirvana.get_execution_state(
            workflow_instance_id=nirvana_workflow_instance_id,
            block_patterns=[{'guid': output.block_meta.block_guid} for output in not_ready_packager_outputs]
        )

        graph_started_datetime = datetime.now()
        if started_raw := execution_state.get('started'):
            graph_started_datetime = datetime.fromisoformat(started_raw[:-len('+0300')])
        logging.info(
            f'{task.task_id} - {graph_meta["nirvana_instance_url"]} '
            f'(created {humanize.naturaldelta(current_time - task.create_time)} ago, '
            f'started {humanize.naturaldelta(current_time - graph_started_datetime)} ago) '
            f'- execution_state: {json.dumps(execution_state, indent=2, cls=DefaultEncoder)}'
        )

        for packager_output_block in execution_state['blocks']:
            # see https://st.yandex-team.ru/OTT-14524
            if packager_output_block.result == 'success' or execution_state.result == 'success':
                for packager_output in not_ready_packager_outputs:
                    if packager_output.block_meta.block_guid == packager_output_block['blockGuid']:
                        self._save_packager_output(task, packager_output)

        task_error = None
        if execution_state.status == 'completed':
            if execution_state.result == 'success':
                task.status = TaskStatus.PACKAGED
            elif execution_state.result == 'cancel':
                task.status = TaskStatus.PACKAGE_CANCELLED
            else:
                task.status = TaskStatus.PACKAGE_FAILED
                failed_blocks = get_failed_operations(self.nirvana, nirvana_workflow_instance_id)
                task_error = PackagingTaskError(TaskErrorType.PACKAGING, failed_blocks)
                logging.info(f'{task.task_id} - Failed blocks: {failed_blocks}')

            task_execution_state = TaskExecutionState(self.sandbox_task_id, 'WATCHER')
            self.tasks_client.update(task, task_execution_state, task_error)

            logging.info(f'{task.task_id} - completed')

    def _save_packager_output(self, task, packager_output):
        packager_output.status = PackagerOutputStatus.READY
        data_str = download_result(
            self.nirvana,
            task.graph_meta,
            ResultMeta(
                packager_output.block_meta.block_name,
                packager_output.block_meta.output_name,
                packager_output.label,
                packager_output.activate_content_version
            ),
            False
        )
        packager_output.data = json.loads(data_str)

        self.tasks_client.update_packager_output(packager_output)

        logging.info(f'READY packager_output: {packager_output}')
