from sandbox import sdk2

from sandbox.projects.ott.packager_management_system.sandbox_tasks import OttPackagerBaseTask


class OttPackagerBinary(sdk2.Resource):
    """Ott packager_management_system binary"""
    executable = True
    releasable = True

    arcadia_revision = sdk2.Attributes.String('arcadia_revision')


class OttPackagerGraphCreator(OttPackagerBaseTask):
    class Parameters(OttPackagerBaseTask.Parameters):
        nirvana_secrets = sdk2.parameters.YavSecret('Yav secret with nirvana oauth token')
        s3_creds_secret = sdk2.parameters.YavSecret('Yav secret with s3 creds')

        max_not_launched_graphs = sdk2.parameters.Integer('max_not_launched_graphs', default_value=5)
        max_graph_creating_workers = sdk2.parameters.Integer('max_graph_creating_workers', default_value=1)
        graph_creation_retries = sdk2.parameters.Integer('graph_creation_retries', default_value=5)
        nirvana_quota = sdk2.parameters.String('nirvana_quota')
        s3_creds_nirvana_secret_name = sdk2.parameters.String('s3_creds_nirvana_secret_name')
        vod_providers = sdk2.parameters.List('vod_providers', value_type=sdk2.parameters.String)

    def on_save(self):
        super(OttPackagerGraphCreator, self).on_save()

        self.Requirements.cores = int(self.Parameters.max_graph_creating_workers)
        self.Requirements.ram = int(self.Parameters.max_graph_creating_workers) * 512 + 512

    def on_execute(self):
        import logging

        from ott.drm.library.python.packager_task.clients import PackagerTasksApiClient
        from sandbox.projects.ott.packager_management_system.lib.graph_creator import GraphCreator
        from sandbox.projects.ott.packager_management_system.lib.graph_creator.ott_packager import (
            CachedOttPackagerRepository,
            OttPackagerRepository,
            OttPackager,
            OttPackagerResourceAttrs,
            OttPackagerReleaseStatus
        )

        class SandboxOttPackagerRepository(OttPackagerRepository):
            def fetch(self, arc_revision):
                sandbox_resource = OttPackagerBinary.find(attrs={'arcadia_revision': arc_revision}).first()

                if not sandbox_resource:
                    logging.warning('Not found OttPackagerBinary by arcadia_revision={}'.format(arc_revision))
                    return

                ott_package = sdk2.ResourceData(sandbox_resource)
                logging.info('Fetched OttPackagerBinary by arcadia_revision={}'.format(arc_revision))

                return OttPackager(
                    '{ott_package_dir}/ott_packager'.format(ott_package_dir=ott_package.path),
                    self._create_ott_packager_resource_attrs(sandbox_resource)
                )

            def fetch_resource_attrs(self, release_status):
                sandbox_resource = OttPackagerBinary.find(attrs={'released': release_status.value}).first()
                logging.info('Fetched OttPackagerBinaryAttrs by released={}'.format(release_status.value))

                return self._create_ott_packager_resource_attrs(sandbox_resource)

            @staticmethod
            def _create_ott_packager_resource_attrs(sandbox_resource):
                return OttPackagerResourceAttrs(
                    sandbox_resource.created,
                    sandbox_resource.arcadia_revision,
                    OttPackagerReleaseStatus(sandbox_resource.released) if sandbox_resource.released else None
                )

        nirvana_oauth_token = self.Parameters.nirvana_secrets.data()['token']

        GraphCreator(
            packager_tasks_api_url=self.Parameters.packager_tasks_api_url,
            ott_packager_repository=CachedOttPackagerRepository(SandboxOttPackagerRepository()),
            sandbox_task_id=sdk2.Task.current.id,
            max_graph_creating_workers=self.Parameters.max_graph_creating_workers,
            max_not_launched_graphs=self.Parameters.max_not_launched_graphs,
            nirvana_oauth_token=nirvana_oauth_token,
            tasks_client=PackagerTasksApiClient(self.Parameters.packager_tasks_api_url),
            nirvana_quota=self.Parameters.nirvana_quota,
            vod_providers=self.Parameters.vod_providers,
            graph_creation_retries=int(self.Parameters.graph_creation_retries),
            s3_creds_nirvana_secret_name=self.Parameters.s3_creds_nirvana_secret_name,
            s3_creds=self.Parameters.s3_creds_secret.data()['creds']
        ).run()
