# coding: utf-8

# Задача добавляет в релизный тикет информацию обо всех тасках которые были вмерджены с последнего релиза
# Тэг последнего релиза берёт с прода

import logging

import sandbox.sdk2 as sdk2

from sandbox.projects.partner.settings import ROBOT_PEREIRO_STARTREK_TOKEN_SECRET, ROBOT_PARTNER_SECRET
from sandbox.projects.partner.utils.arc import Arc
from sandbox.projects.partner.utils.get_production_tag import get_production_tag, VersionType
from sandbox.sandboxsdk import environments
from sandbox.projects.partner.utils.startrek import extract_pi_tickets_from_text
from sandbox.projects.adfox.adfox_ui.metrics import AnalyzableTask


ST_USERAGENT = "Sandbox PARTNER_ADD_TICKETS_TO_RELEASE"


class AddTicketsToRelease(AnalyzableTask):

    name = 'PARTNER_ADD_TICKETS_TO_RELEASE'

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('startrek_client', custom_parameters=["requests==2.18.4"]),
        ]

    class Parameters(AnalyzableTask.Parameters):
        description = 'Put all merged tasks to ticket.'

        st_issue = sdk2.parameters.String('Release ticket', description='PI-NNNNN', required=True)

        arc_release_branch = sdk2.parameters.String(
            "Arc release branch",
            description="Branch ID in arc, ex. releases/partner/PI-00000-release-1",
            default_value="trunk",
            required=True
        )

        with sdk2.parameters.Group('Secrets'):
            st_token = sdk2.parameters.YavSecret(
                'Startrek OAuth token', default=ROBOT_PEREIRO_STARTREK_TOKEN_SECRET, required=True
            )
            arc_token = sdk2.parameters.YavSecret(
                'Arc OAuth token', description='Use arc to create release branch', default=ROBOT_PARTNER_SECRET
            )

    def get_arc_token(self):
        logging.debug('getting arc token')
        if not self.Parameters.arc_token:
            raise Exception('Arc access token is missed')

        token = self.Parameters.arc_token.data()['arc_token']
        logging.debug('success getting arc token')
        return token

    def on_execute(self):
        super(AddTicketsToRelease, self).on_execute()
        if not self.Parameters.st_issue:
            logging.info("No release ticket specified, skipping updating release info")
            return
        logging.info("Attempting to update release ticket description with tickets from arc")
        commit = self.get_java_release_commit()
        logging.info("Arc release commit for current production: " + commit)

        arc = Arc(path='.', token=self.get_arc_token())
        arc.checkout(self.Parameters.arc_release_branch)
        arc_tickets = arc.get_tickets_since(commit)
        logging.info("Tickets from arc: " + str(arc_tickets))

        issue = self.get_st_issue()
        issue_description = str(issue.description) if issue.description is not None else ''
        logging.info("Issue description: \n" + issue_description)

        existing_tickets = extract_pi_tickets_from_text(issue_description)
        logging.info("Tickets already in issue descriprion: " + str(existing_tickets))
        new_tickets = set(arc_tickets).difference(existing_tickets)

        if not new_tickets:
            logging.info("No tickets to add to issue description ")
            return

        logging.info("Tickets to add: " + str(new_tickets))

        new_description = self.add_tickets_to_description(issue_description, new_tickets)
        logging.info("Updated issue description: \n" + new_description)
        issue.update(description=new_description)

        logging.info("Issue successfully updated")
        arc.finish()

    @staticmethod
    def add_tickets_to_description(issue_description, new_tickets):
        for ticket in new_tickets:
            issue_description += "\n1. {}".format(ticket)
        return issue_description

    def get_st_issue(self):
        from startrek_client import Startrek
        from startrek_client.exceptions import NotFound

        if not self.Parameters.st_token:
            raise Exception('Startrek access token is missed')

        token = self.Parameters.st_token.data()['token']
        st = Startrek(token=token, useragent=ST_USERAGENT)

        try:
            issue = st.issues[self.Parameters.st_issue]
        except NotFound:
            raise Exception('Release ticket not found')

        return issue

    @staticmethod
    def get_java_release_commit():
        prod_version_tag = get_production_tag(VersionType.JAVA)
        prod_version = prod_version_tag[2:-2]  # strip 1.commit-1

        return prod_version
