# -*- coding: utf-8 -*

import logging
import requests
import os
import subprocess as sp
import time
from datetime import datetime

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.common.urls import get_task_link
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2
from sandbox.projects.partner.utils.arc import Arc

from sandbox.projects.adfox.adfox_ui.deploy import AdfoxUiBuild, AdfoxUiDeploy
from sandbox.projects.adfox.adfox_ui.util.release_integration import AdfoxReleaseIntegration
from sandbox.projects.partner.tasks.e2e_tests.e2e_all_tests import PartnerE2ERunAll
from sandbox.projects.partner.tasks.deploy_release import DeployRelease
from sandbox.projects.partner.tasks.build_perl import PartnerBuildPerl
from sandbox.projects.partner.tasks.build_frontend import BuildFrontend
from sandbox.projects.partner.tasks.finish_frontend import FinishFrontend
from sandbox.projects.partner.tasks.add_tickets_to_release import AddTicketsToRelease
from sandbox.projects.partner.tasks.build_java_docker import PartnerBuildJavaDocker
from sandbox.projects.partner.tasks.screenshooter_deploy import ScreenshooterDeploy
from sandbox.projects.partner.settings import ROBOT_PEREIRO_INFRA_TOKEN
from sandbox.projects.partner.tasks.misc.infra_notifications import InfraNotifications
from sandbox.projects.partner.tasks.misc.task_graph import TaskGraph, NodeStatus

from sandbox.projects.partner.tasks.misc.partner_front_task_base import \
    Callbackable, \
    NotificationLevels, \
    PartnerFrontTaskBase

from sandbox.projects.partner.tasks.misc import \
    configure_internal_ca, \
    find_clone_source, \
    parse_docker_image_name

from sandbox.projects.partner.settings import \
    BUILD_TYPE_BETA, \
    BUILD_TYPE_RELEASE, \
    DEFAULT_CALLBACK_URL, \
    DOCKER_REGISTRY, \
    TEST_STAGE, \
    PREPROD_STAGE, \
    PRODUCTION_STAGE, \
    BACKEND_DOCKER_IMAGE_NAME, \
    FRONTEND_DEFAULT_SOURCE, \
    FRONTEND_DOCKER_IMAGE_NAME, \
    JAVA_DOCKER_HOURGLASS_IMAGE_NAME, \
    JAVA_DOCKER_INT_API_IMAGE_NAME, \
    JAVA_DOCKER_JSON_API_IMAGE_NAME, \
    JAVA_DOCKER_TEST_API_IMAGE_NAME, \
    PI_FRONTEND_LXC_IMAGE, \
    ROBOT_PEREIRO_STARTREK_TOKEN_SECRET, \
    ROBOT_PARTNER_SECRET

from sandbox.projects.partner.tasks.build_all.constant import \
    ST_USERAGENT, ADFOX_CREATE_TICKET_URL, ADFOX_DEPLOY_FAILED_MESSAGE, \
    PI_DEPLOY_TEST_FAILED_MESSAGE, PI_DEPLOY_PREPROD_FAILED_MESSAGE, START_SUCCESS_MESSAGE, START_FAILURE_MESSAGE, \
    PRODUCTION_SHIP_SUCCESS_MESSAGE, PRODUCTION_SHIP_FAILURE_MESSAGE, \
    PREPROD_DEPLOY_FAILURE_MESSAGE, BETA_DEPLOY_SUCCESS_MESSAGE, BETA_DEPLOY_FAILURE_MESSAGE, \
    WAIT_MAX_TIME, JAVA_FROM_BRANCH, JAVA_FROM_CUSTOM_TASK, ARC_BRANCH_PREFIX, \
    FROM_ORIGINAL_TASK_MSG, FROM_ORIGINAL_TASK, NODE_PI_FRONTEND_BUILD, NODE_PI_PERL_BUILD, \
    NODE_PI_JAVA_BUILD, NODE_PI_TEST_DEPLOY, NODE_PI_PREPROD_DEPLOY, NODE_ARCADIA_ADD_TICKETS, NODE_ADFOX_BUILD, \
    NODE_ADFOX_DEPLOY, NODE_PI_SCREENSHOOTER_DEPLOY, NODE_AUTOTESTS, NODE_PI_PRODUCTION_SHIP, \
    NODE_PI_FRONTEND_FINISH, NODE_PI_BETA_DEPLOY, GROUPS_OF_INHERIT_PARAMETERS, TASKS_DEPENDENCIES, \
    ADFOX_LATEST_TAG


def parse_frontend_tag_from_image_name(image_name):
    (name, tag, repo) = parse_docker_image_name(image_name, FRONTEND_DOCKER_IMAGE_NAME)
    return tag


def parse_frontend_version_from_image_tag(tag):
    (version, hash) = tag.split('_')
    return version


def check_inheritance(node_name):
    def check_inheritance_decorator(sub):
        def check_inheritance_wrapper(self):
            logging.debug('check_inheritance(%s) start' % node_name)
            result = None
            contr_flag_name = 'need_run_' + node_name
            contr_flag = getattr(self.Context, contr_flag_name, None)
            logging.debug('Contr flag name (%s) and value (%s)' % (contr_flag_name, contr_flag))
            if self.should_inherit_results(node_name) and (not contr_flag or contr_flag is ctm.NotExists):
                logging.debug('check_inheritance try to inherit result')
                task = self.inherit_results(node_name)
                if task:
                    logging.debug('check_inheritance(%s) found task: %d' % (node_name, task.id))
                    result = task
            if not result:
                for node in TASKS_DEPENDENCIES:
                    if node_name in TASKS_DEPENDENCIES[node]:
                        setattr(self.Context, 'need_run_' + node, True)
                logging.debug('check_inheritance(%s) create new task' % node_name)
                result = sub(self)
            logging.debug('check_inheritance(%s) leave' % node_name)
            return result

        return check_inheritance_wrapper

    return check_inheritance_decorator


def retrier(count, exception_text):
    def retrier_decorator(sub):
        def retrier_wrapper(self):
            logging.debug('retrier start')
            result = None
            tries = count
            last_error = None
            while tries > 0 and not result:
                tries = tries - 1
                try:
                    result = sub(self)
                except Exception as e:
                    last_error = e
                    logging.error('retrier error: %s' % str(e))
                    logging.error('retrier count: %s' % tries)
                time.sleep(1)

            if not result:
                if last_error:
                    raise last_error
                elif exception_text:
                    raise Exception(exception_text)

            logging.debug('retrier leave')
            return result

        return retrier_wrapper

    return retrier_decorator


class ChildTaskFailed(Exception):
    def __init__(self, message, task):
        super(ChildTaskFailed, self).__init__(message)
        self.task = task


def get_image_address(image, tag, with_protocol=False):
    url = '{registry}/{image}:{tag}'.format(registry=DOCKER_REGISTRY, image=image, tag=tag)

    if with_protocol:
        return 'https://' + url

    return url


class BuildAll(PartnerFrontTaskBase):
    """
    Grab parameters, then run component build tasks in parallel, after that deploy
    at preprod and prepare to finally ship to production.
    """

    name = 'PARTNER_BUILD_ALL'

    task_graph = None
    infra_center = None
    adfox_branch = None

    class Requirements(sdk2.Task.Requirements):
        container_resource = PI_FRONTEND_LXC_IMAGE
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('startrek_client', custom_parameters=["requests==2.18.4"]),
        ]
        privileged = True

    class Parameters(PartnerFrontTaskBase.Parameters):
        description = 'Build and deploy partner interface instance'

        with sdk2.parameters.Group('Callback settings', description='Callback settings'):
            callback_url = sdk2.parameters.List(
                'Callback URL',
                description='Lifecycle notification events will be sent to the URL provided',
                default=DEFAULT_CALLBACK_URL,
            )

        with sdk2.parameters.Group('Build settings. Common', description='Common build settings') as common_settings:

            auto_build = sdk2.parameters.Bool(
                'Auto build', description='Build is started automatically (ex.: nightly build)', default=False
            )

            inherit_status = sdk2.parameters.Bool(
                'Inherit status', description='Inherit results and statuses of original task', default=False
            )

            build_from_trunk = sdk2.parameters.Bool(
                'Build from trunk', description='This option will checkout a new rc branch from trunk', default=True
            )

            with inherit_status.value[False]:
                build_type = sdk2.parameters.String('Build type', default=BUILD_TYPE_RELEASE, required=True)
                build_type.choices = [
                    ('Release', BUILD_TYPE_RELEASE),
                    # ('Beta', BUILD_TYPE_BETA)     # Пока не используется
                ]

                with build_type.value[BUILD_TYPE_RELEASE]:
                    with auto_build.value[False]:
                        st_issue = sdk2.parameters.String(
                            'Release ticket, set empty to autocreate', description='PI-NNNNN'
                        )

                        is_delayed_deploy = sdk2.parameters.Bool(
                            'Delayed deploy',
                            description='Build components and create deploy tasks drafts ' 'to run on demand later',
                        )

                    is_hotfix = sdk2.parameters.Bool('Hotfix', description='Hotfix releases may supplant regular ones')

                    with is_hotfix.value[True]:
                        has_regression = sdk2.parameters.Bool(
                            'Regression',
                            description='Set up the regression testing stage (preprod). '
                            'Warning: this will interrupt any regression testing already '
                            'in progress',
                        )

                        should_patch_incomplete_release = sdk2.parameters.Bool(
                            'Patch incomplete release',
                            description='Automatically patch incomplete release branch ' 'with hotfix updates',
                            default=True,
                        )

                        with should_patch_incomplete_release.value[True]:
                            incomplete_release_st_issue = sdk2.parameters.String(
                                'Incomplete release ticket',
                                description='Ticket of release being interrupted by hotfix. '
                                'Leave blank to let task figure it out. '
                                'Note: this will only affect backend branch',
                            )

                    do_not_notify_via_infra = sdk2.parameters.Bool(
                        'DO NOT create infra notifications',
                        description='Switch off infra notifications on build and deploy',
                        default=False,
                    )

                    skip_not_merged = sdk2.parameters.Bool(
                        'Skip not merged Pull Requests',
                        description='Switch on to not die if Release has unmerged PRs',
                        default=False,
                    )

                with build_type.value[BUILD_TYPE_BETA]:
                    deploy_stage_name = sdk2.parameters.String(
                        'Y.Deploy stage name',
                        description='Choose existing (like madamada-partner-production-stage) '
                        'or leave blank to pick arbitrary',
                    )

        with inherit_status.value[False], build_from_trunk.value[False]:
            with build_type.value[BUILD_TYPE_RELEASE]:
                with sdk2.parameters.Group(
                    'Build settings. Adfox', description='Adfox build settings'
                ) as adfox_settings:
                    adfox_branch = sdk2.parameters.String(
                        'ADFOX source',
                        description='Branch in ADFOX git repository. Leave blank to skip ADFOX build & deploy steps.\n'
                                    'Special values: "latest", "release-candidate", "inherit"',
                        default='',
                        required=False,
                    )

            with sdk2.parameters.Group('Build settings. Perl', description='Perl build settings') as perl_settings:
                perl_source = sdk2.parameters.String(
                    'Perl backend source',
                    description='Use branch from arc repo or custom tag',
                    default='branch',
                    required=True,
                )
                perl_source.choices = [
                    ('Build new from arc branch', 'branch'),
                    (FROM_ORIGINAL_TASK_MSG, FROM_ORIGINAL_TASK),
                    ('Use existing image', 'tag'),
                ]

                with perl_source.value['branch']:
                    perl_branch = sdk2.parameters.String(
                        'Custom perl branchd',
                        description='Arc branch id in form users/abc/PI-NNN or trunk',
                        default='trunk',
                        required=True,
                    )

                with perl_source.value['tag']:
                    perl_tag = sdk2.parameters.String(
                        'Perl backend tag',
                        description='Use perl custom image tag (e.g. 2.18.2714)',
                        required=True,
                    )

            with sdk2.parameters.Group('Build settings. Java', description='Java build settings') as java_settings:
                java_source_type = sdk2.parameters.String(
                    'Java source type',
                    required=True,
                    default=JAVA_FROM_BRANCH,
                    description='Source for java. Build new or use prebuilded docker image.',
                )

                java_source_type.choices = [
                    ('Build new java image from branch', JAVA_FROM_BRANCH),
                    (FROM_ORIGINAL_TASK_MSG, FROM_ORIGINAL_TASK),
                    ('Use existing image', 'tag'),
                    ('Get from custom task', JAVA_FROM_CUSTOM_TASK),
                ]

                with java_source_type.value[JAVA_FROM_BRANCH]:
                    java_branch = sdk2.parameters.String(
                        'Custom java branch',
                        description='Provide ID for SUCCESS PARTNER_BUILD_JAVA_DOCKER task. Java docker images would be taken from it results.',
                        default='trunk',
                        required=True,
                    )

                with java_source_type.value[JAVA_FROM_CUSTOM_TASK]:
                    java_docker_task = sdk2.parameters.Integer(
                        'Java docker build task ID',
                        description='Provide ID for SUCCESS PARTNER_BUILD_JAVA_DOCKER task. Java docker images would be taken from it results.',
                        required=True,
                    )

                with java_source_type.value['tag']:
                    java_tag = sdk2.parameters.String(
                        'Java backend tag',
                        description='Use java custom image tag (e.g. 1.5b424545555595668af5f49ec2df4eaa47fc0a1c-1)',
                        required=True,
                    )

            with sdk2.parameters.Group(
                'Build settings. Frontend', description='Frontend build settings'
            ) as frontend_settings:
                frontend_source = sdk2.parameters.String(
                    'Frontend source',
                    description='Use branch from arc or custom image',
                    default='branch',
                    required=True,
                )
                frontend_source.choices = [
                    ('Build new from arc branch', 'branch'),
                    (FROM_ORIGINAL_TASK_MSG, FROM_ORIGINAL_TASK),
                    ('Use existing image', 'tag'),
                ]

                with frontend_source.value['branch']:
                    frontend_branch = sdk2.parameters.String(
                        'Custom frontend branch',
                        description='Use frontend custom image tag'
                        '(e.g. users/kristinasg/hotfix-220428-151100)',
                        default='trunk',
                        required=True,
                    )

                with frontend_source.value['tag']:
                    frontend_tag = sdk2.parameters.String(
                        'Frontend tag',
                        description='Use frontend custom image tag'
                        '(e.g. release-190422-030411_c8dadc11addfdb8f90ead89defdf5eab8d3fca9b)',
                        required=True,
                    )

                # yharnam_source должен быть доступен даже если для сборки берется готовый образ - для запуска автотестов и скриншутера
                yharnam_source = sdk2.parameters.String(
                    'Frontend tests source',
                    description='Branch, tag or commit in yharnam arc repository. \n'
                    'Used as source for for E2E tests to run',
                    default='trunk',
                    required=True,
                )

                tanker_retry_request = sdk2.parameters.Bool('Retry tanker requests', default=True)

        with sdk2.parameters.Group('Secrets') as secret_settings:
            st_token = sdk2.parameters.YavSecret(
                'Startrek OAuth token',
                description='Use startrek to autodetect release ticket when doing hotfix',
                default=ROBOT_PEREIRO_STARTREK_TOKEN_SECRET,
            )

            arc_token = sdk2.parameters.YavSecret(
                'Arc OAuth token', description='Use arc to create release branch', default=ROBOT_PARTNER_SECRET
            )

    class Context(sdk2.Task.Context):
        pi_perl_tag = None
        graph_state = dict()
        docker_images = None
        arc_branch = None
        is_java_task_created = False
        yharnam_source = None
        debug = ''
        is_delayed_deploy = None
        adfox_ticket = None
        st_issue = None

    def get_adfox_branch(self):
        if not self.adfox_branch:
            self.adfox_branch = AdfoxReleaseIntegration.resolve_branch(self.Parameters.adfox_branch)
        return self.adfox_branch

    def get_build_type(self):
        return self.Parameters.build_type

    def set_arc_branch(self, branch_name):
        if self.Context.arc_branch:
            logging.info('Use {} as arc_branch'.format(self.Context.arc_branch))
            return self.Context.arc_branch
        logging.info('Set arc_branch as {}'.format(branch_name))
        self.Context.arc_branch = branch_name
        return branch_name

    @check_inheritance(NODE_ADFOX_BUILD)
    def run_adfox_build(self):
        logging.debug('run_adfox_build start')

        if not self.should_build_adfox():
            logging.debug('run_adfox_build skip')
            return None

        adfox_branch = self.get_adfox_branch()

        adfox_build = AdfoxUiBuild(self, git_branch=adfox_branch)
        logging.debug('run_adfox_build adfox_ui_build task created')

        adfox_build.enqueue()
        logging.debug('run_adfox_build task enqueued')

        return adfox_build

    @check_inheritance(NODE_PI_FRONTEND_BUILD)
    def run_frontend_build(self):
        logging.debug('run_frontend_build start')

        if not self.should_build_frontend():
            logging.debug('run_frontend_build skip')
            return None

        build_type = self.get_build_type()
        is_hotfix = None
        st_issue = None

        if build_type == BUILD_TYPE_RELEASE:
            is_hotfix = self.Parameters.is_hotfix
            st_issue = self.Context.st_issue

        callback_url = self.Parameters.callback_url
        callback_params = self.get_callback_params()
        # tanker_retry_request = self.Parameters.tanker_retry_request

        frontend_build = BuildFrontend(
            self,
            build_type=build_type,
            is_hotfix=is_hotfix,
            st_issue=st_issue,
            arc_ref=self.set_arc_branch(self.Parameters.frontend_branch),
            callback_url=callback_url,
            callback_params=callback_params,
        )

        logging.debug('run_frontend_build task created')
        frontend_build.enqueue()

        logging.debug('run_frontend_build task enqueued')
        return frontend_build

    def add_tickets(self):
        return AddTicketsToRelease(
            self,
            st_issue=self.Context.st_issue,
            st_token=self.Parameters.st_token,
            arc_token=self.Parameters.arc_token,
            arc_release_branch=self.Context.arc_branch,
        ).enqueue()

    def get_arc_token(self):
        logging.debug('getting arc token')
        if not self.Parameters.arc_token:
            raise Exception('Arc access token is missed')

        token = self.Parameters.arc_token.data()['arc_token']
        logging.debug('success getting arc token')
        return token

    def run_java_build(self):
        logging.debug('run_java_build')

        task = PartnerBuildJavaDocker(
            self,
            arc_release_branch=self.set_arc_branch(self.Parameters.java_branch),
        )
        task.enqueue()
        self.Context.is_java_task_created = True
        logging.debug('run_java_build: task enqueued')
        return task

    def set_java_params_from_task(self, source_type):
        task_docker = None
        if source_type == JAVA_FROM_CUSTOM_TASK:
            logging.debug('Trying to find PARTNER_BUILD_JAVA_DOCKER task %s' % self.Parameters.java_docker_task)
            task_docker = PartnerBuildJavaDocker.find(
                id=self.Parameters.java_docker_task,
                children=True,
                status=ctt.Status.SUCCESS,
            ).limit(1)

        if not task_docker or not task_docker.first():
            raise Exception('No task could be found.')

        task_object = task_docker.first()
        self.Context.docker_images = task_object.Parameters.docker_images
        logging.debug('set_java_params_from_task get java docker images from some old task %s' % task_docker.first().id)

    @check_inheritance(NODE_PI_JAVA_BUILD)
    def make_java(self):
        logging.debug('make_java start')

        if self.Parameters.java_source_type == JAVA_FROM_CUSTOM_TASK:
            self.set_java_params_from_task(source_type=str(self.Parameters.java_source_type))
            return None

        if self.Parameters.java_source_type == 'tag':
            self.Context.docker_images = self.get_result_java()
            return None

        if self.Parameters.java_source_type == JAVA_FROM_BRANCH:
            return self.run_java_build()
        raise Exception('Could do nothing with such strange')

    def should_build_adfox(self):
        return self.Parameters.adfox_branch != ADFOX_LATEST_TAG

    def should_build_frontend(self):
        return self.Parameters.frontend_source == 'branch'

    @check_inheritance(NODE_AUTOTESTS)
    def run_autotests(self):
        logging.debug('run_autotests start')
        rc_branch = self.Context.arc_branch
        should_test_pi = True
        should_test_adfox = self.should_build_adfox()

        adfox_branch = self.get_adfox_branch()
        java_branch = rc_branch

        partner_branch = rc_branch
        yharnam_source = self.Parameters.yharnam_source

        frontend_image = self.get_result_frontend_image_name()

        java_docker_images = self.get_result_java()

        callback_url = self.Parameters.callback_url
        callback_params = self.get_callback_params()

        task = PartnerE2ERunAll(
            self,
            priority=self.Parameters.priority,
            should_test_adfox=should_test_adfox,
            should_test_pi=should_test_pi,
            adfox_branch=adfox_branch,
            java_branch=java_branch,
            partner_branch=partner_branch,
            yharnam_source=yharnam_source,
            use_custom_frontend_image=True,
            frontend_image=frontend_image,
            use_custom_deb_version=True,
            backend_deb_version=self.get_perl_tag(),
            use_custom_java=True,
            java_docker_images=java_docker_images,
            callback_url=callback_url,
            callback_params=callback_params,
            is_release=True,
            st_issue=self.Context.st_issue,
        )

        logging.debug('run_autotests task created: %s', str(task.id))
        task.enqueue()
        logging.debug('run_autotests task enqueued')

        return task

    def get_callback_params(self):
        callback_params = self.Parameters.callback_params
        st_issue = self.Context.st_issue

        if 'st_issue' not in callback_params and st_issue:
            callback_params['st_issue'] = st_issue
        if 'ticketId' not in callback_params and st_issue:
            callback_params['ticketId'] = st_issue

        return callback_params

    @check_inheritance(NODE_PI_SCREENSHOOTER_DEPLOY)
    def run_screenshooter_deploy(self):
        logging.debug('run_screenshooter_deploy start')

        # params for screenshooter task
        arc_ref = self.get_yharnam_rc_branch()
        callback_url = self.Parameters.callback_url
        callback_params = self.get_callback_params()

        # params for screenshooter stage deploy
        frontend_tag = self.get_rc_frontend_tag()
        st_issue = self.Context.st_issue
        java_docker_images = self.get_result_java()

        task = ScreenshooterDeploy(
            self,
            java_docker_images=java_docker_images,
            perl_tag=self.get_perl_tag(),
            frontend_tag=frontend_tag,
            st_issue=st_issue,
            arc_ref=arc_ref,
            callback_url=callback_url,
            callback_params=callback_params,
        )

        logging.debug('run_screenshooter_deploy task created: %s' % str(task.id))
        task.enqueue()
        logging.debug('run_screenshooter_deploy task enqueued')

        return task

    @check_inheritance(NODE_ADFOX_DEPLOY)
    def run_adfox_deploy(self):
        logging.debug('run_adfox_deploy start')

        git_branch = self.get_adfox_branch()

        if not self.should_build_adfox():
            logging.debug('run_adfox_deploy skip')
            return None

        task = AdfoxUiDeploy(self, git_branch=git_branch)
        logging.debug('run_adfox_deploy task created')

        if not self.Context.is_delayed_deploy:
            task.enqueue()
            logging.debug('run_adfox_deploy task enqueued')

        return task

    def get_result_java(self):
        tag = self.Parameters.java_tag
        if tag:
            return ';'.join(
                [
                    get_image_address(JAVA_DOCKER_TEST_API_IMAGE_NAME, tag),
                    get_image_address(JAVA_DOCKER_INT_API_IMAGE_NAME, tag),
                    get_image_address(JAVA_DOCKER_JSON_API_IMAGE_NAME, tag),
                    get_image_address(JAVA_DOCKER_HOURGLASS_IMAGE_NAME, tag),
                ]
            )

        docker_images = None
        if self.Context.is_java_task_created:
            build_java_task = self.find_child_task(NODE_PI_JAVA_BUILD)
            if build_java_task:
                docker_images = build_java_task.Parameters.docker_images
        else:
            docker_images = self.Context.docker_images

        if not docker_images:
            raise Exception('Can`t get java info')

        return docker_images

    def checkout_release_branch(self):
        arc_release_branch = (
            ARC_BRANCH_PREFIX + self.Context.st_issue + '-release-' + datetime.now().strftime('%H%M%S')
        )
        # init arc branch
        logging.debug('initializing arc branch %s' % arc_release_branch)
        arc = Arc(path='.', token=self.get_arc_token())
        logging.debug('checkout release branch "%s" based on "%s"' % (arc_release_branch, 'trunk'))
        arc.checkout(branch=arc_release_branch, create_branch=True, base_branch='trunk')
        logging.debug('push release branch "%s"' % arc_release_branch)
        arc.push(upstream=arc_release_branch)
        logging.debug('release branch pushed')
        self.Context.arc_branch = arc_release_branch
        return self.Context.arc_branch

    def get_perl_tag(self):
        tag = self.Parameters.perl_tag
        if tag:
            return tag
        task = self.find_child_task(NODE_PI_PERL_BUILD)
        if task:
            return task.Parameters.docker_image.split(':')[-1]

    @check_inheritance(NODE_PI_PERL_BUILD)
    def run_pi_perl_build(self):
        if self.Parameters.perl_source == 'tag':
            logging.debug('Skip build perl')
            self.Context.pi_perl_tag = self.Parameters.perl_tag
            return None

        # release_st_ticket_on_preprod = None

        # if self.Parameters.is_hotfix and self.Parameters.should_patch_incomplete_release:
        #     release_st_ticket_on_preprod = self.get_incomplete_release_st_issue()

        task = PartnerBuildPerl(
            self,
            release_ticket=self.Context.st_issue,
            branch=self.set_arc_branch(self.Parameters.perl_branch),
            skip_not_merged=self.Parameters.skip_not_merged
        )
        task.enqueue()

        return task

    @check_inheritance(NODE_PI_TEST_DEPLOY)
    def run_pi_test_deploy(self):
        """
        Deploy to TS.
        """

        logging.debug('run_pi_test_deploy start')

        java_docker_images = self.get_result_java()

        logging.debug(java_docker_images)

        frontend_tag = self.get_rc_frontend_tag()

        task = DeployRelease(
            self,
            description="TS\n" + self.Parameters.description,
            stage=TEST_STAGE,
            st_issue=self.Context.st_issue,
            frontend_tag=frontend_tag,
            frontend_branch=self.get_yharnam_rc_branch(),
            java_docker_images=java_docker_images,
            perl_tag=self.get_perl_tag(),
        )
        logging.debug('run_pi_test_coupled_build task created')

        task.save()
        logging.debug('run_pi_test_coupled_build task saved')

        return task

    @check_inheritance(NODE_PI_PREPROD_DEPLOY)
    def run_pi_preprod_deploy(self):
        """
        Deploy to preprod.
        """

        logging.debug('run_pi_preprod_deploy start')

        java_docker_images = self.get_result_java()

        logging.debug(java_docker_images)

        update_preprod = self.has_regression

        frontend_tag = self.get_rc_frontend_tag()

        task = DeployRelease(
            self,
            description="Preprod\n" + self.Parameters.description,
            stage=PREPROD_STAGE,
            st_issue=self.Context.st_issue,
            frontend_tag=frontend_tag,
            frontend_branch=self.get_yharnam_rc_branch(),
            java_docker_images=java_docker_images,
            perl_tag=self.get_perl_tag(),
            update_preprod=update_preprod,
        )
        logging.debug('run_pi_preprod_coupled_build task created')

        if not self.Context.is_delayed_deploy:
            task.enqueue()
            logging.debug('run_pi_preprod_coupled_build task enqueued')

        return task

    def prepare_production_ship(self):
        """
        Create task to ship previously build and tested PI to production.
        ADFOX is deployed independently.
        """

        logging.debug('prepare_production_ship_task start')

        description = self.Parameters.description

        frontend_tag = self.get_rc_frontend_tag()

        st_issue = self.Context.st_issue

        java_docker_images = self.get_result_java()

        task = DeployRelease(
            self,
            description="Production\n" + description,
            java_docker_images=java_docker_images,
            perl_tag=self.get_perl_tag(),
            frontend_tag=frontend_tag,
            st_issue=st_issue,
            stage=PRODUCTION_STAGE,
            do_not_notify_via_infra=self.Parameters.do_not_notify_via_infra,
        )
        logging.debug('prepare_production_ship_task task created')

        task.save()
        logging.debug('prepare_production_ship_task task draft saved: %s' % str(task.id))

        message = []
        task_url = get_task_link(task.id)
        message.append('Задача на выкладку в прод: {url}'.format(url=task_url))
        self.make_success_callback('\n'.join(message), NODE_PI_PRODUCTION_SHIP)

        return task

    def production_ship_callback(self, status, task_id):
        if status == NodeStatus.SUCCESS:
            build_type = self.get_build_type()
            logging.debug('on_success build_type=%s' % build_type)

            # Запуск таски финализации фронта
            frontend_finish_task = self.get_task_from_graph(self.task_graph, NODE_PI_FRONTEND_FINISH)
            frontend_finish_task.enqueue()

            if build_type == BUILD_TYPE_BETA:
                message = BETA_DEPLOY_SUCCESS_MESSAGE
                step = NODE_PI_BETA_DEPLOY
            else:
                message = PRODUCTION_SHIP_SUCCESS_MESSAGE
                step = NODE_PI_PRODUCTION_SHIP

            self.make_success_callback(message, step)
            self.publish_results()

    def send_failure_message(self, task_name, message):
        task = self.find_child_task(task_name)
        self.send_message(message.format(task_url=get_task_link(task.id)), NotificationLevels.URGENT)
        logging.error('check_child_task_success_report_only ' 'task "%s" failed to finish' % task_name)

    def find_incomplete_release_st_issue(self):
        logging.debug('find_incomplete_release_st_issue start')

        if not self.Parameters.st_token:
            raise Exception('Startrek access token is missed')

        token = self.Parameters.st_token.data()['token']

        with sdk2.helpers.ProcessLog(self, logger='startrek') as pl:
            script = os.path.realpath(os.path.join(os.path.dirname(__file__), '../../utils/startrek.py'))
            st_issue = self.Context.st_issue
            cmd = [script, 'active-release', '--ignore', st_issue, '--oauth', token]
            return sp.check_output(cmd, stderr=pl.stdout).strip()

    def get_incomplete_release_st_issue(self):
        if self.Parameters.incomplete_release_st_issue:
            return self.Parameters.incomplete_release_st_issue
        return self.find_incomplete_release_st_issue()

    def init_infra_notifications_center(self):
        logging.info("=== Init infra notifications token ===")
        secret = sdk2.yav.Secret(ROBOT_PEREIRO_INFRA_TOKEN)
        token = secret.data()['token']
        return InfraNotifications(oauth_token=token, ticket=self.Context.st_issue)

    def pi_deploy_test_callback(self, status, task_id):
        if status == NodeStatus.FAILURE:
            self.send_failure_message(NODE_PI_TEST_DEPLOY, PI_DEPLOY_TEST_FAILED_MESSAGE)

    def pi_deploy_preprod_callback(self, status, task_id):
        if status == NodeStatus.FAILURE:
            self.send_failure_message(NODE_PI_PREPROD_DEPLOY, PI_DEPLOY_PREPROD_FAILED_MESSAGE)

    def pi_perl_build_callback(self, status, task_id):
        if status == NodeStatus.SUCCESS:
            self.Context.pi_perl_tag = self.find_child_task(NODE_PI_PERL_BUILD).Parameters.build_version

    def adfox_deploy_callback(self, status, task_id):
        if status == NodeStatus.FAILURE:
            self.send_failure_message(NODE_ADFOX_DEPLOY, ADFOX_DEPLOY_FAILED_MESSAGE)

    def prepare(self):
        self.infra_center = self.init_infra_notifications_center()

        self.task_graph = TaskGraph(state=self.Context.graph_state)

        self.task_graph.add_node(
            NODE_PI_FRONTEND_BUILD,
            TASKS_DEPENDENCIES[NODE_PI_FRONTEND_BUILD],
            self.run_frontend_build,
        )
        self.task_graph.add_node(
            NODE_PI_PERL_BUILD,
            TASKS_DEPENDENCIES[NODE_PI_PERL_BUILD],
            self.run_pi_perl_build,
            self.pi_perl_build_callback,
        )
        self.task_graph.add_node(
            NODE_PI_JAVA_BUILD,
            TASKS_DEPENDENCIES[NODE_PI_JAVA_BUILD],
            self.make_java,
        )
        self.task_graph.add_node(
            NODE_ARCADIA_ADD_TICKETS,
            TASKS_DEPENDENCIES[NODE_ARCADIA_ADD_TICKETS],
            self.add_tickets,
        )
        self.task_graph.add_node(
            NODE_ADFOX_BUILD,
            TASKS_DEPENDENCIES[NODE_ADFOX_BUILD],
            self.run_adfox_build,
        )

        self.task_graph.add_node(
            NODE_PI_TEST_DEPLOY,
            TASKS_DEPENDENCIES[NODE_PI_TEST_DEPLOY],
            self.run_pi_test_deploy,
            self.pi_deploy_test_callback,
        )

        self.task_graph.add_node(
            NODE_PI_PREPROD_DEPLOY,
            TASKS_DEPENDENCIES[NODE_PI_PREPROD_DEPLOY],
            self.run_pi_preprod_deploy,
            self.pi_deploy_preprod_callback,
        )

        self.task_graph.add_node(
            NODE_PI_PRODUCTION_SHIP,
            TASKS_DEPENDENCIES[NODE_PI_PRODUCTION_SHIP],
            self.prepare_production_ship,
            self.production_ship_callback,
        )

        self.task_graph.add_node(
            NODE_PI_FRONTEND_FINISH,
            TASKS_DEPENDENCIES[NODE_PI_FRONTEND_FINISH],
            self.finish_frontend,
        )

        self.task_graph.add_node(
            NODE_ADFOX_DEPLOY,
            TASKS_DEPENDENCIES[NODE_ADFOX_DEPLOY],
            self.run_adfox_deploy,
            self.adfox_deploy_callback,
        )

        self.task_graph.add_node(
            NODE_PI_SCREENSHOOTER_DEPLOY,
            TASKS_DEPENDENCIES[NODE_PI_SCREENSHOOTER_DEPLOY],
            self.run_screenshooter_deploy,
        )
        self.task_graph.add_node(
            NODE_AUTOTESTS,
            TASKS_DEPENDENCIES[NODE_AUTOTESTS],
            self.run_autotests,
        )

    def process_task_graph(self):
        self.task_graph.process()

        logging.debug('graph_state after all')
        logging.debug(self.task_graph.serialize())

        self.Context.graph_state = self.task_graph.serialize()
        self.Context.save()

        tasks_to_wait = self.task_graph.get_waiting_task_ids()

        logging.debug('tasks_to_wait')
        logging.debug(tasks_to_wait)

        if len(tasks_to_wait):
            raise sdk2.WaitTask(tasks_to_wait, ctt.Status.Group.FINISH, timeout=WAIT_MAX_TIME, wait_all=False)

    # NOTE
    # Данный метод можно вызывать только из on_enqueue и только при первом запуске
    # Потом уже self.Parameters становится read-only
    def copy_parameters(self, task, group_name):
        if group_name in GROUPS_OF_INHERIT_PARAMETERS:
            # Если для этой дочерней таски надо унаследовать параметры запуска из оригинальной таски
            # То пробегаем по списку параметров, которые надо скопировать
            # Сохраняем оригинальное значение в Context с префиксом original
            # И заменяем параметры текущей таски на те, что в оригинальной
            for key in GROUPS_OF_INHERIT_PARAMETERS[group_name]:
                if hasattr(task.Parameters, key):
                    setattr(self.Context, 'original_' + key, getattr(self.Parameters, key))
                    setattr(self.Parameters, key, getattr(task.Parameters, key))
                else:
                    if hasattr(task.Context, key):
                        setattr(self.Context, 'original_' + key, getattr(self.Context, key))
                        setattr(self.Context, key, getattr(task.Context, key))

    def should_inherit_results(self, task_name):
        # Установлена галка полного переиспользования результатов сборки
        if self.Parameters.inherit_status:
            logging.debug('found global inherit status flag')
            return True

        # Установлена галка переиспользование результатов сборки фронта, явы, бека или адфокса
        if task_name in GROUPS_OF_INHERIT_PARAMETERS:
            key_name = GROUPS_OF_INHERIT_PARAMETERS[task_name][0]
            original_key_name = 'original_' + key_name
            # При первом запуске параметры имеют значения заданные пользователем
            # При этом если надо что-то унаследовать, то они копируюются в Context с префиксом original
            # А на их место копируются параметры из оригинальной таски
            # Из-за этого при первом запуске проверяем в Parameters (if)
            # А при последующих в Context, так как в Parameters уже могут быть другие значения
            value = getattr(self.Context, original_key_name, None)
            if value is None or value is ctm.NotExists:
                value = getattr(self.Parameters, key_name, '')
            if value == FROM_ORIGINAL_TASK:
                return True

        # Никаких галок переиспользования не установлено
        return False

    def inherit_results(self, task_name):
        # Поиск таски в графе подпроцессов оригинальной таски
        task = self.find_child_task(task_name, self.Context.copy_of)

        # Проверка, что она завершилась успешно (иначе пересборка)
        if task and task.status == ctt.Status.SUCCESS:
            logging.debug('use found task %d as "%s"' % (task.id, task_name))
            self.send_message('Использованы результаты предыдущей таски для %s' % task_name)
            return task

        logging.debug('appropriate task "%s" not found' % task_name)
        return None

    def on_prepare(self):
        logging.debug('on_prepare enter')
        configure_internal_ca()
        self.recover_from_crash()
        logging.debug('on_prepare leave')

    def recover_from_crash(self):
        """
        Определить факт перезапуска и попытаться частично восстановить результаты для гладкого
        возобновления работы (по возможности, с точки отказа).
        """
        logging.debug('recover_from_crash enter')
        if self.should_inherit_results(NODE_PI_FRONTEND_BUILD):
            logging.debug('recover_from_crash use inherit, leave')
            return
        source = find_clone_source(self)
        if not (source and self.is_release_rebuild(source)):
            logging.debug('recover_from_crash not a rebuild, leave')
            return

        frontend_source = self.Parameters.frontend_source
        yharnam_source = self.Parameters.yharnam_source
        if frontend_source == 'branch' and yharnam_source == FRONTEND_DEFAULT_SOURCE:
            self.resume_frontend_rc_branch(source)

        logging.debug('recover_from_crash leave')

    def is_release_rebuild(self, source):
        return (
            self.Parameters.build_type == BUILD_TYPE_RELEASE
            and self.Parameters.build_type == source.Parameters.build_type
            and self.Context.st_issue == source.Context.st_issue
            and bool(self.Parameters.is_hotfix) == bool(source.Parameters.is_hotfix)
        )

    def resume_frontend_rc_branch(self, source):
        """
        Попытаться возобновить сборку фронта с частичными результатами неудачного предыдущего запуска.
        Сборка фронта при каждом новом запуске отводит специальную ветку RC-ветку, которой нужно
        следовать до окончания релиза, чтобы не собирать каждый раз постоянно растущий транк.
        Метод находит RC-ветку предыдущего запуска сборки фронта в данном релизе, чтобы проставить ее
        в таску новой сборки фронта.
        """
        logging.debug('resume_frontend_rc_branch enter')

        task_graph = TaskGraph(state=source.Context.graph_state)
        build_frontend = self.get_task_from_graph(task_graph, NODE_PI_FRONTEND_BUILD)

        if not build_frontend:
            logging.debug('resume_frontend_rc_branch source FRONTEND_BUILD subtask not found, leave')
            return

        rc_branch = self.Context.arc_branch
        if not rc_branch:
            logging.debug('resume_frontend_rc_branch source FRONTEND_BUILD subtask did not yield rc_branch, leave')
            return

        logging.debug('resume_frontend_rc_branch rc_branch=%s' % rc_branch)
        self.Context.yharnam_source = rc_branch
        self.Context.save()
        logging.debug('resume_frontend_rc_branch leave')

    def check_is_beta(self):
        build_type = self.get_build_type()

        if build_type == BUILD_TYPE_BETA:
            raise Exception('Build type beta is not implemented')

    def on_enqueue(self):
        # При первом запуске проверяем, нужно ли что-то наследовать из оригинальной таски
        try:
            with self.memoize_stage.check_inheritance:
                inherit = False
                # Сначала проверяем нужно ли вообще это делать в целом или для какого-то частного случая
                for group_name in GROUPS_OF_INHERIT_PARAMETERS:
                    if self.should_inherit_results(group_name):
                        inherit = True
                        break
                if inherit:
                    # Если нужно, то проводим дополнительные проверки и операции
                    if not self.Context.copy_of:
                        raise Exception('Inherit status option is avalable only for copied task')
                    self.Context.debug += "\n" + 'try to find original task %d' % self.Context.copy_of
                    original_task = sdk2.Task.find(id=self.Context.copy_of).limit(1).first()
                    if original_task:
                        logging.debug('found original task %d' % original_task.id)

                        # Если всё прошло успешно, то копируем необходимые параметры из оригинальной таски
                        for group_name in GROUPS_OF_INHERIT_PARAMETERS:
                            if self.should_inherit_results(group_name):
                                self.Context.debug += "\n" + 'Copy params of group "%s"' % group_name
                                self.copy_parameters(original_task, group_name)
                    else:
                        self.Context.debug += "\n" + 'not found original task'
                        raise Exception('Inherit status option is used, but not found original task')

            with self.memoize_stage.start_build:
                self.send_message(START_SUCCESS_MESSAGE % str(self.id), NotificationLevels.URGENT)
        except Exception as ex:
            self.send_message(START_FAILURE_MESSAGE % self.id, NotificationLevels.URGENT)
            raise Exception(str(ex) + self.Context.debug)

    def on_execute(self):
        logging.debug('on_execute start')

        # Должно быть до super.on_execute, потому что в super.on_execute используются номера тикетов
        with self.memoize_stage.check_tickets:
            self.Context.is_delayed_deploy = self.Parameters.is_delayed_deploy or self.Parameters.is_hotfix
            self.Context.st_issue = self.Parameters.st_issue
            with self.memoize_stage.check_auto_build:
                if self.Parameters.auto_build:
                    af_ticket = self.create_adfox_ticket()
                    if af_ticket:
                        self.Context.adfox_ticket = af_ticket
                        self.Parameters.description = self.Parameters.description + "\nAdfox ticket: " + af_ticket
                if not self.Context.st_issue:
                    pi_ticket = self.create_pi_ticket()
                    self.Context.st_issue = pi_ticket
                    self.Parameters.description += "\nPI ticket: " + pi_ticket
            super(BuildAll, self).pseudo_on_enqueue()

        super(BuildAll, self).on_execute()
        with self.memoize_stage.check_beta:
            self.check_is_beta()

        if self.Parameters.build_from_trunk:
            self.checkout_release_branch()

        self.prepare()

        self.process_task_graph()

        self.infra_center.resolve_all_created_events()
        logging.debug('on_execute leave')

    @retrier(3, "Can't create adfox ticket")
    def create_adfox_ticket(self):
        af_ticket = None
        r = requests.get(ADFOX_CREATE_TICKET_URL)
        logging.debug('get adfox ticket')
        logging.debug(str(r.content))
        logging.debug(str(r.status_code))
        if r.status_code == 200:
            af_ticket = r.content

        return af_ticket

    @retrier(3, "Can't create PI ticket")
    def create_pi_ticket(self):
        if not self.Parameters.st_token:
            raise Exception('Startrek access token is missed')

        from startrek_client import Startrek

        token = self.Parameters.st_token.data()['token']
        st = Startrek(token=token, useragent=ST_USERAGENT)
        issue = st.issues.create(queue='PI', summary='Auto release', type='release', description="Auto release\n")

        return issue.key

    def get_result_deb_version(self):
        return self.Context.pi_perl_tag

    def find_child_task(self, name, parent_task=None):
        logging.debug('find_child_task start')
        task_graph = None
        result = None

        # Указана родительская таска, начиная с которой надо искать
        if parent_task:
            logging.debug('try to find parent task %d' % parent_task)
            task = sdk2.Task.find(id=parent_task).limit(1).first()
            task_graph = TaskGraph(state=task.Context.graph_state)
        else:
            logging.debug('use current task')
            task_graph = self.task_graph

        result = self.get_task_from_graph(task_graph, name)
        logging.debug('find_child_task leave')
        return result

    @staticmethod
    def get_task_from_graph(task_graph, node_name):
        logging.debug('get_task_from_graph start')
        result = None
        if not task_graph or not task_graph.has_node(node_name):
            logging.debug('get_task_from_graph node %s is not defined' % node_name)
            return None

        task_id = task_graph.get_node_task_id(node_name)
        if task_id:
            logging.debug('try to find task %d' % task_id)
            result = sdk2.Task.find(id=task_id).limit(1).first()
        else:
            logging.debug('not fond task "%s"' % node_name)

        logging.debug('get_task_from_graph leave')
        return result

    def get_rc_frontend_tag_or_version(self, param):
        """
        Get RC frontend tag or version set by frontend build task
        """
        build_type = self.get_build_type()
        if build_type != BUILD_TYPE_RELEASE:
            return None

        is_prebuilt = bool(self.Parameters.frontend_tag)
        if is_prebuilt:
            if param == 'version':
                return parse_frontend_version_from_image_tag(self.Parameters.frontend_tag)
            elif param == 'tag':
                return self.Parameters.frontend_tag

        task = self.find_child_task(NODE_PI_FRONTEND_BUILD)
        if task:
            if param == 'version':
                return task.Parameters.version
            elif param == 'tag':
                return parse_frontend_tag_from_image_name(task.Parameters.image_name)

        raise Exception('Frontend tag not found')

    def get_rc_frontend_tag(self):
        """
        Get RC frontend tag set by frontend build task
        """
        return self.get_rc_frontend_tag_or_version('tag')

    def get_rc_frontend_version(self):
        """
        Get RC frontend version set by frontend build task
        """
        return self.get_rc_frontend_tag_or_version('version')

    def get_result_frontend_image_name(self):
        tag = self.Parameters.frontend_tag
        if tag:
            return get_image_address(FRONTEND_DOCKER_IMAGE_NAME, tag)
        task = self.find_child_task(NODE_PI_FRONTEND_BUILD)
        if task:
            return task.Parameters.image_name

    def get_yharnam_rc_branch(self):
        branch = self.Context.arc_branch
        logging.info('Current rc branch is {}'.format(branch))
        return branch

    def get_result_backend_image_name(self):
        tag = self.get_result_deb_version()
        if not tag:
            return None

        return get_image_address(BACKEND_DOCKER_IMAGE_NAME, tag, True)

    def was_shipment_started(self):
        task = self.find_child_task(NODE_PI_PRODUCTION_SHIP)
        if not task:
            return False
        return task.status != ctt.Status.Group.DRAFT

    @property
    def error_callback_title(self):
        build_type = self.get_build_type()
        if build_type == BUILD_TYPE_RELEASE:
            if self.was_shipment_started():
                return PRODUCTION_SHIP_FAILURE_MESSAGE
            return PREPROD_DEPLOY_FAILURE_MESSAGE

        return BETA_DEPLOY_FAILURE_MESSAGE

    def publish_results(self):
        pass

    def make_success_callback(self, message, step='finish'):
        build_type = self.get_build_type()
        frontend_image = self.get_result_frontend_image_name()
        backend_image = self.get_result_backend_image_name()

        payload = {
            'step': step,
            'status': 'success',
            'message': message,
            'backend_image': backend_image,
            'frontend_image': frontend_image,
        }

        if build_type == BUILD_TYPE_RELEASE:
            frontend_version = self.get_rc_frontend_version()

            deb_version = None
            if step == NODE_PI_PRODUCTION_SHIP:
                deb_version = self.get_result_deb_version()

            payload.update(
                {
                    'deb_version': deb_version,
                    'frontend_version': frontend_version,
                }
            )
        else:
            deploy_stage_name = self.Parameters.deploy_stage_name
            payload.update({'deploy_stage_name': deploy_stage_name})

        self.make_callback(payload)

    def make_callback(self, data, type=NotificationLevels.INFO):
        payload = self.get_callback_context()
        payload.update(data)
        Callbackable.make_callback(self, payload, type)

    def get_callback_context(self):
        build_type = self.Parameters.build_type
        st_issue = self.Context.st_issue

        return {'build_type': build_type, 'st_issue': st_issue, 'ticketId': st_issue}

    def get_metrics_release_id(self):
        return '{}|{}|{}|{}|{}'.format(
            'auto' if self.Parameters.auto_build else 'manual',
            'hotfix' if self.Parameters.is_hotfix else 'planed',
            self.Context.st_issue,
            self.Context.adfox_ticket,
            self.Context.arc_branch
        )

    @property
    def has_regression(self):
        """
        Нужен ли запуск тестирования асессорами
        """
        if self.Parameters.build_type != BUILD_TYPE_RELEASE:
            return False
        return not self.Parameters.is_hotfix or self.Parameters.has_regression

    @check_inheritance(NODE_PI_FRONTEND_FINISH)
    def finish_frontend(self):
        task = FinishFrontend(
            self,
            description=self.Parameters.description,
            arcadia_rc_branch=self.get_yharnam_rc_branch(),
            frontend_version=self.get_rc_frontend_version(),
        )

        task.save()
        return task
