# coding: utf-8

import os
import shutil

import sandbox.common.types.misc as ctm
import sandbox.sdk2.helpers
from sandbox import sdk2
from sandbox.projects.adfox.adfox_ui.metrics import AnalyzableTask

from sandbox.projects.partner.settings import ROBOT_PARTNER_SECRET, PI_LXC_WITH_DOCKER
from sandbox.projects.partner.tasks.misc.docker_registry import DockerRegistry
from sandbox.projects.partner.utils.arc import Arc
from sandbox.projects.partner.utils.docker import docker_login, docker_logout, docker_build, docker_push

TEMPLATE_DOCKERFILE_PATH = "configs/deploy_backend/%s/Dockerfile"
REGISTRY = "registry.yandex.net"
BASE_REGISTRY_TAG = "partners/perl-backend"
DEFAULT_UBUNTU_VERSION = 'bionic'
DEFAULT_DOCKER_VERSION = 'auto'
UBUNTU_VERSIONS = ['precise', 'bionic']


def prepare_dir(path):
    if os.path.exists(path):
        shutil.rmtree(path)
    else:
        os.makedirs(path)


def get_image_tag(version):
    return '{}/{}:{}'.format(REGISTRY, BASE_REGISTRY_TAG, version)


class BuildDockerImage(AnalyzableTask):
    """
    Builds docker image from Arcadia.
    Inspired by BUILD_MARKET_DOCKER_WTO
    """
    checkout_path = 'partner2'
    name = 'PARTNER_BUILD_DOCKER'

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        container_resource = PI_LXC_WITH_DOCKER
        # Тесты показали что разницы нету
        # client_tags = ctc.Tag.SSD

    class Parameters(AnalyzableTask.Parameters):
        description = 'Build partner backend docker image. Push to registry.'
        ubuntu_version = sdk2.parameters.String(
            'Ubuntu version (def. bionic)',
            choices=[(x, x) for x in UBUNTU_VERSIONS],
            default=DEFAULT_UBUNTU_VERSION,
            required=True
        )
        version = sdk2.parameters.String(
            "Version",
            required=True,
            description="For ex.: 2.18.2000, auto (sha will be used)",
            default=DEFAULT_DOCKER_VERSION,
        )
        force_build = sdk2.parameters.Bool(
            "Force rebuild package",
            required=False,
            description="Ignore existing package in registry and rebuild anyway.",
            default=False,
        )

        docker_token = sdk2.parameters.YavSecret(
            'Docker Registry OAuth token',
            description='Docker token for robot-partner',
            default='{}#docker_token'.format(ROBOT_PARTNER_SECRET),
            required=True,
        )

        with sdk2.parameters.Group("Arc repository parameters") as arc_block:
            arc_branch = sdk2.parameters.String(
                "Arc Branch",
                required=True,
                description="For ex.: releases/partner/PI-XXXXX-release-XXXXXX",
                default="trunk",
            )
            arc_project_path = sdk2.parameters.String(
                "Path to project",
                required=True,
                description="from arcadia/..",
                default="partner/perl/partner2",
            )
            arc_token = sdk2.parameters.YavSecret(
                'Arc OAuth token',
                required=True,
                description='Use arc to create release branch',
                default='{}#arc_token'.format(ROBOT_PARTNER_SECRET),
            )

        with sdk2.parameters.Output:
            built_new_image = sdk2.parameters.String("True - if a new image was built, False - if found in the registry")
            backend_image_name = sdk2.parameters.String("Name of the backend docker image")

    def exec_command(self, command, logname='exec_log', cwd=None):
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logname) as process_log:
            return sandbox.sdk2.helpers.subprocess.Popen(
                command,
                stdout=process_log.stdout,
                stderr=process_log.stdout,
                cwd=cwd
            ).wait()

    def on_execute(self):
        super(BuildDockerImage, self).on_execute()

        version = self.calc_version()
        if not self.need_build(version):
            self.set_output_parameters(False, version)
            return

        root_path = str(sdk2.task.Task.current.path())
        export_path = os.path.join(root_path, self.checkout_path, 'export')
        arc_mount_path = os.path.join(root_path, self.checkout_path, 'mount')
        src_path, hash_commit = self.export_src(export_path, arc_mount_path)

        # recalc because now we know hash_commit
        version = self.calc_version(hash_commit)
        if not self.need_build(version):
            self.set_output_parameters(False, version)
            return

        docker_tag = get_image_tag(version)
        self.prepare_content(src_path)

        docker_login('robot-partner', self.Parameters.docker_token.value())
        try:
            docker_file_path = TEMPLATE_DOCKERFILE_PATH % self.Parameters.ubuntu_version
            docker_build(path=src_path, tag=docker_tag, build_args=['VERSION={}'.format(version)],
                         docker_file=docker_file_path)
            docker_push(docker_tag)
        finally:
            docker_logout()

        self.set_output_parameters(True, version)

    def need_build(self, version):
        if version == DEFAULT_DOCKER_VERSION:
            return True

        registry = DockerRegistry()
        return self.Parameters.force_build or not registry.check_package_exist(BASE_REGISTRY_TAG, version)

    def calc_version(self, hash_commit=None):
        if hash_commit is None:
            return self.Parameters.version
        elif self.Parameters.version == DEFAULT_DOCKER_VERSION:
            return hash_commit
        else:
            return self.Parameters.version

    def export_src(self, export_path, arc_mount_path):
        prepare_dir(export_path)
        prepare_dir(arc_mount_path)

        arc = Arc(path=arc_mount_path, token=self.Parameters.arc_token.value())
        try:
            arc.export(export_path, self.Parameters.arc_branch, self.Parameters.arc_project_path)
            arc.checkout(self.Parameters.arc_branch)
            hash_commit = str(arc.get_head_commit_hash()).strip()
            return os.path.join(export_path, str(self.Parameters.arc_project_path)), hash_commit
        finally:
            arc.finish()

    def prepare_content(self, src_path):
        return_code = self.exec_command(command=['make', 'prepare_content'],
                                        logname='prepare_content',
                                        cwd=src_path)
        if return_code > 0:
            raise Exception('Failed to run make tar_prepare_content')

    def set_output_parameters(self, built_new_image, version):
        self.Parameters.built_new_image = built_new_image
        self.Parameters.backend_image_name = get_image_tag(version)
