# coding: utf-8

import os
import re
import time
import boto3
import logging
from datetime import datetime

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import yav
from sandbox.projects.partner.utils.arc import Arc
from sandbox.projects.common import task_env
import sandbox.common.types.client as ctc

from sandbox.projects.partner.settings import \
    ROBOT_PEREIRO_STARTREK_TOKEN_SECRET, \
    ROBOT_PARTNER_SECRET, \
    FRONTEND_DOCKER_IMAGE_NAME, \
    DOCKER_REGISTRY, \
    AWS_ENDPOINT_URL, \
    AWS_PARTNER_BUCKET, \
    ROBOT_BLACK_BEARD_STARTREK_TOKEN_SECRET, \
    ROBOT_PEREIRO_AWS_SECRET, \
    ROBOT_PEREIRO_DOCKER_SECRET, \
    BUILD_TYPE_BETA, \
    BUILD_TYPE_RELEASE

from sandbox.projects.partner.tasks.misc.partner_front_task_base import \
    NotificationLevels, \
    PartnerFrontTaskBase

from sandbox.projects.partner.tasks.misc import \
    call_cmd, \
    is_valid_startrack_issue, \
    make_script_call_cmd

from sandbox.projects.partner.tasks.misc.arc import \
    validate_branch

from sandbox.projects.partner.tasks.misc.docker_registry import DockerRegistry

from sandbox.projects.partner.tasks.misc.frontend import \
    get_version_from_docker_tag, \
    BUILD_VERSION_RE


def make_build_beta_cmd(prefix):
    return make_script_call_cmd(
        'buildYharnam/index.ts',
        'beta',
        '--prefix', prefix,
        '--short'
    )


def make_build_release_cmd(ticket, version):
    return make_script_call_cmd(
        'buildYharnam/index.ts',
        'release',
        '--ticket', ticket,
        '--release_version', version,
        '--short'
    )


BUILD_START_MESSAGE = """Сборка фронтенда началась (тикет {ticket_url}).
Таска: {task_url}
"""

SUCCESS_END_MESSAGE = """Собрал образ yharnam и запушил его в registry.
{image_name}
"""


class BuildFrontendNew(PartnerFrontTaskBase):
    """ Build PI frontend app docker image """

    name = 'PARTNER_BUILD_FRONTEND_NEW'

    class Requirements(task_env.BuildRequirements):
        container_resource = 3192257161
        dns = ctm.DnsType.DNS64
        ram = 32 * 1024
        client_tags = ctc.Tag.SSD
        environments = (
            PipEnvironment('semver'),
        )

    class Parameters(PartnerFrontTaskBase.Parameters):
        description = 'Собрать докер образ из trunk или пользовательствой ветки'

        arc_ref = sdk2.parameters.String(
            'Ветка с исходным кодом',
            default='trunk',
            description='Аркадийная ветка, откуда будет собираться фронт ПИ.\n'
                        'В формате user/<username/<branch> или releases/partner/<branch>\n'
                        'Например, users/deivanov/PI-28263_update-scripts',
            required=True
        )

        is_debug = sdk2.parameters.Bool(
            'Debug',
            description='Перевести таксу в suspend после уснановки зависимостей',
            default=False
        )

        with sdk2.parameters.String('Тип сборки', required=True) as build_type:
            for type_of_build in [BUILD_TYPE_BETA, BUILD_TYPE_RELEASE]:
                build_type.values[type_of_build] = type_of_build

        with build_type.value[BUILD_TYPE_RELEASE]:
            st_issue = sdk2.parameters.String(
                'Релизный тикет',
                description='PI-NNNNN',
                required=False
            )

            is_hotfix = sdk2.parameters.Bool('Хотфикс', default=False)

        with sdk2.parameters.Group('Secrets'):
            st_token = sdk2.parameters.YavSecret(
                'Startrek OAuth token', default=ROBOT_PEREIRO_STARTREK_TOKEN_SECRET, required=True
            )
            arc_token = sdk2.parameters.YavSecret(
                'Arc OAuth token', description='Use arc to create release branch', default=ROBOT_PARTNER_SECRET
            )
            docker_auth_token = sdk2.parameters.YavSecret(
                'Docker auth token',
                default='sec-01e0wr3ferw4jph544mvbqscz0',
                required=True
            )

        with sdk2.parameters.Output:
            image_name = sdk2.parameters.String('Docker image name')
            version = sdk2.parameters.String('Version')

    def get_arc_token(self):
        logging.debug('getting arc token')
        if not self.Parameters.arc_token:
            raise Exception('Arc access token is missed')

        token = self.Parameters.arc_token.data()['arc_token']
        logging.debug('success getting arc token')
        return token

    def on_prepare(self):
        logging.debug('on_prepare enter')

        super(BuildFrontendNew, self).on_prepare()
        secret = sdk2.yav.Secret(ROBOT_BLACK_BEARD_STARTREK_TOKEN_SECRET)
        data = secret.data()['key']
        os.environ['ST_TOKEN'] = data

        logging.debug('on_prepare leave')

    def on_execute(self):
        super(BuildFrontendNew, self).on_execute()
        self.validate_input_params()
        branch_name = self.Parameters.arc_ref
        self.arc = Arc(path='.', token=self.get_arc_token())
        self.do_checkout(branch_name)
        os.chdir('{}/adv/frontend/services/yharnam'.format(self.arc.path))
        logging.info('Current path is {}'.format(os.getcwd()))

        with sdk2.helpers.ProcessLog(self, logger='execution') as pl:
            for i in range(10):
                try:
                    call_cmd([
                        'docker', 'login',
                        '-urobot-pereiro', '-p%s' % self.Parameters.docker_auth_token.data()['key'],
                        'registry.yandex.net',
                    ], pl)
                    break
                except Exception as ex:
                    if i == 9:
                        raise ex
                    time.sleep(3)

            logging.info('Install deps {}')
            self.install_deps(pl)
            logging.debug('deps installed')

            if self.Parameters.is_debug:
                self.suspend()

            if self.Parameters.build_type == BUILD_TYPE_RELEASE:
                self.build_release(branch_name, pl)
            elif self.Parameters.build_type == BUILD_TYPE_BETA:
                self.build_beta(branch_name, pl)
        self.arc.finish()

    def do_checkout(self, branch_name):
        if branch_name == 'trunk':
            logging.info('No need to checkout - you are on trunk already')
            return
        self.arc.checkout(branch_name)

    def build_beta(self, branch_name, pl):
        short_branch_name = branch_name.split('/')[-1]
        logging.info('Createing beta with {} name'.format(short_branch_name))
        cmd = make_build_beta_cmd(short_branch_name)
        image_name = call_cmd(cmd, pl)
        if not image_name:
            raise Exception('Build failed')
        self.Parameters.image_name = image_name

    def send_build_start_notification(self, pl):
        ticket_url = 'https://st.yandex-team.ru/{ticket}'.format(
            ticket=self.Parameters.st_issue
        )
        msg = BUILD_START_MESSAGE.format(
            ticket_url=ticket_url,
            task_url=self.task_url
        )
        self.send_message(msg)

    def send_build_success_notification(self, image_name, pl):
        """
        Пишем сообщение о завершении сборки
        Сообщение тригерное -> запускает процесс выкладки в препрод/ТС
        """

        success_end_message = SUCCESS_END_MESSAGE.format(
            image_name=image_name
        )
        self.send_message(success_end_message)

    def build_release(self, branch_name, pl):
        try:
            self.send_build_start_notification(pl)
            if self.use_prebuilt_rc():
                logging.debug('build_release using ready build, leave')
                return
            version = self.get_version(branch_name)
            self.Parameters.version = version
            self.try_download_bundles(pl)
            image_name = self.run_build(self.Parameters.version, pl)

            if not image_name:
                self.send_message(self.failure_message, NotificationLevels.INFO)
                raise Exception('Build failed')

            self.Parameters.image_name = image_name
            self.send_build_success_notification(image_name, pl)

        except Exception as err:
            self.send_message(self.failure_message, NotificationLevels.INFO)
            raise err

    def run_build(self, version, pl):
        """Запускаем саму сборку, возвращая имя докерного образа"""

        ticket = self.Parameters.st_issue

        release_cmd = make_build_release_cmd(
            ticket=ticket,
            version=version
        )

        return call_cmd(release_cmd, pl)

    def use_prebuilt_rc(self, pl=None):
        """
        Найти и использовать готовую сборку текущей версии кода, включая RC-ветку сборки
        При успехе будут проставлены выходные параметры `image_name`, `rc_branch` и `version`.
        image_name - используется непосредственно при сборке/деплое,
        rc_branch - для запуска автотестов и скриншутера, и при финализации,
        version - в настоящее время чисто информационное поле.
        Возвращает статус операции
        """
        logging.debug('use_prebuilt_rc enter')

        current_sha = self.arc.get_head_commit_hash()
        docker_tag = self.find_docker_tag_by_commit(current_sha)

        if not docker_tag:
            logging.debug('use_prebuilt_rc no images found, leave')
            return False

        logging.debug('use_prebuilt_rc found prebuilt image %s' % docker_tag)
        self.Parameters.image_name = self.format_docker_image_name(docker_tag)
        self.send_message('Шаги сборки пропущены. Будет использован docker-образ с тегом: {}'.format(docker_tag))

        version = get_version_from_docker_tag(docker_tag)
        self.Parameters.version = version

        logging.debug('use_prebuilt_rc leave')
        return True

    def find_docker_tag_by_commit(self, current_sha):
        """
        Находит подходящий докерный тег для текущей версии кода при пересборке
        """

        tag_re = re.compile('^%s_%s' % (BUILD_VERSION_RE, current_sha))
        def filter_func(tag):
            return bool(re.match(tag_re, tag))

        docker_oauth_token_secret = yav.Secret(ROBOT_PEREIRO_DOCKER_SECRET)
        docker_oauth_token = docker_oauth_token_secret.data()['key']
        registry = DockerRegistry(oauth_token=docker_oauth_token)
        tags = registry.tag_list(FRONTEND_DOCKER_IMAGE_NAME)
        filtered_tags = filter(filter_func, tags)
        if len(filtered_tags) != 1:
            # Больше одного образа из одного коммита (с разными версиями)
            return None
        appropriate_tag = filtered_tags[0]
        image_version = get_version_from_docker_tag(appropriate_tag)
        logging.debug('Docker image build version: {} and commit SHA: {}'.format(image_version, current_sha))

        if not self.check_cdn_static(image_version):
            return None

        return appropriate_tag

    def check_cdn_static(self, version):
        """
        Проверить статику в CDN
        """
        # TODO дополнительная валидация или изолирование разных пересборок одной и той же версии
        aws_secret = yav.Secret(ROBOT_PEREIRO_AWS_SECRET)
        key_id = aws_secret.data()['id']
        access_key = aws_secret.data()['secret']
        session = boto3.session.Session(aws_access_key_id=key_id, aws_secret_access_key=access_key)
        client = session.client(service_name='s3', endpoint_url=AWS_ENDPOINT_URL, verify=False)
        files = client.list_objects(Bucket=AWS_PARTNER_BUCKET, Prefix=version, MaxKeys=1)
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/s3.html?highlight=list bucket#S3.Client.list_parts
        # Если не было найдено файлов с подходящим префиксом, то поле "Contents" будет отсутствовать
        return 'Contents' in files

    @property
    def failure_message(self):
        return 'Ошибка при сборке yharnam!\nTask: {}'.format(self.task_url)

    def validate_input_params(self):
        branch = self.Parameters.arc_ref

        if self.Parameters.build_type == BUILD_TYPE_RELEASE:
            st_issue = self.Parameters.st_issue
            if not is_valid_startrack_issue(st_issue):
                raise Exception('Ticket "{}" is not valid'.format(st_issue))

            callback_url = self.Parameters.callback_url
            if not callback_url:
                raise Exception('Callback URL is required for release builds')

        if not validate_branch(branch):
            raise Exception('Invalide branch name')

    def get_version(self, rc_branch):
        type = self.release_type
        yymmdd = datetime.now().strftime('%y%m%d')
        hhmmss = rc_branch.split('-')[-1]
        if not hhmmss:
            hhmmss = datetime.now().strftime('%H%M%S')
        version = '{type}-{yymmdd}-{hhmmss}'.format(
            type=type,
            yymmdd=yymmdd,
            hhmmss=hhmmss
        )
        return version

    @property
    def release_type(self):
        return 'hotfix' if self.Parameters.is_hotfix else 'release'

    @staticmethod
    def format_docker_image_name(tag):
        return '{}/{}:{}'.format(DOCKER_REGISTRY, FRONTEND_DOCKER_IMAGE_NAME, tag)
