# coding: utf-8

import logging

import sandbox.common.types.task as ctt
import sandbox.sdk2 as sdk2
from sandbox.projects.adfox.adfox_ui.metrics import AnalyzableTask
from sandbox.projects.common.build.YaPackage2 import YaPackage2
from sandbox.projects.partner.utils.arc import Arc
from sandbox.projects.partner.settings import ROBOT_PARTNER_SECRET

JSONAPI_APP_NAME = "JSONAPI"
INTAPI_APP_NAME = "INTAPI"
TESTAPI_APP_NAME = "TESTAPI"
HOURGLASS_APP_NAME = "HOURGLASS"
PACKAGE_DICT = {
    INTAPI_APP_NAME: "partner/java/apps/intapi/package-intapi.json",
    JSONAPI_APP_NAME: "partner/java/apps/jsonapi/package-jsonapi.json",
    TESTAPI_APP_NAME: "partner/java/apps/testapi/package-testapi.json",
    HOURGLASS_APP_NAME: "partner/java/apps/hourglass/package-hourglass.json",
}


def _get_last_commit_sha(branch, arc_token):
    arc = Arc(path='.', token=arc_token)
    try:
        arc.checkout(branch=branch)
        return arc.get_head_commit_hash()
    finally:
        arc.finish()


class PartnerBuildJavaDocker(AnalyzableTask):
    """
    Собирает приложения java ПИ в Docker и возвращает ссылки на них
    """

    class Parameters(AnalyzableTask.Parameters):
        arc_release_branch = sdk2.parameters.String(
            "Arc release branch",
            description="Branch ID in arc, ex. releases/partner/PI-00000-release-1",
            default_value="trunk",
            required=True
        )
        arc_token = sdk2.parameters.YavSecret(
            'Arc OAuth token',
            description='Use arc to calc sha',
            default="{}#ya_token".format(ROBOT_PARTNER_SECRET),
            required=True
        )

        build_apps = sdk2.parameters.CheckGroup(
            'Build apps',
            choices=[(_, _) for _ in PACKAGE_DICT],
            default=[_ for _ in PACKAGE_DICT],
            required=True,
        )

        with sdk2.parameters.Output:
            docker_images = sdk2.parameters.String("Links to docker images")

    def _build_task(self, arcadia_url, package_name, package_path):
        ya_package_task = YaPackage2(
            self,
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            description=package_path,
            checkout_arcadia_from_url=arcadia_url,
            package_type="docker",
            docker_image_repository="partners/java",
            docker_push_image=True,
            packages=package_path,
            docker_user="robot-partner",
            docker_token_vault_name="robot-partner-docker-registry-oauth",
            tags=[package_name, "JAVA", "PARTNER"],
            ya_yt_token_yav_secret='{}#yt_token'.format(ROBOT_PARTNER_SECRET),
        )
        return ya_package_task

    def _wait_tasks(self, tasks):
        raise sdk2.WaitTask(
            tasks,
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
        )

    def on_execute(self):
        super(PartnerBuildJavaDocker, self).on_execute()
        with self.memoize_stage.wait_tasks:
            logging.info('Run sub tasks')
            sha = _get_last_commit_sha(
                branch=self.Parameters.arc_release_branch,
                arc_token=self.Parameters.arc_token.value()
            )
            arcadia_url = "arcadia-arc:/#{}".format(sha)
            self.Context.task_dict = {}
            tasks = []
            for app in self.Parameters.build_apps:
                task = self._build_task(arcadia_url, app, PACKAGE_DICT[app])
                task.enqueue()
                tasks.append(task)
                self.Context.task_dict[app] = task.id

            logging.info('Wait sub tasks')
            self._wait_tasks(tasks)

        logging.info('Get results from tasks: {}'.format(self.Context.task_dict))
        docker_arr = []
        for app, task_id in self.Context.task_dict.items():
            task = sdk2.Task.find(id=task_id).limit(1).first()
            if task.status == ctt.Status.SUCCESS:
                docker_arr.append(task.Context.output_resource_version)
            else:
                raise Exception('Что-то пошло не так с задачей сборки %s' % task_id)

        self.Parameters.docker_images = ';'.join(docker_arr)

    def get_metrics_release_id(self):
        return self.Parameters.arc_release_branch
