# -*- coding: utf-8 -*

import logging
import subprocess
import os

import sandbox.sdk2.helpers
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import DnsType
from sandbox.projects.partner.settings import \
    ROBOT_PARTNER_SECRET, \
    PI_BACKEND_LXC_IMAGE
from sandbox.projects.partner.tasks.build_docker import BuildDockerImage
import sandbox.common.types.task as ctt
from sandbox.projects.partner.utils.perl_release_script import PerlReleaseScript
from sandbox.projects.adfox.adfox_ui.metrics import AnalyzableTask

DEFAULT_ENV_TYPE = "development"
DEFAULT_ARC_SVN_BRANCH = "trunk/arcadia/partner"

ROOT_PATH = "/opt/partner2_release_robot/"
WORKING_PATH = ROOT_PATH + "partner2/"
RELEASE_SCRIPT_PATH = ROOT_PATH + "bin/script"

ANSIBLE_REPO_PATH = "/ansible"
ANSIBLE_RELEASE_SCRIPT_PATH = "roles/partner2_release_robot/files/script"


class PartnerBuildPerl(AnalyzableTask):
    """
    Собирает perl часть бэкенда.
    """

    arc_binary = None
    arc_path = None
    yav_token = None

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64
        container_resource = PI_BACKEND_LXC_IMAGE  # PRECISE
        privileged = True

    class Parameters(AnalyzableTask.Parameters):
        description = "Build and release PI"
        push_tasks_resource = True

        with sdk2.parameters.Group("General backend build parameters") as general_block:
            release_ticket = sdk2.parameters.String(
                "Release ticket (PI-xxxxx)", required=True)
            branch = sdk2.parameters.String(
                "Backend branch", default="trunk", required=True)
            release_st_ticket_on_preprod = sdk2.parameters.String(
                "Preprod hotfix release ticket (PI-xxxxx)")
            skip_not_merged = sdk2.parameters.Bool(
                "Skip not merged Pull Requests", default=False)
            arc_token = sdk2.parameters.YavSecret(
                'Arc OAuth token', description='Use arc to create release branch', default=ROBOT_PARTNER_SECRET
            )

        do_not_notify_via_infra = sdk2.parameters.Bool(
            'DO NOT create infra notifications',
            description='Switch off infra notifications on build and deploy',
            default=False
        )

        with sdk2.parameters.Output:
            build_version = sdk2.parameters.String(
                "Version of the packages built.")
            release_branch = sdk2.parameters.String(
                "Name of the backend release branch")
            docker_image = sdk2.parameters.String("Docker image name")

    def exec_command(self, command, logname='exec_log', cwd=None):
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logname) as process_log:
            return sandbox.sdk2.helpers.subprocess.Popen(
                ('bash', '-c', command),
                stdout=process_log.stdout,
                stderr=process_log.stdout,
                cwd=cwd
            ).wait()

    def run_release_script(self, logname):
        logging.info("=== Run release script ===")
        release_script = PerlReleaseScript(
            branch=self.Parameters.branch, executor=self.exec_command)

        arguments = '--st_ticket={ticket} --action=build'.format(
            ticket=self.Parameters.release_ticket,
        )

        arguments += ' --checkout_to={branch}'.format(
            branch=self.Parameters.branch)
        if self.Parameters.release_st_ticket_on_preprod:
            arguments += ' --release_st_ticket_on_preprod={}'.format(
                self.Parameters.release_st_ticket_on_preprod)

        if self.Parameters.skip_not_merged:
            arguments += ' --skip_not_merged'

        logging.info(arguments)

        status = release_script.run(
            arguments=arguments,
            logname=logname
        )
        return status

    def on_execute(self):
        with self.memoize_stage.first_step:
            logname = '/var/log/release_script'
            os.environ["ARC_TOKEN"] = self.get_arc_token()
            status = self.run_release_script(logname=logname)
            self.exec_command('cat %s.out.log' %
                              logname, logname='release_script')

            if status != 0:
                raise TaskFailure('Failed to build backend perl')

            version = subprocess.check_output(
                r'grep -Pom 1 "BUILD_NUMBER_PERL:\K\S+" %s.out.log' % logname, shell=True)
            self.Parameters.build_version = version.strip()
            branch = subprocess.check_output(
                r'grep -Pom 1 "arc checkout -b \K\S+" %s.out.log' % logname, shell=True)
            self.Parameters.release_branch = branch.strip()

            task = BuildDockerImage(
                self,
                repo="arc",
                version=self.Parameters.build_version,
                arc_branch=self.Parameters.release_branch,
            )
            task.enqueue()
            self.Context.subtask_id = task.id
            raise sdk2.WaitTask(
                [task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        with self.memoize_stage.second_step:
            task = BuildDockerImage.find(id=self.Context.subtask_id).first()
            logging.info("Status task {}")
            logging.info(task.status)
            if task.status not in ctt.Status.Group.SUCCEED:
                raise Exception("Package build failed, see logs for details")
            self.Parameters.docker_image = task.Parameters.backend_image_name

    def get_arc_token(self):
        logging.debug('getting arc token')
        if not self.Parameters.arc_token:
            raise Exception('Arc access token is missed')

        token = self.Parameters.arc_token.data()['arc_token']
        logging.debug('success getting arc token')
        return token
