# -*- coding: utf-8 -*

import logging
import os

import sandbox.sdk2.helpers as helpers
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
import sandbox.common.types.task as ctt
from sandbox.common.types.misc import DnsType
from sandbox.projects.partner.settings import \
    PI_BACKEND_LXC_IMAGE, \
    ROBOT_PARTNER_SECRET, \
    ROBOT_PEREIRO_INFRA_TOKEN
from sandbox.sdk2 import yav
from sandbox.projects.partner.tasks.misc.partner_front_task_base import \
    PartnerFrontTaskBase
from sandbox.projects.partner.tasks.deploy_stage import PartnerDeployStage
from sandbox.projects.partner.utils.perl_release_script import PerlReleaseScript
from sandbox.projects.partner.settings import TEST_STAGE, PREPROD_STAGE, PRODUCTION_STAGE, SCREENSHOOTER_BETA
from sandbox.projects.partner.tasks.misc.infra_notifications import Services, Environments, InfraNotifications


from sandbox.projects.partner.settings import \
    DOCKER_REGISTRY, \
    BACKEND_DOCKER_IMAGE_NAME, \
    FRONTEND_DOCKER_IMAGE_NAME, \
    GENERAL_DB_DOCKER_IMAGE_NAME, \
    AUTOTEST_DB_DOCKER_IMAGE_NAME

DEPLOY_PRODUCTION = "deploy-production|%s"


class Tags:
    def __init__(self):
        pass

    PROD = 'PROD'
    PREPROD = 'PREPROD'
    TEST = 'TEST'
    SCREEN_SHOOTER = 'SCREEN_SHOOTER'


class DeployRelease(PartnerFrontTaskBase):
    """
    Выкладывает общий релиз ПИ в Деплой.
    """
    name="PARTNER_DEPLOY_RELEASE"
    arc_binary = None
    arc_path = None
    infra_center = None
    _release_script = None

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64
        container_resource = PI_BACKEND_LXC_IMAGE  # PRECISE
        privileged = True

    class Context(sdk2.Task.Context):
        infra_queue = None

    class Parameters(PartnerFrontTaskBase.Parameters):
        description = "Build and release PI"

        st_issue = sdk2.parameters.String(
            'Release ticket',
            description='PI-NNNNN',
            required=False
        )

        with sdk2.parameters.Group("General build parameters") as general_block:
            stage = sdk2.parameters.String(
                "Stage",
                choices=[(s, s) for s in (TEST_STAGE, PREPROD_STAGE, PRODUCTION_STAGE, SCREENSHOOTER_BETA)],
                default=PREPROD_STAGE
            )
            java_docker_images = sdk2.parameters.String(
                'Java docker images',
                description='String with java docker images, separated by ";"'
            )

            perl_tag = sdk2.parameters.String(
                "Perl package tag",
                description="(full version like a 2.18.2100)",
                required=True
            )

            frontend_tag = sdk2.parameters.String(
                "Frontend tag",
                required=True,
                description="For ex.: 0.91.0_a129cb95249184b57ba2e36b18cfe2325734c316",
            )

            with stage.value[PRODUCTION_STAGE]:
                check_before_migrations = sdk2.parameters.Bool("Check before migrations", default=True)
                do_not_notify_via_infra = sdk2.parameters.Bool(
                    'DO NOT create infra notifications',
                    description='Switch off infra notifications on build and deploy',
                    default=False
                )

            release_script_branch = sdk2.parameters.String("Release script branch", default='trunk')

            arc_token = sdk2.parameters.YavSecret(
                'Arc OAuth token', description='Use arc to create release branch', default=ROBOT_PARTNER_SECRET
            )

    def exec_command(self, command, logname='exec_log', cwd=None):
        with helpers.ProcessLog(self, logger=logname) as process_log:
            return helpers.subprocess.Popen(
                ('bash', '-c', command),
                stdout=process_log.stdout,
                stderr=process_log.stdout,
                cwd=cwd
            ).wait()

    def get_build_number(self):
        return self.Parameters.perl_tag.split('.')[-1]

    def init_infra_notifications_center(self):
        logging.info("=== Init infra notifications token ===")
        secret = yav.Secret(ROBOT_PEREIRO_INFRA_TOKEN)
        token = secret.data()['token']
        self.infra_center = InfraNotifications(
            oauth_token=token,
            ticket=self.Parameters.st_issue,
            queue=self.Context.infra_queue.split(',') if self.Context.infra_queue is not None else None
        )

    def init_release_script(self):
        self._release_script = \
            PerlReleaseScript(branch=self.Parameters.release_script_branch, executor=self.exec_command)

    def check_before_migrations(self, logname='/var/log/release_script'):
        logging.info("=== Check before migrations ===")

        arguments = '--st_ticket={ticket} --action=check-before-migrations'.format(
            ticket=self.Parameters.st_issue,
        )

        arguments += " --build_number='%s' " % self.get_build_number()

        status = self._release_script.run(
            arguments=arguments,
            logname=logname
        )
        if status != 0:
            raise TaskFailure('Failed to check before migrations')

    def deploy_release(self, type=None):
        if type is None:
            raise Exception("Release type can`t be empty")

        perl_image = '%s/%s:%s' % (DOCKER_REGISTRY, BACKEND_DOCKER_IMAGE_NAME, self.Parameters.perl_tag)
        frontend_image = '%s/%s:%s' % (DOCKER_REGISTRY, FRONTEND_DOCKER_IMAGE_NAME, self.Parameters.frontend_tag)
        mysql_image = '%s/%s' % (DOCKER_REGISTRY, GENERAL_DB_DOCKER_IMAGE_NAME)
        if type == 'testing':
            mysql_image += ';%s/%s' % (DOCKER_REGISTRY, AUTOTEST_DB_DOCKER_IMAGE_NAME)

        task = PartnerDeployStage(
            self,
            st_issue=self.Parameters.st_issue,
            images='%s;%s;%s;%s' % (frontend_image, perl_image, self.Parameters.java_docker_images, mysql_image),
            ticket_type=type
        )

        task.enqueue()

        return task

    def finalize_backend(self, logname='/var/log/release_script'):
        logging.info("=== Finalize release ===")

        arguments = '--st_ticket={ticket} --action=finalize-release'.format(
            ticket=self.Parameters.st_issue,
        )

        arguments += " --build_number='%s' " % self.get_build_number()

        status = self._release_script.run(
            arguments=arguments,
            logname=logname
        )
        if status != 0:
            raise TaskFailure('Failed to finalize release')

    def on_save(self):
        new_tags = [x for x in self.Parameters.tags if x not in [k for (k, v) in vars(Tags).items()]]

        if self.Parameters.stage == PRODUCTION_STAGE:
            new_tags.append(Tags.PROD)
        elif self.Parameters.stage == SCREENSHOOTER_BETA:
            new_tags.append(Tags.SCREEN_SHOOTER)
        elif self.Parameters.stage == PREPROD_STAGE:
            new_tags.append(Tags.PREPROD)
        else:
            new_tags.append(Tags.TEST)

        self.Parameters.tags = new_tags

    def on_execute(self):
        os.environ["ARC_TOKEN"] = self.get_arc_token()
        self.init_release_script()
        self.init_infra_notifications_center()
        with self.memoize_stage.first:
            tasks = []
            tasks_ids = []
            if self.Parameters.stage == PRODUCTION_STAGE:
                if self.Parameters.check_before_migrations:
                    self.check_before_migrations()
                if not self.Parameters.do_not_notify_via_infra:
                    self.infra_center.push_release_event(Services.partner_backend, Environments.production)
                    self.infra_center.push_release_event(Services.partner_frontend, Environments.production)

                self.track_start(DEPLOY_PRODUCTION % self.Parameters.st_issue)
                task = self.deploy_release(type='stable')
                tasks.append(task)  # тип соответствует типу релиза в deploy
                tasks_ids.append(task.id)
            elif self.Parameters.stage == SCREENSHOOTER_BETA:
                # deploy screenshooter stages
                task = self.deploy_release(type='screenshooter')
                tasks.append(task)
                tasks_ids.append(task.id)
            elif self.Parameters.stage == PREPROD_STAGE:
                # deploy preprod stages
                task = self.deploy_release(type='prestable')
                tasks.append(task)
                tasks_ids.append(task.id)
            elif self.Parameters.stage == TEST_STAGE:
                # deploy TS stages
                task = self.deploy_release(type='testing')
                tasks.append(task)
                tasks_ids.append(task.id)

            self.Context.subtasks = tasks_ids

            if len(self.infra_center.queue) > 0:
                self.Context.infra_queue = ','.join(str(e) for e in self.infra_center.queue)

            if len(tasks_ids) == 0:
                raise Exception('No subtasks create')

            raise sdk2.WaitTask(tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.last:
            self.infra_center.resolve_all_created_events()
            for task_id in self.Context.subtasks:
                task = self.find(id=task_id).limit(1).first()
                if task.status not in ctt.Status.Group.SUCCEED:
                    raise Exception("Sub task failed. See task id #%s." % task_id)

            if self.Parameters.stage == PRODUCTION_STAGE:
                self.track_finish(DEPLOY_PRODUCTION % self.Parameters.st_issue)
                self.finalize_backend()

    def get_arc_token(self):
        logging.debug('getting arc token')
        if not self.Parameters.arc_token:
            raise Exception('Arc access token is missed')

        token = self.Parameters.arc_token.data()['arc_token']
        logging.debug('success getting arc token')
        return token
