# coding: utf-8
# Задача создаёт релизный тикет с переданными docker контейнерами и пишет об этом в крашер
import requests
import logging
import sandbox.sdk2 as sdk2
import datetime
from sandbox.projects.partner.tasks.misc.partner_front_task_base import \
    Callbackable, \
    NotificationLevels
from sandbox.projects.partner.settings import PI_STAGES_BY_RELEASE_TYPE, PI_RELEASE_TYPES
from sandbox.projects.partner.tasks.misc.docker_registry import DockerRegistry
from sandbox.projects.deploy.CreateDeployDockerRelease import CreateDeployDockerRelease
from sandbox.projects.common.ya_deploy import release_integration
import sandbox.common.types.task as ctt
from sandbox.projects.adfox.adfox_ui.metrics import AnalyzableTask


WAIT_MAX_TIME = 15 * 60
MAX_WAIT_RELEASE_TASK_CALL_NUMBER = 6  # максимальное количество итераций запуска ожидания после создания тикета
RELEASE_TASK_TAKES_TOO_LONG_MESSAGE = 'Деплой идёт слишком долго'
SUCCESS_MESSAGE = 'Создал %s релизный тикет'
WAIT_STATUS = 'Ожидаю применения %s релизного тикета'
ALIVE_STATUS = {
    'backend-fail': 'ALERT! Проверка бэкенда на %s завершилась с ошибкой.',
    'frontend-fail': 'ALERT! Проверка фронтенда на %s завершилась с ошибкой.',
    'all-good': 'Всё впорядке на %s. Стейдж успешно обновлён и базово работает',
}
STAGE_URLS = {
    'stable': 'https://partner2.yandex.ru',
    'testing': 'https://partner2-test.yandex.ru',
    'screenshooter': 'https://screen-test.beta.partner.yandex.ru',
}
BACKEND_ALIVE_PATH = '/devel/alive'
FRONTEND_ALIVE_PATH = '/v2/simple_alive'


class PartnerDeployStageParametersMeta(type(AnalyzableTask.Parameters), type(Callbackable.Parameters)):
    pass


class PartnerDeployStage(AnalyzableTask, Callbackable):

    class Parameters(AnalyzableTask.Parameters, Callbackable.Parameters):
        __metaclass__ = PartnerDeployStageParametersMeta

        description = 'Create release ticket for deploy'

        st_issue = sdk2.parameters.String(
            'Release ticket',
            description='PI-NNNNN',
            required=False
        )
        images = sdk2.parameters.String(
            'Docker images',
            description='Docker images to release ticket separated by ;',
            required=True
        )
        #
        ticket_type = sdk2.parameters.String(
            'Release type',
            description='Release type in Ya.Deploy',
            required=True,
        )
        release_types = release_integration.RELEASE_TYPES + PI_RELEASE_TYPES
        ticket_type.choices = release_types

        need_wait_stage_redeploy = sdk2.parameters.Bool(
            'Wait stage redeploy',
            description='Check if need to wait stage redeploy',
            default=True,
        )

        with need_wait_stage_redeploy.value[True]:
            wait_stage_list = sdk2.parameters.List("List of stages to wait")

        send_notifications = sdk2.parameters.Bool(
            'Send notification',
            description='Need to send notification to all chanels',
            default=True,
        )
        oauth_secret = sdk2.parameters.YavSecret(
            'Secret for OAuth',
            default='sec-01ctrdr10j7ejcnxak8wb3k5d1',
            required=True,
        )
        lock_list = sdk2.parameters.List("List of locks to wait", required=False)

    def check_service_working(self, stage):
        url = STAGE_URLS.get(stage)
        if url is None:
            return True

        success = True
        if requests.get(url + BACKEND_ALIVE_PATH).status_code != 200:
            self.send_message(ALIVE_STATUS.get('backend-fail') % url, NotificationLevels.URGENT)
            success = False

        if requests.get(url + FRONTEND_ALIVE_PATH).status_code != 200:
            self.send_message(ALIVE_STATUS.get('frontend-fail') % url, NotificationLevels.URGENT)
            success = False

        if success:
            self.send_message(ALIVE_STATUS.get('all-good') % url)
        else:
            raise Exception

    def on_enqueue(self):
        super(PartnerDeployStage, self).on_enqueue()
        if self.Parameters.lock_list:
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=self.Parameters.lock_list,
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
            )

    def get_arc_token(self):
        return self.Parameters.oauth_secret.data()['arc_token']

    def get_dctl_token(self):
        return self.Parameters.oauth_secret.data()['dctl_yp_token']

    def is_task_successfully_accomplished(self, task_id):
        return self.find(id=task_id, status=ctt.Status.SUCCESS).limit(1).first() is not None

    def wait_release_task(self):
        raise sdk2.WaitTask(
            [self.Context.release_task],
            ctt.Status.Group.FINISH,
            timeout=WAIT_MAX_TIME,
            wait_all=False
        )

    def on_execute(self):
        super(PartnerDeployStage, self).on_execute()
        logging.debug('Start task')
        if not self.Context.release_task:
            logging.debug('Create release task')
            images = self.Parameters.images.split(';')
            docker_registry = DockerRegistry()
            new_images = []
            for image in images:
                image_splitted = image.split(':')
                if len(image_splitted) < 2:
                    tag_list = docker_registry.tag_list(image, r'^\d\.\d{2}\.\d{4}-\d{4}-\d{2}-\d{2}$')
                    if len(tag_list) == 0:
                        raise Exception('Cannot find tags for %s' % image)
                    last_tag = tag_list[-1]
                    self.notify_if_db_image_is_old(image, last_tag)
                    image = '%s:%s' % (image, last_tag)
                new_images.append(image)

            images = new_images

            release_task = CreateDeployDockerRelease(
                self,
                description="Deploy task for PI. Child of task {}".format(self.id),
                create_sub_task=False
            )
            release_task.Parameters.yp_token_yav_secret = self.Parameters.oauth_secret
            release_task.Parameters.yp_token_yav_secret_key = 'dctl_yp_token'
            release_task.Parameters.release_type = self.Parameters.ticket_type
            release_task.Parameters.docker_images = images

            if self.Parameters.need_wait_stage_redeploy:
                logging.debug('Wait stages deploy')

                stage_list = self.Parameters.wait_stage_list \
                    if len(self.Parameters.wait_stage_list) > 0 \
                    else PI_STAGES_BY_RELEASE_TYPE[self.Parameters.ticket_type]
                if len(stage_list) > 0:
                    self.send_message(WAIT_STATUS % self.Parameters.ticket_type)
                    release_task.Parameters.wait_deploy_tickets = True
                    release_task.Parameters.wait_stages = stage_list

            self.Context.release_task = release_task.save().enqueue().id
            logging.debug('Release task created #%s' % self.Context.release_task)
            self.Context.wait_release_task_call_counter = 1
            self.wait_release_task()

        is_accomplished = self.is_task_successfully_accomplished(self.Context.release_task)

        if is_accomplished is False:
            if self.Context.wait_release_task_call_counter < MAX_WAIT_RELEASE_TASK_CALL_NUMBER:
                self.send_message(RELEASE_TASK_TAKES_TOO_LONG_MESSAGE)
                self.Context.wait_release_task_call_counter += 1
                self.wait_release_task()
            else:
                raise Exception("Sub task failed. See task id #%s." % self.Context.release_task)

        if self.Parameters.send_notifications:
            self.send_message(SUCCESS_MESSAGE % self.Parameters.ticket_type)

        self.check_service_working(self.Parameters.ticket_type)

    def notify_if_db_image_is_old(self, image, last_db_image_tag):
        image_date = datetime.datetime.strptime(last_db_image_tag[10:], '%Y-%m-%d').date()
        today_date = datetime.date.today()

        delta = (today_date - image_date).days
        if delta > 1:
            self.send_message('Используется старый образ БД: ' + image + ':' + last_db_image_tag)
