# coding: utf-8

import json
import logging
import math
import os
import re
import shutil
import time

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.adfox.adfox_ui.util.config import GitConfig
from sandbox.projects.adfox.adfox_ui.util.duty_tools import dutyTools
from sandbox.projects.adfox.adfox_ui.util.release_integration import AdfoxReleaseIntegration
from sandbox.projects.common import binary_task, network, task_env

import constants as cnt
import sandbox.projects.partner.settings as ps
from sandbox.projects.partner.tasks.build_docker import BuildDockerImage
from sandbox.projects.partner.tasks.build_frontend import BuildFrontend
from sandbox.projects.partner.tasks.build_java_docker import PartnerBuildJavaDocker, JSONAPI_APP_NAME, TESTAPI_APP_NAME
from sandbox.projects.partner.tasks.e2e_tests.misc import HermioneReportLayer, REPORT_PROJECT_ADFOX, \
    REPORT_PROJECT_PARTNER
from sandbox.projects.partner.tasks.misc import exec_cmd
from sandbox.projects.partner.tasks.misc.arc import is_rc_branch
from sandbox.projects.partner.tasks.misc.docker import Docker
from sandbox.projects.partner.tasks.misc.docker_compose import DockerCompose
from sandbox.projects.partner.tasks.misc.docker_registry import DockerRegistry
from sandbox.projects.partner.tasks.misc.krush_logger import KrushLogger
from sandbox.projects.partner.tasks.misc.partner_front_task_base import PartnerFrontTaskBase
from sandbox.projects.partner.tasks.misc.template import ResourceTemplate, StringTemplate
from sandbox.projects.partner.utils.arc import Arc
from sandbox.projects.partner.utils.autotests import send_tests_result_message
from sandbox.projects.partner.utils.dns_api import DnsApi
from sandbox.projects.partner.utils.get_production_tag import get_production_tag, VersionType

PARTNER_YHARNAM_PATH = 'tests/yharnam'
PARTNER_AUTOTESTS_PATH = os.path.join(PARTNER_YHARNAM_PATH, 'test/autotests')
PARTNER_REPORTS_PATH = os.path.join(PARTNER_AUTOTESTS_PATH, 'hermione/autotests-reports')

PARTNER_REPORT_HTML = 'commonReport'
PARTNER_REPORT_JSON = 'jsonReport.json'

ADFOX_REPORT_HTML = 'hermione-report'
ADFOX_REPORT_JSON = 'report.json'

SLEEP_TIME = 60
SLEEP_TIME_MAX = 7200

RESULT_MESSAGE_ENTITY_NAME = 'ПИ'
ADFOX_TESTS_ROOT = '/usr/local/www/login/.tests'


class SourceType(object):
    TAG = 'tag'
    BRANCH = 'branch'
    LATEST = 'latest'
    PRODUCTION = 'production'


class TestsOptions(object):
    DEFAULT = 'default'
    ADFOX = 'adfox'
    PCODE = 'pcode'
    PATTERN = 'pattern'


class TestsVendor(object):
    ADFOX = 'adfox'
    PARTNER = 'partner'


class AdfoxTestType(object):
    ADFOX = 'adfox'
    PARTNER = 'partner'


class DockerHermione(binary_task.LastBinaryTaskRelease, HermioneReportLayer):
    """
    Автотесты ADFOX и PI.
    """
    name = 'PARTNER_DOCKER_HERMIONE'
    _arc_client = None  # кэш Arc-клиент
    _dns_api = None  # кэш Dns Api

    class Requirements(task_env.BuildRequirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 16 * 1024, None)
        container_resource = 3323237980
        dns = ctm.DnsType.DNS64
        cores = 32
        ram = 120 * 1024
        client_tags = ctc.Tag.SSD

    class Context(sdk2.Task.Context):
        partner_frontend_tag = ''
        partner_perl_tag = ''
        partner_java_tag = ''
        partner_mysql_tag = ''
        current_run = 0
        dns_ip = None
        dns_name = None
        frontend_build_task_id = 0
        perl_build_task_id = 0
        java_build_task_id = 0
        adfox_login_tag = None
        report_data = {
            'total': 0,
            'passed': 0,
            'failed': 0,
            'skipped': 0,
            'retries': 0,
        }
        timing = {}
        start_time = 0
        task_status = None
        partner_test_pattern = ''
        partner_test_option = ''

        # ID сборочной линии и ее подзадачи (например для GitLab), передаются в context при создании задачи
        ci_pipeline_id = None
        ci_job_id = None

    class Parameters(PartnerFrontTaskBase.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        description = 'Create docker container for hermione tests'

        tests_vendor = sdk2.parameters.String(
            'Tests vendor',
            description='Vendor of tests',
            required=True,
            default=TestsVendor.PARTNER,
        )

        tests_vendor.choices = [
            ('Partner', TestsVendor.PARTNER),
            ('Adfox', TestsVendor.ADFOX),
        ]

        with sdk2.parameters.Group('Partner') as partner_settings:
            st_issue = sdk2.parameters.String(
                'Release ticket',
                description='PI-NNNNN',
            )

            with sdk2.parameters.Group('Build settings. Perl') as perl_build_settings:
                perl_source_type = sdk2.parameters.String(
                    'Perl backend source',
                    description='Use production version, branch from git repo or custom tag',
                    default=SourceType.PRODUCTION,
                    required=True,
                )

                perl_source_type.choices = [
                    ('Use production version', SourceType.PRODUCTION),
                    ('Use existing image', SourceType.TAG),
                    ('Build new from arcadia branch', SourceType.BRANCH),
                ]

                with perl_source_type.value[SourceType.BRANCH]:
                    perl_branch = sdk2.parameters.String(
                        'Perl backend branch',
                        description='Perl arcadia branch',
                        default='trunk',
                        required=True,
                    )

                with perl_source_type.value[SourceType.TAG]:
                    perl_tag = sdk2.parameters.String(
                        'Perl backend tag',
                        description='Use perl custom image tag (e.g. 2.18.2570)',
                        required=True,
                    )

            with sdk2.parameters.Group('Build settings. Java') as java_build_settings:
                java_source_type = sdk2.parameters.String(
                    'Java source type',
                    default=SourceType.PRODUCTION,
                    description='Source for java. Build new or use prebuilt docker image.',
                    required=True,
                )

                java_source_type.choices = [
                    ('Use production version', SourceType.PRODUCTION),
                    ('Use existing image', SourceType.TAG),
                    ('Build new java image from branch', SourceType.BRANCH),
                ]

                with java_source_type.value[SourceType.BRANCH]:
                    java_branch = sdk2.parameters.String(
                        'Java arcadia branch',
                        description='Url to arc branch ex. releases/partner/PI-0000-release-20',
                        default='trunk',
                        required=True,
                    )

                with java_source_type.value[SourceType.TAG]:
                    java_tag = sdk2.parameters.String(
                        'Java backend tag',
                        description='Use java custom image tag (e.g. 1.5b424545555595668af5f49ec2df4eaa47fc0a1c-1)',
                        required=True,
                    )

            with sdk2.parameters.Group('Build settings. Frontend') as frontend_build_settings:
                frontend_source_type = sdk2.parameters.String(
                    'Frontend source',
                    description='Use production version, arc branch or custom image',
                    default=SourceType.PRODUCTION,
                    required=True,
                )

                frontend_source_type.choices = [
                    ('Use production version', SourceType.PRODUCTION),
                    ('Build new from arc branch', SourceType.BRANCH),
                    ('Use existing image', SourceType.TAG),
                ]

                with frontend_source_type.value[SourceType.TAG]:
                    frontend_tag = sdk2.parameters.String(
                        'Frontend tag',
                        description='Use frontend custom image tag'
                                    '(e.g. 0.222.0_73a567ac7ca012d12dc6ba1d1750d982dfef559d)',
                        required=True,
                    )

                frontend_branch = sdk2.parameters.String(
                    'Frontend test branch',
                    description='Branch, tag or commit in yharnam. \n'
                                'Used as source for building image and for E2E tests to run. \n'
                                'users/<username>/<branch>',
                    default='trunk',
                    required=True,
                )

                with tests_vendor.value[TestsVendor.PARTNER]:
                    partner_test_option = sdk2.parameters.String(
                        'Partner tests options',
                        description='Choose one of the options',
                        default=TestsOptions.DEFAULT,
                        required=True,
                    )

                    partner_test_option.choices = [
                        ('Run all tests', TestsOptions.DEFAULT),
                        ('Run only adfox tests', TestsOptions.ADFOX),
                        ('Run only pcode tests', TestsOptions.PCODE),
                        ('Run tests by pattern', TestsOptions.PATTERN),
                    ]

                    with partner_test_option.value[TestsOptions.PATTERN]:
                        partner_test_pattern = sdk2.parameters.String(
                            'Pattern for tests',
                            description='Test1 Test2 Test3 \n'
                            'Test*',
                        )

            with sdk2.parameters.Group('Build settings. MySQL') as mysql_build_settings:
                mysql_source_type = sdk2.parameters.String(
                    'MySQL source',
                    description='Use custom image',
                    default=SourceType.LATEST,
                    required=True,
                )

                mysql_source_type.choices = [
                    ('Use latest version', SourceType.LATEST),
                    ('Use existing image', SourceType.TAG),
                ]

                with mysql_source_type.value[SourceType.TAG]:
                    mysql_tag = sdk2.parameters.String(
                        'MySQL tag',
                        description='Use MySQL custom image tag (e.g. 2.18.2601-2021-12-07)',
                        required=True,
                    )

        with sdk2.parameters.Group('Adfox') as adfox_settings:
            adfox_git_branch = sdk2.parameters.String(
                'Adfox tag',
                description="Версия docker-образа ветки Adfox",
                required=True,
                default='pre-release-login',
            )

            with tests_vendor.value[TestsVendor.ADFOX]:
                adfox_test_type = sdk2.parameters.String(
                    'Test type',
                    description='Тип тестов (чистый adfox или с вызовами partner api)',
                    required=True,
                    default=AdfoxTestType.ADFOX,
                )

                adfox_test_type.choices = [
                    ('Adfox', AdfoxTestType.ADFOX),
                    ('Partner', AdfoxTestType.PARTNER),
                ]

                adfox_test_paths = sdk2.parameters.String(
                    'Test paths',
                    description='Пути тестов от корня папки ".tests" (через пробел)',
                    required=True,
                    default='tests',
                )

                adfox_test_sets = sdk2.parameters.String(
                    'Hermione sets',
                    description='Фильтры (наборы) тестов hermione (через пробел)',
                    required=False,
                    default=None
                )

                adfox_test_threads = sdk2.parameters.Integer(
                    'Threads count',
                    description='Количество потоков hermione\n'
                                '(для тестов типа partner должно соответствовать количеству PI instances 1:1)',
                    required=True,
                    default=1,
                )

        with sdk2.parameters.Group('Test environment') as test_environment_settings:
            use_production_msf = sdk2.parameters.Bool(
                'Use production MSF',
                default=True
            )

            debug_run = sdk2.parameters.Bool(
                'Debug run',
                default=False,
                description='Task will suspend after docker start'
            )

            is_release = sdk2.parameters.Bool(
                'Is release',
                default=False,
                description='Task started in release (not in developers pipeline)'
            )

            instance_count = sdk2.parameters.Integer(
                'Count of partner instances',
                description='Количество инстансов ПИ',
                default=7,
                required=True,
            )

        with sdk2.parameters.Group('Secrets', collapse=True) as secrets_settings:
            mysql_password = sdk2.parameters.YavSecret(
                'MySQL password',
                default=cnt.MYSQL_PASSWORD_SECRET,
                required=True
            )
            pi2_backend_secret = sdk2.parameters.YavSecret(
                'Backend secret',
                default=cnt.PI2_BACKEND_SECRET,
                required=True
            )
            pi2_frontend_secret = sdk2.parameters.YavSecret(
                'Frontend secret',
                default=cnt.PI2_FRONTEND_SECRET,
                required=True
            )
            pi2_docker_pem = sdk2.parameters.YavSecret(
                'Docker PEM',
                default=cnt.PI2_DOCKER_PEM_SECRET,
                required=True
            )
            pi2_docker_pem_key = sdk2.parameters.String(
                'Docker PEM secret key',
                description='Key of _Docker PEM_ YaV record holding both private key and cert',
                default=cnt.PI2_DOCKER_PEM_SECRET_KEY,
                required=True
            )
            clickhouse_password = sdk2.parameters.YavSecret(
                'Clickhouse password',
                default=cnt.PARTNER2_CLICKHOUSE_PASSWORD_SECRET,
                required=True
            )
            juggler_token = sdk2.parameters.YavSecret(
                'Juggler token',
                default=cnt.JUGGLER_TOKEN_SECRET,
                required=True
            )
            metrika_management_token = sdk2.parameters.YavSecret(
                'Metrika management token',
                default=cnt.METRIKA_MANAGEMENT_TOKEN_SECRET,
                required=True
            )
            yql_token = sdk2.parameters.YavSecret(
                'YQL token',
                default=cnt.YQL_TOKEN_SECRET,
                required=True
            )
            yt_token = sdk2.parameters.YavSecret(
                'YT token',
                default=cnt.YT_TOKEN_SECRET,
                required=True
            )
            oauth_tus_token = sdk2.parameters.YavSecret(
                'OAuth TUS token',
                default=cnt.OAUTH_TUS_TOKEN_SECRET,
                required=True
            )
            api_key_token = sdk2.parameters.YavSecret(
                'API key token',
                default=cnt.API_KEY_TOKEN_SECRET,
                required=True
            )
            startrek_oauth_token = sdk2.parameters.YavSecret(
                'Startrek OAuth token',
                default=cnt.STARTREK_OAUTH_TOKEN_SECRET,
                required=True
            )
            s3_access_key = sdk2.parameters.YavSecret(
                'S3 API media storage access key',
                default=cnt.S3_ACCESS_KEY_SECRET,
                required=True
            )
            api_cabinet_service_token = sdk2.parameters.YavSecret(
                'Cabinet API service token',
                default=cnt.API_CABINET_SERVICE_TOKEN_SECRET,
                required=True
            )
            api_adfox_salt = sdk2.parameters.YavSecret(
                'Adfox API salt',
                default=cnt.API_ADFOX_SALT_SECRET,
                required=True
            )
            api_http_banner_storage_token = sdk2.parameters.YavSecret(
                'API HTTP banner storage token',
                default=cnt.API_HTTP_BANNER_STORAGE_TOKEN_SECRET,
                required=True
            )
            api_media_storage_auth_token = sdk2.parameters.YavSecret(
                'API media storage auth token',
                default=cnt.API_MEDIA_STORAGE_AUTH_TOKEN_SECRET,
                required=True
            )
            docker_auth_token = sdk2.parameters.YavSecret(
                'Docker auth token',
                default=cnt.DOCKER_AUTH_TOKEN_SECRET,
                required=True
            )
            adfox_pi_token = sdk2.parameters.YavSecret(
                'Adfox PI token',
                default=cnt.ADFOX_PI_SECRET,
                required=True
            )
            adfox_yav_token = sdk2.parameters.YavSecret(
                'Adfox YAV token',
                default=cnt.ADFOX_YAV_SECRET,
                required=True
            )
            adfox_tvm_token = sdk2.parameters.YavSecret(
                'Adfox TVM token',
                default=cnt.ADFOX_TVM_SECRET,
                required=True
            )
            robot_partner = sdk2.parameters.YavSecret(
                'Robot-partner secret',
                default=ps.ROBOT_PARTNER_SECRET,
                required=True
            )

    def on_execute(self):
        try:
            self.main_execute()

            if self.Parameters.st_issue:
                send_tests_result_message(
                    self.send_message,
                    RESULT_MESSAGE_ENTITY_NAME,
                    self.Context.report_data,
                    self.id
                )

            self.Context.task_status = 'SUCCESS'
        except Exception as ex:
            logging.error(ex, exc_info=True)

            if self.Parameters.st_issue:
                send_tests_result_message(
                    self.send_message,
                    RESULT_MESSAGE_ENTITY_NAME,
                    self.Context.report_data,
                    self.id,
                    True
                )

            self.Context.task_status = 'FAILURE'
            raise TaskFailure(ex)
        finally:
            # noinspection PyArgumentList
            self.Context.save()

    def shutdown(self):
        try:
            self.dns_switch(state=False)
        except Exception as ex:
            logging.error(ex)

        try:
            logging.info('Удаление контейнеров...')
            DockerCompose.down(project=str(self.id), volumes=True)
        except Exception as ex:
            logging.error(ex)

        try:
            if self.vendor_partner:
                logging.info('Отключение репозитория yharnam...')
                self.arc_client.finish()
        except Exception as ex:
            logging.error(ex)

    # noinspection PyUnusedLocal
    def on_finish(self, prev_status, status):
        self.shutdown()

    # noinspection PyUnusedLocal
    def on_break(self, prev_status, status):
        self.shutdown()

    def main_execute(self):
        """
        Основная функция.
        """
        self.Context.current_run += 1
        self.Context.start_time = time.time()

        # регистрируем DNS
        self.dns_switch()
        self.add_timing('dns-register')

        # инициализация
        self.init_docker_tags()
        self.init_resources()
        self.add_timing('init-resources')

        with sdk2.helpers.ProcessLog(self, stdout_level=logging.DEBUG, stderr_level=logging.DEBUG) as pl:
            # инициализация репозитория yharnam
            if self.vendor_partner:
                self.partner_init_yharnam()
            self.add_timing('init-yharnam')

            try:
                # запускаем контейнеры
                self.docker_start(pl)
                self.dump_logs(full=True, pl=pl)
                self.add_timing('start-docker')

                # тесты
                if self.vendor_partner:
                    self.partner_wait_tests(pl)
                elif self.vendor_adfox:
                    self.adfox_run_tests(pl)

                self.add_timing('tests-done')
            finally:
                try:
                    self.dump_logs(full=True)
                except Exception as ex:
                    logging.error(ex)

                # создаем ресурсы, отправляем отчеты
                if self.vendor_partner:
                    self.partner_make_report(pl)
                elif self.vendor_adfox:
                    self.adfox_make_report(pl)

        self.add_timing('reports-done')

        # оповещение adfox об успешном завершении теста релиза
        if self.vendor_partner and self.Parameters.is_release and self.Context.report_data['failed'] == 0:
            self.adfox_notify_success()

    def init_docker_tags(self):
        """
        Инициализация тегов образов перед поднятием сервисов.
        """
        logging.info('Подготовка образов docker...')

        tasks_to_wait = []

        result = self.init_partner_java_tag()
        if result:
            tasks_to_wait.append(result)

        result = self.init_partner_perl_tag()
        if result:
            tasks_to_wait.append(result)

        result = self.init_partner_frontend_tag()
        if result:
            tasks_to_wait.append(result)

        # не создают дочерних задач
        self.init_partner_mysql_tag()

        # выходим с ожиданием завершения дочерних тасок
        if len(tasks_to_wait):
            raise sdk2.WaitTask(tasks_to_wait, ctt.Status.Group.FINISH, timeout=2 * 3600)

    def init_partner_frontend_tag(self):
        if self.Context.partner_frontend_tag != '':
            return
        if self.Parameters.frontend_source_type == SourceType.PRODUCTION:
            prod_version = get_production_tag(VersionType.FRONTEND)
            docker_registry = DockerRegistry()
            image_name = '%s/%s' % (ps.DOCKER_REGISTRY, ps.FRONTEND_DOCKER_IMAGE_NAME)
            tag_list = docker_registry.tag_list(image_name, r'^%s' % prod_version)
            if len(tag_list) == 0:
                raise Exception('Cannot find tags for %s' % image_name)
            self.Context.partner_frontend_tag = tag_list[-1]
        elif self.Parameters.frontend_source_type == SourceType.TAG:
            self.Context.partner_frontend_tag = self.Parameters.frontend_tag
        elif self.Parameters.frontend_source_type == SourceType.BRANCH:
            if self.Context.frontend_build_task_id:
                task = sdk2.Task.find(id=self.Context.frontend_build_task_id).limit(1).first()
                if task.status == ctt.Status.SUCCESS:
                    logging.info(task)
                    logging.info('front build task')
                    logging.info(task.Parameters.image_name)
                    logging.info(task.Parameters)
                    self.Context.partner_frontend_tag = task.Parameters.image_name.split(':')[1]
                    logging.info(self.Context.partner_frontend_tag)
                else:
                    raise Exception('Что-то пошло не так с задачей сборки %s' % self.Context.frontend_build_task_id)
            else:
                frontend_build = BuildFrontend(
                    self,
                    build_type=ps.BUILD_TYPE_BETA,
                    arc_ref=self.Parameters.frontend_branch,
                )
                frontend_build.enqueue()
                self.Context.frontend_build_task_id = frontend_build.id
                return frontend_build.id
        else:
            raise Exception(
                'Тип %s не поддерживается, реализуйте этот функционал' % self.Parameters.frontend_source_type)

    def init_partner_mysql_tag(self):
        if self.Context.partner_mysql_tag != '':
            return
        if self.Parameters.mysql_source_type == SourceType.LATEST:
            docker_registry = DockerRegistry()
            image_name = '%s/%s' % (ps.DOCKER_REGISTRY, ps.AUTOTEST_DB_DOCKER_IMAGE_NAME)
            tag_list = docker_registry.tag_list(image_name, r'^\d\.\d{2}\.\d{4}-\d{4}-\d{2}-\d{2}$')
            if len(tag_list) == 0:
                raise Exception('Cannot find tags for %s' % image_name)
            self.Context.partner_mysql_tag = tag_list[-1]
        elif self.Parameters.mysql_source_type == SourceType.TAG:
            self.Context.partner_mysql_tag = self.Parameters.mysql_tag
        else:
            raise Exception('Тип %s не поддерживается, реализуйте этот функционал' % self.Parameters.mysql_source_type)

    def init_partner_perl_tag(self):
        if self.Context.partner_perl_tag != '':
            return
        if self.Parameters.perl_source_type == SourceType.PRODUCTION:
            self.Context.partner_perl_tag = get_production_tag(VersionType.PERL)
        elif self.Parameters.perl_source_type == SourceType.TAG:
            self.Context.partner_perl_tag = self.Parameters.perl_tag
        elif self.Parameters.perl_source_type == SourceType.BRANCH:
            if self.Context.perl_build_task_id:
                task = sdk2.Task.find(id=self.Context.perl_build_task_id).limit(1).first()
                if task.status == ctt.Status.SUCCESS:
                    logging.info(task)
                    logging.info('perl build task')
                    logging.info(task.Parameters.backend_image_name)
                    logging.info(task.Parameters)
                    self.Context.partner_perl_tag = task.Parameters.backend_image_name.split(':')[1]
                    logging.info(self.Context.partner_perl_tag)
                else:
                    raise Exception('Что-то пошло не так с задачей сборки %s' % self.Context.perl_build_task_id)
            else:
                # noinspection PyArgumentList
                perl_build = BuildDockerImage(
                    self,
                    build_type=ps.BUILD_TYPE_BETA,
                    ubuntu_version='bionic',
                    version='auto',
                    force_build=False,
                    repo='arc',
                    arc_branch=self.Parameters.perl_branch,
                    arc_project_path='partner/perl/partner2',
                )
                perl_build.enqueue()
                self.Context.perl_build_task_id = perl_build.id
                return perl_build.id
        else:
            raise Exception('Тип %s не поддерживается, реализуйте этот функционал' % self.Parameters.perl_source_type)

    def init_partner_java_tag(self):
        if self.Context.partner_java_tag != '':
            return
        if self.Parameters.java_source_type == SourceType.PRODUCTION:
            self.Context.partner_java_tag = get_production_tag(VersionType.JAVA)
        elif self.Parameters.java_source_type == SourceType.TAG:
            self.Context.partner_java_tag = self.Parameters.java_tag
        elif self.Parameters.java_source_type == SourceType.BRANCH:
            if self.Context.java_build_task_id:
                task = sdk2.Task.find(id=self.Context.java_build_task_id).limit(1).first()
                if task.status == ctt.Status.SUCCESS:
                    logging.info(task)
                    logging.info('java build task')
                    logging.info(task.Parameters.docker_images)
                    logging.info(task.Parameters)
                    self.Context.partner_java_tag = task.Parameters.docker_images.split(':')[-1]
                    logging.info(self.Context.partner_java_tag)
                else:
                    raise Exception('Что-то пошло не так с задачей сборки %s' % self.Context.java_build_task_id)
            else:
                # noinspection PyArgumentList
                java_build = PartnerBuildJavaDocker(
                    self,
                    arc_release_branch=self.Parameters.java_branch,
                    build_apps=[JSONAPI_APP_NAME, TESTAPI_APP_NAME],
                )
                java_build.enqueue()
                self.Context.java_build_task_id = java_build.id
                return java_build.id
        else:
            raise Exception('Тип %s не поддерживается, реализуйте этот функционал' % self.Parameters.java_source_type)

    def init_resources(self):
        """
        Подготовка файлов-ресурсов для запуска контейнеров.
        """
        logging.info('Подготовка ресурсов и конфигов...')

        pcode_api_host = ps.PCODE_API_HOST_PROD if self.Parameters.use_production_msf else ps.PCODE_API_HOST_TEST

        ResourceTemplate('/adfox/login-init-ci.sh', {
            '#{ADFOX_CURL_INIT_CONFIG}': self.adfox_curl_init_config,
        }).save()

        ResourceTemplate('/adfox/supervisor-ci.conf').save()

        ResourceTemplate('/backend/start.sh').save()
        ResourceTemplate('/backend/WebInterface.json').save()
        ResourceTemplate('/backend/nginx.conf').save()
        ResourceTemplate('/backend/fqdn').save()
        ResourceTemplate('/backend/tvmtool').save()

        ResourceTemplate('/backend/Application.json', {
            '{STARTREK_OAUTH_TOKEN}': self.Parameters.startrek_oauth_token.data()['key'],
            '{S3_ACCESS_KEY}': self.Parameters.s3_access_key.data()['access-secret-key'],
            '{API_CABINET_SERVICE_TOKEN}': self.Parameters.api_cabinet_service_token.data()['service_token'],
            '{API_ADFOX_SALT}': json.dumps(self.Parameters.api_adfox_salt.data()['salt']).strip('"'),
            '{API_HTTP_BANNER_STORAGE_TOKEN}': self.Parameters.api_http_banner_storage_token.data()['token'],
            '{API_MEDIA_STORAGE_AUTH_TOKEN}': self.Parameters.api_media_storage_auth_token.data()['token'],
            '{PCODE_API_HOST}': pcode_api_host,
        }).save()

        ResourceTemplate('/backend/DatabaseConfig.json', {
            '{MYSQL_PASSWORD}': self.Parameters.mysql_password.data()['password'],
        }).save()

        ResourceTemplate('/backend/env-locations.conf', {
            '{PCODE_API_HOST}': pcode_api_host,
        }).save()

        ResourceTemplate('/backend/tvm-config.json', {
            '{PI2_BACKEND_SECRET}': self.Parameters.pi2_backend_secret.data()['secret'],
            '{PI2_FRONTEND_SECRET}': self.Parameters.pi2_frontend_secret.data()['key'],
        }).save()

        StringTemplate(
            self.Parameters.pi2_docker_pem.data()[str(self.Parameters.pi2_docker_pem_key)]
        ).save('./backend/docker.partner.yandex.ru.pem')

        ResourceTemplate('/mysql/start.sh').save()

        ResourceTemplate('/docker-compose.yml', {
            '{TASK_ID}': self.id,
            '{PARTNER_DOMAIN}': cnt.PARTNER_DOMAIN,
            '{DOCKER_DOMAIN}': cnt.DOCKER_DOMAIN_TEMPLATE,
            '{ADFOX_TEST_TYPE}': self.Parameters.adfox_test_type,
            '{ADFOX_THREADS_COUNT}': self.adfox_threads_count,
            '{ADFOX_HOSTS_CONFIG}': self.adfox_hosts_pair_config,
            '{ADFOX_PI_TOKEN}': self.Parameters.adfox_pi_token.data()['pibot_devel'],
            '#{ADFOX_HOSTS_LIST}': self.adfox_hosts_list_config,
            '{ADFOX_RAMDISK_PATH}': os.path.join(str(self.ramdrive.path), 'adfox'),

            '{ADFOX_YAV_TOKEN}': self.Parameters.adfox_yav_token.data()['yav_token'],
            '{ADFOX_TVM_TOKEN}': self.Parameters.adfox_tvm_token.data()['TVMTOOL_SECRET'],
            '{PI2_BACKEND_SECRET}': self.Parameters.pi2_backend_secret.data()['secret'],
            '{STARTREK_OAUTH_TOKEN}': self.Parameters.startrek_oauth_token.data()['key'],
            '{TVMTOOL_LOCAL_AUTHTOKEN}': cnt.TVMTOOL_LOCAL_AUTHTOKEN,
            '{MYSQL_PASSWORD}': self.Parameters.mysql_password.data()['password'],
            '{SECRET_PARTNER2_CLICKHOUSE_PASSWORD}': self.Parameters.clickhouse_password.data()['testing'],
            '{SECRET_JUGGLER_TOKEN}': self.Parameters.juggler_token.data()['key'],
            '{SECRET_METRIKA_MANAGEMENT_TOKEN}': self.Parameters.metrika_management_token.data()['testing'],
            '{SECRET_YQL_TOKEN}': self.Parameters.yql_token.data()['yql-token'],
            '{SECRET_YT_TOKEN}': self.Parameters.yt_token.data()['key'],
            '{OAUTH_TUS_TOKEN}': self.Parameters.oauth_tus_token.data()['oauth_token'],
            '{API_KEY_TOKEN}': self.Parameters.api_key_token.data()['service_token'],

            '{ADFOX_LOGIN_TAG}': self.adfox_login_tag,
            '{PARTNER_FRONTEND_TAG}': self.Context.partner_frontend_tag,
            '{PARTNER_PERL_TAG}': self.Context.partner_perl_tag,
            '{PARTNER_JAVA_TAG}': self.Context.partner_java_tag,
            '{PARTNER_MYSQL_TAG}': self.Context.partner_mysql_tag,
            '{FRONTEND_BRANCH}': self.Parameters.frontend_branch,

            '#{PARTNER_INSTANCE_LIST}': ResourceTemplate('/docker-compose-tmpl.yml').clone(
                self.partner_instance_count,
                lambda i: {
                    '{THREAD_ID}': i,
                    '{INSTANCE_ID}': i + 1,
                    '{TEST_CHUNK}': "%d/%s" % (i + 1, str(self.partner_instance_count)),
                    '{TEST_PATTERN}': "%s" % (str(self.partner_test_pattern)),
                    '{TEST_OPTION}': "%s" % (str(self.partner_test_option)),
                    '{PARTNER_OUT_HTTP_PORT}': cnt.PARTNER_HTTP_ZERO_PORT + i * cnt.PORT_INCREMENT,
                    '{PARTNER_OUT_HTTPS_PORT}': cnt.PARTNER_HTTPS_ZERO_PORT + i * cnt.PORT_INCREMENT,
                    '{ADFOX_HOST}': cnt.ADFOX_DOMAIN_TEMPLATE,
                    '{E2E_BASE_URL_MOCK}': 'https://{DOCKER_DOMAIN}:{PARTNER_OUT_HTTPS_PORT}',
                    '{PARTNER_RAMDISK_PATH}': os.path.join(str(self.ramdrive.path), 'partner', str(i)),
                    '#{TESTS_SERVICE}': ResourceTemplate(
                        '/docker-compose-tests-tmpl.yml') if self.vendor_partner else '',
                }
            ),
        }).save() \
            .save(str(self.log_path('docker-compose.yml')))  # дополнительно сохраняем в логи

        ResourceTemplate('/tests/Dockerfile').save()
        ResourceTemplate('/tests/start.sh').save()

        ResourceTemplate('/config.json', {
            '{DOCKER_AUTH_TOKEN}': self.Parameters.docker_auth_token.data()['base64'],
        }).save('/home/sandbox/.docker/config.json')

    @property
    def arc_client(self):
        """
        :rtype: Arc
        """
        if not self._arc_client:
            self._arc_client = Arc(token=self.arc_token)
        return self._arc_client

    @property
    def arc_token(self):
        """
        :rtype: str
        """
        if not self.Parameters.robot_partner:
            raise Exception('Не задан параметр robot_partner')

        token = self.Parameters.robot_partner.data()['arc_token']
        return token

    @property
    def vendor_partner(self):
        """
        :return: true если запущен режим тестов partner
        :rtype: bool
        """
        return self.Parameters.tests_vendor == TestsVendor.PARTNER

    @property
    def vendor_adfox(self):
        """
        :return: true если запущен режим тестов adfox
        :rtype: bool
        """
        return self.Parameters.tests_vendor == TestsVendor.ADFOX

    @property
    def dns_name(self):
        """
        :return: dns-имя хоста таски
        :rtype: str
        """
        return cnt.DOCKER_DOMAIN_TEMPLATE.format(TASK_ID=self.id)

    @property
    def dns_api(self):
        """
        :rtype: DnsApi
        """
        if not self._dns_api:
            self._dns_api = DnsApi('robot-partner', self.Parameters.robot_partner.data()['dns_api'])
        return self._dns_api

    def dns_switch(self, state=True):
        """
        :param state: создать/удалить
        :type state: bool
        """
        # удаляем предыдущую запись если имеется
        if self.Context.dns_name and self.Context.dns_ip:
            try:
                self.dns_api.delete_dns_record('*.' + self.Context.dns_name, self.Context.dns_ip)
                time.sleep(1)
                self.dns_api.delete_dns_record(self.Context.dns_name, self.Context.dns_ip)
            except Exception as ex:
                logging.warning(ex)

        # добавляем новую если необходимо
        if state:
            dns_name = self.dns_name
            dns_ip = network.get_my_ipv6()
            self.retry(3, 5, lambda: self.dns_api.create_dns_record(dns_name, dns_ip))
            self.retry(3, 5, lambda: self.dns_api.create_dns_record('*.%s' % dns_name, dns_ip))
            self.Context.dns_name = dns_name
            self.Context.dns_ip = dns_ip

    def docker_start(self, pl):
        """
        :type pl: sandbox.sdk2.helpers.ProcessLog
        """
        os.environ['DOCKER_CLIENT_TIMEOUT'] = '300'
        os.environ['COMPOSE_HTTP_TIMEOUT'] = '300'

        logging.info('Авторизация в registry.yandex.net...')
        self.retry(5, 5, lambda: Docker.login(
            'registry.yandex.net', user='robot-pereiro',
            passwd=self.Parameters.docker_auth_token.data()['key'], pl=pl
        ))

        logging.info('Загрузка обновлений образов...')
        self.retry(3, 5, lambda: DockerCompose.pull(pl=pl))
        self.add_timing('docker-pull')

        # очищаем агент от старых процессов (уже после того как сделали docker login)
        Docker.rm_all(force=True, volumes=True, pl=pl)

        logging.info('Запуск контейнеров...')
        self.retry(3, 5, lambda: DockerCompose.up(
            project=str(self.id), build=True, force_recreate=True, detach=True, pl=pl
        ))

        if self.Parameters.debug_run:
            self.suspend()
        else:
            logging.info('Ожидание 30 секунд запуска контейнеров...')
            time.sleep(30)

    def dump_logs(self, full=False, pl=None):
        """
        :param full: полный лог всех контейнеров отдельно, либо только общий лог docker-compose
        :type full: bool
        :type pl: sandbox.sdk2.helpers.ProcessLog
        """
        logging.debug('Дамп логов ...')

        # общий лог compose
        try:
            DockerCompose.dump_logs(str(self.log_path()), project=str(self.id), pl=pl)
        except Exception as ex:
            logging.warning(ex)

        if full:
            # логи каждого контейнера
            try:
                # используем "show_all" чтобы получить logs также и для упавших (not running) контейнеров
                Docker.dump_all_logs(str(self.log_path()), show_all=True, pl=pl)
            except Exception as ex:
                logging.warning(ex)

            # достаем файлы логов adfox
            for log in ('nginx/.', 'php-error.log', 'supervisor/.'):
                # noinspection PyBroadException
                try:
                    Docker.cp(
                        '%s:/var/log/%s' % (self.adfox_login_container, log),
                        str(self.log_path('adfox-login-%s' % log.rstrip('/.').replace('/', '-'))),
                        pl=pl,
                    )
                except Exception as ex:
                    logging.warning(ex)

            # логи PI mysql
            for i in range(self.partner_instance_count):
                instance_id = i + 1
                container_log_dir_path = self.partner_mysql_container(instance_id) + ":" + "/var/log/mysql"
                task_log_dir_path = "pi-mysql-%s" % instance_id
                try:
                    Docker.cp(
                        container_log_dir_path,
                        str(self.log_path(task_log_dir_path)),
                        pl=pl
                    )
                except Exception as ex:
                    logging.warning(ex)

    @staticmethod
    def retry(count, sleep, func):
        """
        :param count: кол-во попыток
        :type count: int
        :param sleep: пауза между попытками, сек
        :type sleep: int
        :param func: вызываемая функция
        :return: результат работы функции
        """
        last_error = None
        for i in range(count):
            try:
                return func()
            except Exception as ex:
                last_error = ex
                logging.warning(ex)
                logging.debug('Повторяем, попытка %d из %d' % (i + 2, count))
                time.sleep(sleep)
        raise last_error

    def add_timing(self, name):
        """
        Сохранение метки времени процесса.

        :param name: название метки
        :type name: str
        """
        self.Context.timing[name] = math.floor(time.time() - self.Context.start_time)

    def krush_send_logs(self):
        logging.info('Отправка статистики Krush из: ' + self.report_json_path + ' ...')

        common_data = {
            'project': self.report_project_name,
            'start_time': self.Context.start_time,
            'instance_count': self.partner_instance_count,
            'frontend_hash': self.commit_sha,
            'frontend_source': self.Parameters.frontend_source_type,
            'frontend_branch': self.Parameters.frontend_branch,
            'perl_source': self.Parameters.perl_source_type,
            'perl_branch': self.Parameters.perl_branch,
            'java_source': self.Parameters.java_source_type,
            'java_branch': self.Parameters.java_branch,
            'adfox_source': self.adfox_login_tag,
            'task_id': self.id,
            'ci_pipeline_id': self.Context.ci_pipeline_id,
            'ci_job_id': self.Context.ci_job_id,
            'test_option': self.Context.partner_test_option,
        }

        # конвертируем json_report
        data = KrushLogger.convert_report_json(self.report_json_path, record_enricher=common_data)

        # статистика по тестам
        KrushLogger.send_data('autotests', data['records'])

        # суммарная статистика
        totals = data['totals']
        totals.update(common_data)
        totals.update({'timing': self.Context.timing})
        KrushLogger.send_data('autotests-total', totals)

    def partner_init_yharnam(self):
        """
        Инициализация локального репозитория arc для тестов Partner.
        """
        logging.info('Инициализация arc-репозитория yharnam...')

        self.arc_client.checkout(self.Parameters.frontend_branch)

        shutil.move(
            os.path.join(str(self.arc_client.path), 'adv/frontend/services/yharnam'),
            str(self.path('tests', 'yharnam'))
        )

    def partner_wait_tests(self, pl):
        """
        Ожидание окончания тестов Partner.

        :type pl: sandbox.sdk2.helpers.ProcessLog
        """
        logging.info('Ожидаем завершения тестов PARTNER %d сек...' % SLEEP_TIME_MAX)
        sleeping_time = 0

        while True:
            self.dump_logs(full=True, pl=pl)

            processes = filter(
                # ожидаем только свои процессы, исключая зависшие с прошлых задач
                lambda s: s.startswith('%d_tests-' % int(self.id)),
                Docker.ps(fmt='{{.Names}} {{.RunningFor}}', pl=pl).split("\n")
            )

            if not processes:
                break

            if sleeping_time > SLEEP_TIME_MAX:
                raise Exception('Превышено время ожидания завершения тестов')

            logging.debug('Ожидаем завершения процессов:\n' + '\n'.join(processes))
            sleeping_time += SLEEP_TIME
            time.sleep(SLEEP_TIME)

    logging.info('Tests FINISHED')

    def partner_make_report(self, pl):
        """
        Обработка отчета тестов Partner.

        :type pl: sandbox.sdk2.helpers.ProcessLog
        """
        logging.info('Обработка отчетов PARTNER...')
        os.environ['FRONTEND_BRANCH'] = str(self.report_branch)

        os.environ['JAVA_BRANCH_OR_TAG'] = str(
            self.Parameters.java_branch if self.Parameters.java_source_type == SourceType.BRANCH
            else self.Context.partner_java_tag
        )

        os.environ['PERL_BRANCH_OR_TAG'] = str(
            self.Parameters.perl_branch if self.Parameters.perl_source_type == SourceType.BRANCH
            else self.Context.partner_perl_tag
        )

        # сохраняем результаты тестов (отчеты) в логи
        exec_cmd(
            ['cp', '-R'] +
            [str(self.path('tests', 'result-%d' % i)) for i in range(1, int(self.Parameters.instance_count) + 1)] +
            [str(self.log_path())], pl=pl
        )

        # объединяем отчеты
        try:
            os.chdir(str(self.path(PARTNER_AUTOTESTS_PATH)))

            exec_cmd(['npm', 'ci'], pl)

            exec_cmd(
                ['npm', 'run', 'test:autotests:merge-reports:docker'] +
                [str(self.path('tests', 'result-%d' % i, PARTNER_REPORT_HTML)) for i in
                 range(1, int(self.Parameters.instance_count) + 1)],
                pl=pl
            )
        finally:
            os.chdir(str(self.path()))

        self.send_reports(
            is_release=(self.report_branch == 'trunk') or is_rc_branch(self.report_branch)
        )

    def adfox_run_tests(self, pl):
        """
        Запуск тестов Adfox.

        :type pl: sandbox.sdk2.helpers.ProcessLog
        """
        logging.info("Запускаем тесты ADFOX...")

        cmd = [
            './node_modules/.bin/hermione', '-c', './tests/hermione/.hermione-pipeline.conf.js',
        ]

        # пути тестов
        cmd += map(
            lambda path: re.sub(r'[.]ts$', '.js', path),
            re.split(
                r'[\s,]+', str(self.Parameters.adfox_test_paths).strip()
            )
        )

        # фильтры тестов
        if self.Parameters.adfox_test_sets:
            sets_args = []
            for s in re.split(r'[\s,]+', str(self.Parameters.adfox_test_sets).strip()):
                if len(s):
                    sets_args += ['--set', s]
            cmd += sets_args

        try:
            Docker.execute(self.adfox_login_container, cmd, workdir=ADFOX_TESTS_ROOT, pl=pl)
            logging.info('Tests SUCCESS !!!')
        except Exception as ex:
            logging.info('Tests FAILED !!!')
            logging.info(ex)

    def adfox_make_report(self, pl):
        """
        Обработка отчета тестов Adfox.

        :type pl: sandbox.sdk2.helpers.ProcessLog
        """
        logging.info('Обработка отчетов ADFOX...')

        # проверяем наличие отчета
        if not self.path(ADFOX_REPORT_HTML, ADFOX_REPORT_JSON).exists():
            logging.warning('Отчет не найден')
        else:
            # меняем права на отчет, так как он записывался в контейнере от root
            Docker.execute(
                self.adfox_login_container,
                ['chmod', '-R', 'ugo=rwX', ADFOX_TESTS_ROOT + '/' + ADFOX_REPORT_HTML],
                pl=pl
            )

            # достаем report.json из html-отчета для публикации в отдельные ресурсы
            shutil.copyfile(
                str(self.path(ADFOX_REPORT_HTML, ADFOX_REPORT_JSON)),
                self.report_json_path,
            )

            self.send_reports(
                is_release=self.Parameters.is_release or
                           (self.report_branch.find(GitConfig.RELEASE_CANDIDATE_PREFIX) >= 0)
            )

    def send_reports(self, is_release=False):
        """
        Обработка отчета.

        :param is_release: отчет для релизной ветки
        :type is_release: bool
        """
        self.Context.json_report_filepath = self.report_json_path

        try:
            self.krush_send_logs()
        except Exception as ex:
            logging.error(ex)

        try:
            self.publish_json_report(self.report_json_path, self.report_branch, project=self.report_project_name)
        except Exception as ex:
            logging.error(ex)

        self.make_report(
            branch=self.report_branch,
            collect_stats=str(is_release).lower(),
            commit_sha=self.commit_sha,
            project=self.report_project_name,
        )

    def adfox_notify_success(self):
        """
        Отправка оповещения для релизной ветки.
        """
        logging.debug('Отправляем ADFOX сигнал успешного завершения тестов ...')
        try:
            dutyTools.mark_release(self.adfox_login_tag, 'pi_tests_ok')
        except Exception as ex:
            logging.error(ex, exc_info=True)

    @property
    def adfox_login_tag(self):
        """
        :return: branch для Adfox.
        :rtype: str
        """
        if not self.Context.adfox_login_tag:
            self.Context.adfox_login_tag = AdfoxReleaseIntegration.resolve_branch(
                self.Parameters.adfox_git_branch
            )

            logging.debug('ADFOX_LOGIN_TAG=' + self.adfox_login_tag)

        return self.Context.adfox_login_tag

    @property
    def adfox_login_container(self):
        """
        :return: название процесса контейнера adfox-login
        :rtype: str
        """
        return '%d_adfox-login_1' % int(self.id)

    @property
    def adfox_map_single_pi(self):
        """
        :return: true если все потоки adfox ходят в один хост PI
        :rtype: bool
        """
        return self.vendor_adfox and self.Parameters.adfox_test_type == 'adfox'

    @property
    def partner_instance_count(self):
        """
        :return: количество ПИ instances для тестов Adfox
        :rtype: int
        """
        return 1 if self.adfox_map_single_pi else int(self.Parameters.instance_count)

    def partner_mysql_container(self, instance_id):
        return "%s_mysql-%s_1" % (self.id, instance_id)

    @property
    def partner_test_pattern(self):
        """
        :return: кастомный паттерн тестов ПИ
        :rtype: str
        """
        pattern = self.Parameters.partner_test_pattern if self.Parameters.partner_test_option == TestsOptions.PATTERN else ''
        self.Context.partner_test_pattern = pattern
        return pattern

    @property
    def partner_test_option(self):
        """
        :return: вид тестов ПИ
        :rtype: str
        """
        option = self.Parameters.partner_test_option
        self.Context.partner_test_option = option
        return option

    @property
    def adfox_threads_count(self):
        """
        :return: кол-во потоков ADFOX
        :rtype: int
        """
        return int(self.Parameters.adfox_test_threads if self.adfox_map_single_pi else self.Parameters.instance_count)

    @property
    def adfox_hosts_list_config(self):
        """
        :return: конфиг хостов adfox (extra_hosts)
        :rtype: StringTemplate
        """
        return StringTemplate(cnt.ADFOX_HOSTS_LIST_TEMPLATE).clone(
            self.adfox_threads_count,
            lambda i: {
                '{THREAD_ID}': i,
                '{TASK_ID}': self.id
            }
        )

    @property
    def adfox_hosts_pair_config(self):
        """
        :return конфигурация пар хостов для автотестов adfox
        :rtype: str
        """

        return ";".join([
            'https://{ADFOX_DOMAIN}|https://{DOCKER_DOMAIN}:{PARTNER_PORT}'.format(
                ADFOX_DOMAIN=cnt.ADFOX_DOMAIN_TEMPLATE.format(THREAD_ID=i, TASK_ID=self.id),
                DOCKER_DOMAIN=self.dns_name,
                PARTNER_PORT=cnt.PARTNER_HTTPS_ZERO_PORT + cnt.PORT_INCREMENT * (0 if self.adfox_map_single_pi else i)
            ) for i in range(0, self.adfox_threads_count)
        ])

    @property
    def adfox_curl_init_config(self):
        """
        :return: конфиг curl-инициализации adfox-login для тестов partner
        :rtype: str
        """
        return StringTemplate(cnt.ADFOX_CURL_INIT_TEMPLATE).clone(
            self.adfox_threads_count,
            lambda i: {
                '{THREAD_ID}': i,
                '{TASK_ID}': self.id,
            }
        ) if self.vendor_partner else ''

    @property
    def template_dir_path(self):
        return os.path.dirname(os.path.abspath(__file__))

    @property
    def report_dir_path(self):
        """
        :return: путь директории hermione html-отчета
        :rtype: str
        """

        return str(
            self.path(PARTNER_REPORTS_PATH, PARTNER_REPORT_HTML) if self.vendor_partner
            else self.path(ADFOX_REPORT_HTML)
        )

    @property
    def report_json_path(self):
        """
        :return: путь hermione json-отчета
        :rtype: str
        """
        return str(
            self.path(PARTNER_REPORTS_PATH, PARTNER_REPORT_JSON) if self.vendor_partner
            else self.path(ADFOX_REPORT_JSON)
        )

    @property
    def report_branch(self):
        """
        :return: ветка кода (branch) для отчета
        :rtype: str
        """
        return self.Parameters.frontend_branch if self.vendor_partner else self.adfox_login_tag

    @property
    def report_project_name(self):
        """
        :return: проект testcop
        :rtype: str
        """
        return REPORT_PROJECT_PARTNER if self.vendor_partner else REPORT_PROJECT_ADFOX

    @property
    def commit_sha(self):
        """
        :return: commit sha ветки
        :rtype: str
        """
        if not self.Context.commit_sha:
            self.Context.commit_sha = self.arc_client.get_head_commit_hash() if self.vendor_partner else 'none'
        return self.Context.commit_sha
