# coding=utf-8

import json
import os
import re

import requests

import constants as cnt

try:
    from library.python.vault_client.instances import Production as VaultClient
except ImportError:
    from vault_client.instances import Production as VaultClient

INSTANCE_COUNT = 1
PCODE_API_HOST_PROD = 'pcode-msf-prod.yandex-team.ru'


class Tpl(object):
    _path = None
    _content = ''

    def __init__(self, content=None, replace=None):
        if content:
            if os.path.exists(content):
                self._path = content
                with open(content, 'r') as f:
                    self._content = f.read()
            else:
                self._content = content

            if replace:
                old_content = None
                while self._content != old_content:
                    old_content = self._content
                    for key in replace:
                        self._content = self._content.replace(key, str(replace[key]))

    def save(self, path=None):
        if not path:
            if not self._path:
                raise Exception('path not specified')
            path = self._path
        with open(path, 'w') as f:
            f.write(self._content)

    def __str__(self):
        return self._content


class LocalHermione(object):
    """
    Класс для инициализации конфигов docker-контейнеров при локальном запуске
    """
    _prod_version_json = None
    _vault_client = None
    _partner_frontend_tag = None
    _partner_perl_tag = None
    _partner_java_tag = None
    _partner_mysql_tag = None
    _adfox_login_tag = None
    _adfox_mysql_tag = None

    @staticmethod
    def replace(text, values):
        replaced = True
        while replaced:
            replaced = False
            for key in values:
                old_content = text
                text = text.replace(key, str(values[key]))
                if text != old_content:
                    replaced = True

        return text

    @staticmethod
    def docker_registry(package, resource):
        return cnt.DOCKER_REGISTRY_TEMPLATE % (package, resource)

    def docker_fetch(self, package, resource, method='GET'):
        return requests.request(
            method,
            LocalHermione.docker_registry(package, resource),
            headers={
                'Authorization': 'OAuth %s' % self.secret(cnt.DOCKER_AUTH_TOKEN_SECRET, 'key'),
                'Accept': 'application/vnd.docker.distribution.manifest.v2+json',
            }
        )

    def docker_tag_list(self, package, tag_filter=None):
        resp = self.docker_fetch(package, 'tags/list')
        tags = resp.json().get('tags', [])
        if tag_filter:
            tags = filter(lambda x: re.match(tag_filter, x), tags)
        return sorted(tags)[-1]

    @property
    def prod_version_json(self):
        if not self._prod_version_json:
            print('Получение продовых версий PI: ' + cnt.PARTNER_PROD_VERSION_URL + '...')
            self._prod_version_json = json.loads(
                requests.get(cnt.PARTNER_PROD_VERSION_URL, verify=False).text
            )
        return self._prod_version_json

    @property
    def partner_frontend_tag(self):
        if not self._partner_frontend_tag:
            self._partner_frontend_tag = self.docker_tag_list(
                'partners/frontend-node', r'^%s' % self.prod_version_json.get('f')
            )
            print('PARTNER_FRONTEND_TAG=' + self._partner_frontend_tag)
        return self._partner_frontend_tag

    @property
    def partner_perl_tag(self):
        if not self._partner_perl_tag:
            self._partner_perl_tag = self.prod_version_json.get('s')
            print('PARTNER_PERL_TAG=' + self._partner_perl_tag)
        return self._partner_perl_tag

    @property
    def partner_java_tag(self):
        if not self._partner_java_tag:
            self._partner_java_tag = self.prod_version_json.get('j')
            print('PARTNER_JAVA_TAG=' + self._partner_java_tag)
        return self._partner_java_tag

    @property
    def partner_mysql_tag(self):
        if not self._partner_mysql_tag:
            self._partner_mysql_tag = self.docker_tag_list(
                'partners/partner2-db-autotest', r'^\d\.\d{2}\.\d{4}-\d{4}-\d{2}-\d{2}$'
            )
            print('PARTNER_MYSQL_TAG=' + self._partner_mysql_tag)
        return self._partner_mysql_tag

    @property
    def adfox_login_tag(self):
        if not self._adfox_login_tag:
            self._adfox_login_tag = 'pre-release-login'  # переделать на получение продового
            print('ADFOX_LOGIN_TAG=' + self._adfox_login_tag)
        return self._adfox_login_tag

    @property
    def adfox_mysql_tag(self):
        if not self._adfox_mysql_tag:
            self._adfox_mysql_tag = '5.7'  # переделать на получение нужной версии
            print('ADFOX_MYSQL_TAG=' + self._adfox_mysql_tag)
        return self._adfox_mysql_tag

    @property
    def vault_client(self):
        if not self._vault_client:
            self._vault_client = VaultClient()
        return self._vault_client

    def secret(self, version_id, key):
        return self.vault_client.get_version(version_id)['value'][key]

    def init_resources(self):
        """
        Подготовка файлов-ресурсов для запуска контейнеров.
        """
        print('Инициализация конфигов контейнеров docker...')

        basedir = os.path.dirname(__file__)

        Tpl(os.path.join(basedir, 'docker/adfox/login-init-ci.sh'), {
            '#{ADFOX_CURL_INIT_CONFIG}': cnt.ADFOX_CURL_INIT_TEMPLATE,
            '{THREAD_ID}': 0,
            '{TASK_ID}': 1,
        }).save()

        Tpl(os.path.join(basedir, 'docker/backend/Application.json'), {
            '{STARTREK_OAUTH_TOKEN}': self.secret(cnt.STARTREK_OAUTH_TOKEN_SECRET, 'key'),
            '{S3_ACCESS_KEY}': self.secret(cnt.S3_ACCESS_KEY_SECRET, 'access-secret-key'),
            '{API_CABINET_SERVICE_TOKEN}': self.secret(cnt.API_KEY_TOKEN_SECRET, 'service_token'),
            '{API_ADFOX_SALT}': json.dumps(self.secret(cnt.API_ADFOX_SALT_SECRET, 'salt')).strip('"'),
            '{API_HTTP_BANNER_STORAGE_TOKEN}': self.secret(cnt.API_HTTP_BANNER_STORAGE_TOKEN_SECRET, 'token'),
            '{API_MEDIA_STORAGE_AUTH_TOKEN}': self.secret(cnt.API_MEDIA_STORAGE_AUTH_TOKEN_SECRET, 'token'),
            '{PCODE_API_HOST}': PCODE_API_HOST_PROD,
        }).save()

        Tpl(os.path.join(basedir, 'docker/backend/DatabaseConfig.json'), {
            '{MYSQL_PASSWORD}': self.secret(cnt.MYSQL_PASSWORD_SECRET, 'password'),
        }).save()

        Tpl(os.path.join(basedir, 'docker/backend/env-locations.conf'), {
            '{PCODE_API_HOST}': PCODE_API_HOST_PROD,
        }).save()

        Tpl(os.path.join(basedir, 'docker/backend/tvm-config.json'), {
            '{PI2_BACKEND_SECRET}': self.secret(cnt.PI2_BACKEND_SECRET, 'secret'),
            '{PI2_FRONTEND_SECRET}': self.secret(cnt.PI2_FRONTEND_SECRET, 'key'),
        }).save()

        Tpl(
            self.secret(cnt.PI2_DOCKER_PEM_SECRET, cnt.PI2_DOCKER_PEM_SECRET_KEY)
        ).save(os.path.join(basedir, 'docker/backend/docker.partner.yandex.ru.pem'))

        Tpl(os.path.join(basedir, 'docker/docker-compose.yml'), {
            '{TASK_ID}': 1,
            '{THREAD_ID}': 0,
            '{INSTANCE_ID}': 1,
            '{PARTNER_DOMAIN}': cnt.PARTNER_DOMAIN,
            '{DOCKER_DOMAIN}': cnt.DOCKER_DOMAIN_TEMPLATE,
            '{ADFOX_TEST_TYPE}': '-',
            '{ADFOX_THREADS_COUNT}': '-',
            '{ADFOX_HOSTS_CONFIG}': '-',
            '#{ADFOX_HOSTS_LIST}': cnt.ADFOX_HOSTS_LIST_TEMPLATE,
            '    volumes:\n      - "{ADFOX_RAMDISK_PATH}:/mnt/ramdisk"\n': '',
            '{ADFOX_TVM_TOKEN}': self.secret(cnt.ADFOX_TVM_SECRET, 'TVMTOOL_SECRET'),
            '{ADFOX_YAV_TOKEN}': self.secret(cnt.ADFOX_YAV_SECRET, 'yav_token'),
            '{ADFOX_PI_TOKEN}': self.secret(cnt.ADFOX_PI_SECRET, 'pibot_devel'),
            '{PI2_BACKEND_SECRET}': self.secret(cnt.PI2_BACKEND_SECRET, 'secret'),
            '{STARTREK_OAUTH_TOKEN}': self.secret(cnt.STARTREK_OAUTH_TOKEN_SECRET, 'key'),
            '{TVMTOOL_LOCAL_AUTHTOKEN}': 'rJSJ51mAAAECDMN9WmuwErCt4YKZZEfp',
            '{MYSQL_PASSWORD}': self.secret(cnt.MYSQL_PASSWORD_SECRET, 'password'),
            '{SECRET_PARTNER2_CLICKHOUSE_PASSWORD}': self.secret(cnt.PARTNER2_CLICKHOUSE_PASSWORD_SECRET, 'testing'),
            '{SECRET_JUGGLER_TOKEN}': self.secret(cnt.JUGGLER_TOKEN_SECRET, 'key'),
            '{SECRET_METRIKA_MANAGEMENT_TOKEN}': self.secret(cnt.METRIKA_MANAGEMENT_TOKEN_SECRET, 'testing'),
            '{SECRET_YQL_TOKEN}': self.secret(cnt.YQL_TOKEN_SECRET, 'yql-token'),
            '{SECRET_YT_TOKEN}': self.secret(cnt.YT_TOKEN_SECRET, 'key'),
            '{OAUTH_TUS_TOKEN}': self.secret(cnt.OAUTH_TUS_TOKEN_SECRET, 'oauth_token'),
            '{API_KEY_TOKEN}': self.secret(cnt.API_CABINET_SERVICE_TOKEN_SECRET, 'service_token'),

            '{ADFOX_LOGIN_TAG}': self.adfox_login_tag,
            '{PARTNER_FRONTEND_TAG}': self.partner_frontend_tag,
            '{PARTNER_PERL_TAG}': self.partner_perl_tag,
            '{PARTNER_JAVA_TAG}': self.partner_java_tag,
            '{PARTNER_MYSQL_TAG}': self.partner_mysql_tag,

            '#{PARTNER_INSTANCE_LIST}': Tpl(os.path.join(basedir, 'docker/docker-compose-tmpl.yml'), {
                '{TEST_CHUNK}': "1/1",
                '{ADFOX_HOST}': cnt.ADFOX_DOMAIN_TEMPLATE,
                '{PARTNER_OUT_HTTP_PORT}': cnt.PARTNER_HTTP_ZERO_PORT,
                '{PARTNER_OUT_HTTPS_PORT}': cnt.PARTNER_HTTPS_ZERO_PORT,
                '{E2E_BASE_URL_MOCK}': 'https://{DOCKER_DOMAIN}:{PARTNER_OUT_HTTPS_PORT}',
                '      - "{PARTNER_RAMDISK_PATH}:/mnt/ramdisk"\n': '',
                '#{TESTS_SERVICE}': '',
            }),

            # перенаправляем все сервисы на мок в adfox (при запуске в sandbox мок запускается в контейнере tests)
            '127.0.0.1:3000': '172.17.0.1:3000',
        }).save()

    @staticmethod
    def help():
        print(Tpl(
            """
            На ноутбуке прописать адреса в /etc/hosts:
                {IP}\t{PARTNER_DOMAIN} {ADFOX_DOMAIN}

            * где '{IP}' - адрес стенда, на котором запускается docker.

            Для запуска docker-контейнеров:
                docker-compose -f ./docker/docker-compose.yml up -d --force-recreate

            Адреса в браузере:
                Adfox: https://{ADFOX_DOMAIN}
                Partner: https://{PARTNER_DOMAIN}:{PARTNER_HTTPS_PORT}

            Запуск тестов:
                export ADFOX_HOST="https://{ADFOX_DOMAIN}"
                export MOCK_URL="http://{PARTNER_DOMAIN}:3000/api"
                export STAND_URL="https://{PARTNER_DOMAIN}:{PARTNER_HTTPS_PORT}"

                npm run test:autotests:docker:local -- --mockUrl $MOCK_URL --stand $STAND_URL --test МойТест

            Полное описание в документации:
                https://a.yandex-team.ru/arcadia/sandbox/projects/partner/tasks/docker_hermione/Readme.md
            """, {
                '{THREAD_ID}': 0,
                '{TASK_ID}': 1,
                '{ADFOX_DOMAIN}': cnt.ADFOX_DOMAIN_TEMPLATE,
                '{PARTNER_DOMAIN}': cnt.PARTNER_DOMAIN,
                '{PARTNER_HTTPS_PORT}': cnt.PARTNER_HTTPS_ZERO_PORT
            }
        ))

    def process(self):
        self.init_resources()
        self.help()
