# coding: utf-8

import logging
import os
import re
import subprocess as sp
from urlparse import urlparse

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.projects.partner.settings import DOCKER_REGISTRY
from sandbox.sdk2 import Task


def configure_internal_ca():
    cacert_path = '/etc/ssl/certs/ca-certificates.crt'
    os.environ['CURL_CA_BUNDLE'] = cacert_path
    os.environ['NODE_EXTRA_CA_CERTS'] = cacert_path
    os.environ['REQUESTS_CA_BUNDLE'] = cacert_path


def get_error_status_explanation(status):
    if status == ctt.Status.TIMEOUT:
        return 'превышен таймаут ожидания'
    if status == ctt.Status.STOPPED:
        return 'остановлено'
    return 'подробности в логах'


def make_cmd_exec_args(args):
    """
    :type args: list | tuple
    :rtype: list
    """
    if type(args) != list:
        args = [args]

    return args


def log_cmd(args, pl):
    """
    :type args list
    :type pl: sandbox.sdk2.helpers.ProcessLog
    """
    logger = pl.logger if pl else logging
    logger.debug('+ ' + ' '.join(args))


def call_cmd(args, pl=None, grab_stderr=False):
    """
    Call shell command and return its output

    :type args: list
    :type pl: sandbox.sdk2.helpers.ProcessLog
    :param grab_stderr: возвращать stderr также
    :type grab_stderr: bool
    :return: вывод команды
    :rtype: str
    """
    full_args = make_cmd_exec_args(args)
    log_cmd(full_args, pl)

    if grab_stderr:
        stderr = sp.STDOUT
    elif pl:
        stderr = pl.stderr
    else:
        stderr = sp.PIPE

    return sp.check_output(full_args, stderr=stderr).strip()


def exec_cmd(args, pl=None):
    """
    Execute shell command redirecting all output to process log

    :type args: list
    :type pl: sandbox.sdk2.helpers.ProcessLog
    """
    full_args = make_cmd_exec_args(args)
    log_cmd(full_args, pl)
    sp.check_call(
        full_args,
        stderr=pl.stderr if pl else sp.PIPE,
        stdout=pl.stdout if pl else sp.PIPE
    )


def make_script_call_cmd(script, *args):
    # @formatter:off
    return [
        'npx',
        'ts-node',
        '-P', 'tasks/tsconfig.json',
        '-r', 'tsconfig-paths/register',
        '--files',
        os.path.join('tasks', script)
    ] + list(args)
    # @formatter:on


def is_commit_hash(ref):
    commit_hash_re = r'^[0-9a-f]{40}$'
    return bool(re.match(commit_hash_re, ref))


def is_dev_mode(registry):
    return registry.common.installation in ctm.Installation.Group.LOCAL


def is_valid_startrack_issue(st_issue):
    return bool(re.match(r'PI-\d+', st_issue))


def parse_docker_image_name(image_name, base_name):
    if re.match(r'^https?://', image_name):
        parsed = urlparse(image_name)
        repo = parsed.netloc
        local_name = parsed.path
    else:
        repo = DOCKER_REGISTRY
        local_name = image_name

    if local_name.find(':') > 0:
        (name, tag) = local_name.split(':')
    else:
        name = base_name
        tag = local_name

    return name, tag, repo


def parse_hostname(host):
    return host if host.find('//') < 0 else urlparse(host).netloc


def find_clone_source(current):
    """
    Найти задачу - источник клонирования текущей

    :param current:  sandbox.sdk2.Task
    :return: sdk2.Task | None
    """

    logging.debug('find_clone_source of %d' % current.id)

    if current.Context.copy_of:
        source_id = current.Context.copy_of
        logging.debug('find_clone_source source_id: #%d' % source_id)
        return Task.find(id=source_id).first()
