# -*- coding: utf-8 -*-

import logging
import os
import re

from template import StringTemplate
from . import call_cmd, exec_cmd

DOCKER = 'docker'


class Docker(object):

    @staticmethod
    def login(host, user=None, passwd=None, pl=None):
        """
        docker login

        :param host: registry host
        :type host: str
        :param user: логин
        :type user: str
        :param passwd: пароль
        :type passwd: str
        :type pl: sandbox.sdk2.helpers.ProcessLog
        :return: output
        :rtype: str
        """
        cmd = [DOCKER, 'login']

        if user:
            cmd += ['-u%s' % user]

        if passwd:
            cmd += ['-p%s' % passwd]

        cmd += [host]

        return exec_cmd(cmd, pl)

    @staticmethod
    def ps(show_all=False, latest=False, fmt=None, pl=None):
        """
        docker ps

        :param show_all: все контейнеры (включая остановленные)
        :type show_all: bool
        :param latest: latest created containers
        :type latest: bool
        :param fmt: формат вывода
        :type fmt: str
        :type pl: sandbox.sdk2.helpers.ProcessLog
        :return: вывод docker ps
        :rtype: str
        """
        cmd = [DOCKER, 'ps']

        if show_all:
            cmd += ['-a']

        if latest:
            cmd += ['-l']

        if fmt:
            cmd += ['--format', fmt]

        return call_cmd(cmd, pl=pl, grab_stderr=True)

    @staticmethod
    def processes(show_all=False, latest=False, pl=None):
        """
        Список процессов docker.

        :param show_all: все контейнеры (включая остановленные)
        :type show_all: bool
        :param latest: show the latest created container (includes all states)
        :type latest: bool
        :type pl: sandbox.sdk2.helpers.ProcessLog
        :return: названия процессов
        :rtype: list
        """
        processes = []

        out = Docker.ps(fmt='{{.Names}}', show_all=show_all, latest=latest, pl=pl).strip()
        if out:
            processes = filter(
                lambda p: bool(p),
                re.split(r'[\s\n\t]+', out)
            )

        return processes

    @staticmethod
    def logs(process, pl=None):
        """
        docker logs

        :param process: название процесса
        :type process: str
        :type pl: sandbox.sdk2.helpers.ProcessLog
        :rtype: str
        """

        return call_cmd([DOCKER, 'logs', process], pl=pl, grab_stderr=True)

    @staticmethod
    def dump_logs(process, file_path, pl=None):
        """
        Дампит логи процесса docker в файл.

        :param process: название процесса
        :type process: str
        :param file_path: путь файла лога
        :type file_path: str
        :type pl: sandbox.sdk2.helpers.ProcessLog
        """

        StringTemplate(
            Docker.logs(process, pl=pl)
        ).save(
            file_path
        )

    @staticmethod
    def dump_all_logs(dir_path, show_all=False, latest=False, pl=None):
        """
        Дампит логи всех процессов docker в файлы директории

        :param show_all: все контейнеры (включая остановленные)
        :type show_all: bool
        :param latest: show the latest created container (includes all states)
        :type latest: bool
        :param dir_path: директория логов (в ней создается файл <process>.logs.txt)
        :type dir_path: str
        :type pl: sandbox.sdk2.helpers.ProcessLog
        """

        for process in Docker.processes(show_all=show_all, latest=latest, pl=pl):
            service = re.sub(r'^.+?_(.+?)_1$', r'\g<1>', process)
            try:
                Docker.dump_logs(
                    process,
                    os.path.join(dir_path, 'docker-%s.log' % service),
                    pl=pl
                )
            except Exception as ex:
                logging.error(ex)

    @staticmethod
    def execute(process, args, workdir=None, detach=None, pl=None):
        """
        Выполнение команды в контейнере.

        :param process: идентификатор процесса
        :type process: str
        :param args: аргументы exec
        :type args: list
        :param workdir: рабочий каталог
        :type workdir: str
        :param detach: фоновый процесс
        :type detach: bool
        :type pl: sandbox.sdk2.helpers.ProcessLog
        :return: вывод команды
        :rtype: str
        """
        cmd = [DOCKER, 'exec']

        if workdir:
            cmd += ['-w', workdir]

        if detach:
            cmd += ['-d']

        cmd += [process] + list(args)

        return exec_cmd(cmd, pl)

    @staticmethod
    def cp(src, dst, archive=False, follow=False, pl=None):
        """
        Копирование файлов.

        :param src: source
        :type src: str
        :param dst: destination
        :type dst: str
        :param archive: copy all perms
        :type archive: bool
        :param follow: follow symlinks
        :type follow: bool
        :type pl: sandbox.sdk2.helpers.ProcessLog
        :return: output
        :rtype: str
        """
        cmd = [DOCKER, 'cp']

        if archive:
            cmd += ['-a']

        if follow:
            cmd += ['-L']

        cmd += [src, dst]

        return exec_cmd(cmd, pl)

    @staticmethod
    def kill(processes, pl=None):
        """
        Убивает процесс.

        :param processes: процесс или процессы
        :type processes: list|str
        :type pl: sandbox.sdk2.helpers.ProcessLog
        """
        cmd = [DOCKER, 'kill']
        cmd += processes if type(processes) == list else [processes]

        exec_cmd(cmd, pl)

    @staticmethod
    def kill_all(pl=None):
        """
        Убивает все процессы.

        :type pl: sandbox.sdk2.helpers.ProcessLog
        """

        processes = Docker.processes(pl=pl)
        if processes:
            logging.debug('Останавливаем docker-процессы ...')
            try:
                Docker.kill(processes, pl=pl)
            except Exception as ex:
                logging.warning(ex)

    @staticmethod
    def rm(processes, force=False, link=None, volumes=False, pl=None):
        """
        Удаляет контейнеры.

        :param processes: контейнер или контейнеры
        :type processes: list|str
        :param force: не только остановленные, но и работающие
        :type force: bool
        :param link: --link
        :type link: str
        :param volumes: --volumes
        :type volumes: bool
        :type pl: sandbox.sdk2.helpers.ProcessLog
        """
        cmd = [DOCKER, 'rm']

        if force:
            cmd += ['-f']

        if link:
            cmd += ['-l', link]

        if volumes:
            cmd += ['-v']

        cmd += processes if type(processes) == list else [processes]

        exec_cmd(cmd, pl=pl)

    @staticmethod
    def rm_all(force=False, link=None, volumes=False, pl=None):
        """
        Удаляет все контейнеры.

        :param force: не только остановленные, но и работающие
        :type force: bool
        :param link: --link
        :type link: str
        :param volumes: --volumes
        :type volumes: bool
        :type pl: sandbox.sdk2.helpers.ProcessLog
        """

        processes = Docker.processes(show_all=force, pl=pl)
        if processes:
            logging.debug('Удаляем docker-контейнеры ...')
            try:
                Docker.rm(processes, force=force, link=link, volumes=volumes, pl=pl)
            except Exception as ex:
                logging.warning(ex)
