# -*- coding: utf-8 -*-
import os.path

from template import StringTemplate
from . import call_cmd, exec_cmd

DOCKER_COMPOSE = 'docker-compose'


class DockerCompose(object):
    @staticmethod
    def pull(yml_file=None, pl=None):
        """
        docker-compose pull

        :param yml_file: путь docker-compose.yml
        :type yml_file: str
        :type pl: sandbox.sdk2.helpers.ProcessLog
        :return: вывод команды
        :rtype: str
        """
        cmd = [DOCKER_COMPOSE]

        if yml_file:
            cmd += ['-f', yml_file]

        cmd += ['pull', '-q']

        return exec_cmd(cmd, pl)

    @staticmethod
    def build(yml_file=None, pull=False, pl=None):
        """
        docker-compose build

        :param yml_file: путь файла docker-compose.yml
        :type yml_file: str
        :param pull: --pull
        :type pull: bool
        :type pl: sandbox.sdk2.helpers.ProcessLog
        :return: вывод команды
        :rtype: str
        """
        cmd = [DOCKER_COMPOSE]

        if yml_file:
            cmd += ['-f', yml_file]

        cmd += ['build']

        if pull:
            cmd += ['--pull']

        return exec_cmd(cmd, pl)

    @staticmethod
    def up(yml_file=None, project=None, build=False, force_recreate=None, detach=None, pl=None):
        """
        docker-compose up

        :param yml_file: путь к docker-compose.yml
        :type yml_file: str
        :param project: название проекта (-p <project>)
        :type project: str
        :param build: --build
        :type build: bool
        :param force_recreate: --force-recreate
        :type force_recreate: bool
        :param detach: -d
        :type detach: bool
        :type pl: sandbox.sdk2.helpers.ProcessLog
        :return: вывод команды
        :rtype: str
        """
        cmd = [DOCKER_COMPOSE]

        if yml_file:
            cmd += ['-f', yml_file]

        if project:
            cmd += ['-p', project]

        cmd += ['up', '--quiet-pull']

        if build:
            cmd += ['--build']

        if force_recreate:
            cmd += ['--force-recreate']

        if detach:
            cmd += ['-d']

        return exec_cmd(cmd, pl)

    @staticmethod
    def logs(yml_file=None, project=None, pl=None):
        """
        docker-compose logs

        :param yml_file: путь docker-compose.yml
        :type yml_file: str
        :param project: название проекта docker-compose (-p <project>)
        :type project: str
        :type pl: sandbox.sdk2.helpers.ProcessLog
        :rtype: str
        """
        cmd = [DOCKER_COMPOSE]

        if yml_file:
            cmd += ['-f', yml_file]

        if project:
            cmd += ['-p', project]

        cmd += ['logs']

        return call_cmd(cmd, pl=pl, grab_stderr=True)

    @staticmethod
    def dump_logs(log_dir, yml_file=None, project=None, pl=None):
        """
        Дампит лог docker-compose в файл

        :param log_dir: директория логов (в ней создается файл docker-compose.logs.txt)
        :type log_dir: str
        :param yml_file: путь docker-compose.yml
        :type yml_file: str
        :param project: название проекта (-p <project>)
        :type project: str
        :type pl: sandbox.sdk2.helpers.ProcessLog
        """

        StringTemplate(
            DockerCompose.logs(yml_file=yml_file, project=project, pl=pl),
        ).save(
            os.path.join(log_dir, 'docker-compose.log'),
        )

    @staticmethod
    def down(yml_file=None, project=None, rmi=None, volumes=False, orphans=False, timeout=None, pl=None):
        """
        Останавливает контейнеры.

        :param yml_file: путь docker-compose.yml
        :type yml_file: str
        :param project: название проекта (-p <project>)
        :type project: str
        :param rmi --rmi type
        :type rmi: str
        :param volumes --volumes
        :type volumes: bool
        :param orphans: --remove-orphans
        :type orphans: bool
        :param timeout: --timeout
        :type timeout: int
        :type pl: sandbox.sdk2.helpers.ProcessLog
        """
        cmd = [DOCKER_COMPOSE]

        if yml_file:
            cmd += ['-f', yml_file]

        if project:
            cmd += ['-p', project]

        cmd += ['down']

        if rmi:
            cmd += ['--rmi', rmi]

        if volumes:
            cmd += ['-v']

        if orphans:
            cmd += ['--remove-orphans']

        if timeout:
            cmd += ['-t', timeout]

        return exec_cmd(cmd, pl=pl)

    @staticmethod
    def kill(yml_file=None, project=None, pl=None):
        """
        Убивает контейнеры.

        :param yml_file: путь docker-compose.yml
        :type yml_file: str
        :param project: название проекта (-p <project>)
        :type project: str
        :type pl: sandbox.sdk2.helpers.ProcessLog
        """
        cmd = [DOCKER_COMPOSE]

        if yml_file:
            cmd += ['-f', yml_file]

        if project:
            cmd += ['-p', project]

        cmd += ['kill']

        return exec_cmd(cmd, pl=pl)
