# coding: utf-8
import requests
import time
import logging
from enum import Enum


class Services(Enum):
    partner_frontend = 2033
    partner_backend = 2034


class Environments(Enum):
    production = dict([[Services.partner_frontend, 3160], [Services.partner_backend, 3163]])
    pre_production = dict([[Services.partner_frontend, 3161], [Services.partner_backend, 3164]])
    pre_stable = dict([[Services.partner_frontend, 3162], [Services.partner_backend, 3165]])


class InfraNotifications:
    def __init__(self, oauth_token, ticket, queue=None):
        if queue is None:
            queue = list()
        else:
            queue = list(int(e) for e in queue)

        self.headers = {
            'Authorization': 'OAuth %s' % oauth_token,
        }
        self.base_url = 'https://infra-api.yandex-team.ru/v1'
        self.ticket = ticket
        self.queue = queue

    DataCenters = dict([
        [Environments.production, ['man', 'sas', 'vla']],
        [Environments.pre_production, ['vla', 'iva']],
        [Environments.pre_stable, ['sas', 'vla']],
    ])

    def push_release_event(self, service, env):
        """Создает отметку начала релиза в сервисе https://infra.yandex-team.ru

        Args:
            service (Service): updated service (partner-frontend or partner-backend)
            env (Environment): updated environment (production, pre-stable, pre-production)
        """
        data_centers = dict.fromkeys(self.DataCenters[env], True)
        payload = dict(
            title='Релиз',
            description='Релиз: {}'.format(self.ticket),
            startTime=int(time.time()),
            type='maintenance',
            severity='minor',
            serviceId=service.value,
            environmentId=env.value[service],
            tickets=self.ticket
        )
        payload.update(data_centers)
        event_id = self._create_event(payload)
        if event_id is not None:
            self.queue.append(event_id)

    def resolve_all_created_events(self):
        """ Резолвит все сгенерированные события """
        while len(self.queue) != 0:
            event_id = self.queue.pop()
            self._resolve_event(event_id)

    def delete_all_created_events(self):
        """ Удаляет все ранее созданные не завершенные события """
        while len(self.queue) != 0:
            event_id = self.queue.pop()
            self._delete_event(event_id)

    def _create_event(self, payload):
        api_url = '{}/{}'.format(self.base_url, 'events')
        try:
            r = requests.post(api_url, json=payload, headers=self.headers)
            r.raise_for_status()
            return r.json()['id']
        except Exception:
            logging.exception('Exception occurred')
            return None

    def _resolve_event(self, id):
        api_url = '{}/{}/{}'.format(self.base_url, 'events', id)
        payload = dict(finishTime=int(time.time()))
        try:
            r = requests.put(api_url, json=payload, headers=self.headers)
            r.raise_for_status()
        except Exception:
            logging.exception('Exception occurred')

    def _delete_event(self, id):
        api_url = "{}/{}/{}".format(self.base_url, "events", id)
        try:
            r = requests.delete(api_url, headers=self.headers)
            r.raise_for_status()
        except Exception:
            logging.exception('Exception occurred')
