# coding=utf-8

import json
import logging
import math

import requests

from sandbox.projects.partner.settings import KRUSH_URL


class KrushLogger(object):
    """
    Сервис Krush Logger
    """

    @staticmethod
    def convert_report_json(json_report_path, record_enricher=None, totals_enricher=None):
        """
        Конвертирует данные report.json в данные krush logger

        :param json_report_path: путь файла report.json
        :type json_report_path: str
        :param record_enricher: dict или функция для добавления данных к записям: lambda record: record
        :param totals_enricher: dict или функция для добавления данных к суммарной информации totals: lambda record: record
        :return: dict(records,totals)
        :rtype: dict
        """

        records = []
        success = 0
        fail = 0
        skip = 0

        with open(json_report_path, "r") as stream:
            data = json.loads(stream.read())

        for key in data:
            message = ''

            if data[key]['status'] == 'success':
                success += 1
            else:
                message = data[key]['errorReason']['message'] if 'errorReason' in data[key] else data[key]['skipReason']

                if data[key]['status'] == 'fail':
                    fail += 1
                else:
                    skip += 1

            record = {
                'testName': key,
                'status': data[key]['status'],
                'duration': math.floor(data[key]['duration'] / 1000) if data[key]['duration'] else 0,
                'file': data[key]['file'] if 'file' in data[key] else None,
                'url': data[key]['url'] if 'url' in data[key] else None,
                'customData': data[key]['meta']['customData'] if 'meta' in data[key]
                                                                 and 'customData' in data[key]['meta'] else {},
                'message': message,
                'retries': len(data[key]['retries']) if 'retries' in data[key] else 1,
            }

            if callable(record_enricher):
                record = record_enricher(record)
            elif isinstance(record_enricher, dict):
                record.update(record_enricher)

            records.append(record)

        totals = {
            'success': success,
            'skip': skip,
            'fail': fail
        }

        if callable(totals_enricher):
            totals = totals_enricher(totals)
        elif isinstance(totals_enricher, dict):
            totals.update(totals_enricher)

        return {
            'records': records,
            'totals': totals
        }

    @staticmethod
    def send_data(initiator, records, base_url=KRUSH_URL):
        """
        Отправка данных в krush.

        :param initiator: значение initiator данных
        :type initiator: str
        :param records: список записей для отправки
        :type records: list
        :param base_url: базовый адрес сервиса Krush
        :type base_url: str
        :return: ответ сервера
        """
        url = base_url + '/logger/' + initiator
        logging.debug('Отправка данных в krush: ' + url)
        result = requests.post(url, json=records)
        if not result.ok:
            logging.error(result)

        return result
