# coding: utf-8


import os
import json
import logging
import requests
import subprocess as sp
import sandbox.common.types.task as ctt

from os import path
from sandbox.common import config
from sandbox.common.urls import get_task_link
from sandbox.projects.adfox.adfox_ui.metrics import AnalyzableTask

from sandbox.projects.partner.settings import \
    YAV_TOKEN_VAULT_NAME, \
    YAV_TOKEN_VAULT_OWNER
from sandbox.projects.partner.tasks.misc import \
    call_cmd, \
    exec_cmd, \
    is_dev_mode, \
    configure_internal_ca, \
    get_error_status_explanation

from sandbox import sdk2
from enum import Enum


def get_targets_array(type):
    targets = ['startrek']

    if type == NotificationLevels.URGENT:
        targets.extend(['slack', 'telegram'])

    return targets


class NotificationLevels(str, Enum):
    INFO = 'info'
    URGENT = 'urgent'


class Callbackable(object):
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Callback settings', description='Callback settings') as callback_settings:
            callback_url = sdk2.parameters.List(
                'Callback URL',
                description='Lifecycle notification events will be sent to the URL provided',
                default='https://krush.partner.yandex-team.ru/v3/notifications/release'
            )
            callback_params = sdk2.parameters.Dict(
                'Callback parameters',
                description='Will be added to notification payload. '
                            'Example: st_issue=PI-12345'
            )

    def get_callback_headers(self):
        return {'content-type': 'application/json'}

    def make_callback(self, payload, type=NotificationLevels.INFO):
        callback_url = self.Parameters.callback_url

        if not callback_url:
            return

        logging.debug('Make callback to %s' % callback_url)
        headers = self.get_callback_headers()

        if 'st_issue' in payload and 'ticketId' not in payload:
            payload['ticketId'] = payload['st_issue']

        # Если disableNotifications = True, то в телеграм в позднее время будут отправлены сообщения без уведомлений
        payload['disableNotifications'] = True

        # Места, в которые нужно слать уведомления
        payload['targets'] = get_targets_array(type)

        data = dict(task=self.type.name)
        data.update(self.Parameters.callback_params)
        data.update(payload)

        for url in callback_url:
            self.make_single_callback(url, headers, data)

    def make_single_callback(self, url, headers, payload):
        try:
            data = json.dumps(payload)
            logging.debug('make_single_callback %s %s' % (url, data))

            r = requests.post(
                url,
                headers=headers,
                data=data
            )
            if r.status_code / 100 == 2:
                logging.debug('Callback was made: %s' % str(r.text))
            else:
                logging.debug('Failed to make callback: HTTP %d - %s' % (r.status_code, str(r.text)))
        except sp.CalledProcessError as e:
            logging.error('Failed to make request at callback URL: %s' % str(e))
        except UnicodeDecodeError as e:
            logging.error('Failed to encode callback request payload: %s' % str(e))
        except Exception as e:
            logging.error('Error while making callback request: %s' % str(e))

    def send_message(self, message, type=NotificationLevels.INFO):
        payload = dict(message=message)
        if hasattr(self.Parameters, 'st_issue'):
            payload['ticketId'] = self.Parameters.st_issue
        self.make_callback(payload, type)

    def send_message_with_task_url(self, message, task_id):
        task_id_list = task_id if type(task_id) == list else [task_id]
        urls = ['https://sandbox.yandex-team.ru/task/{}'.format(tid) for tid in task_id_list]
        new_message = '{}\nТаски: \n{}'.format(message, '\n'.join(urls))
        self.send_message(new_message)


class PartnerFrontTaskBaseParametersMeta(type(AnalyzableTask.Parameters), type(Callbackable.Parameters)):
    pass


class PartnerFrontTaskBase(AnalyzableTask, Callbackable):
    class Context(AnalyzableTask.Context):
        pass

    class Parameters(AnalyzableTask.Parameters, Callbackable.Parameters):
        __metaclass__ = PartnerFrontTaskBaseParametersMeta

    def on_execute(self):
        self.metrics_on_execute()
        os.environ['YAV_TOKEN'] = sdk2.Vault.data(YAV_TOKEN_VAULT_OWNER, YAV_TOKEN_VAULT_NAME)

    def get_ssh_key(self):
        return sdk2.ssh.Key(self, 'PARTNER_FRONT', 'robot-pereiro-ssh')

    def find_child_task(self, name):
        if name not in self.Context.tasks:
            return None
        return self.find(id=self.Context.tasks[name]).first()

    def install_deps(self, pl=None):
        self.load_deps('deps', pl)

    def load_deps(self, dtype, pl=None):
        """Установка зависимостей
        :param dtype: Набор зависимостей - 'deps' | 'autotests_deps'
        :param pl: Перенаправление вывода
        """

        logging.debug('PartnerFrontTaskBase.load_deps enter dtype=%s' % dtype)

        dependencies = {
            'deps': {
                'path': './',
                'resource_name': 'deps'
            },
            'autotests_deps': {
                'path': './test/autotests/',
                'resource_name': 'autotests_deps'
            }
        }
        if dtype not in dependencies:
            raise Exception(
                'Unknown type of dependency set "%s" - expected on of %s' % (
                    dtype,
                    list(dependencies.keys())
                ))
        dep = dependencies[dtype]
        deps_sha = call_cmd(['./tasks/trendbox/deps-sha.sh', '-p', dep['path']], pl)
        logging.debug('PartnerFrontTaskBase.load_deps sha=%s' % deps_sha)

        deps_resource = sdk2.Resource.find(
            type='TRENDBOX_CI_RESOURCE',
            attrs=dict(
                key=dep['resource_name'],
                sha=deps_sha,
                format='squashfs'
            ),
            state='READY'
        ).first()

        if deps_resource:
            logging.debug('PartnerFrontTaskBase.load_deps get resource')
            resource_data = sdk2.ResourceData(deps_resource)
            src_path = str(resource_data.path)
            logging.debug('PartnerFrontTaskBase.load_deps resource path %s' % src_path)
            cmd = ['unsquashfs', '-d', './node_modules', src_path]
            logging.debug('PartnerFrontTaskBase.load_deps extract command: ')
            logging.debug(cmd)
            exec_cmd(cmd, pl)
            logging.debug('PartnerFrontTaskBase.load_deps extract command done')
            return

        logging.info('Can not find existing resource - run npm ci')
        retry_count = 5
        current_retry = 1

        while current_retry <= retry_count:
            logging.info('Try to run npm ci №{}'.format(current_retry))
            result = call_cmd(['npm', 'ci', '-C', dep['path']], pl)
            if result == 0:
                return
            current_retry += 1

        raise Exception('Error during deps installation')

    def try_download_bundles(self, pl):
        """ Попытка загрузить ресурс собранной статики (если такой есть). Если успешно - True, при не успехе - False """
        bundles_sha = call_cmd(['./tasks/trendbox/bundles-sha.sh'], pl)
        bundles_resource = sdk2.Resource.find(type='TRENDBOX_CI_RESOURCE', attrs=dict(key='bundles', sha=bundles_sha, format='squashfs'), state='READY').first()
        if bundles_resource is None:
            logging.debug('No appropriate static bundles found')
            return False
        resource_data = sdk2.ResourceData(bundles_resource)
        path_to_unsquash = '{}/static/desktop.bundles'.format(os.getcwd())
        # TODO do use tar instead of squashfs
        exec_cmd(['unsquashfs', '-d', str(path_to_unsquash), str(resource_data.path)], pl)
        if path.exists('./static/desktop.bundles'):
            return True
        return False

    @property
    def error_callback_title(self):
        return 'Ошибка'

    @property
    def is_dev_mode(self):
        registry = config.Registry()
        return is_dev_mode(registry)

    def make_failure_callback(self, error_text, link=None):
        if not link:
            link = self.task_url
        message = '{}: {}\n{}'.format(
            self.error_callback_title,
            error_text,
            link
        )
        self.send_message(message, NotificationLevels.URGENT)

    def on_break(self, prev_status, status):
        logging.debug('on_break %s -> %s' % (str(prev_status), str(status)))
        self.metrics_on_break(prev_status, status)

        if status in (ctt.Status.EXCEPTION, ctt.Status.STOPPED, ctt.Status.TIMEOUT):
            error = get_error_status_explanation(status)
            self.make_failure_callback(error)

    def on_finish(self, prev_status, status):
        logging.debug('on_finish %s -> %s' % (str(prev_status), str(status)))
        self.metrics_on_finish(prev_status, status)

        if status == ctt.Status.FAILURE:
            error = get_error_status_explanation(status)
            self.make_failure_callback(error)

    def on_prepare(self):
        configure_internal_ca()
        # монтировать аркадию тут

    def dry_run(self):
        """
        Тестовый запуск таски, переопределяется в конкретном классе
        """
        logging.debug('PartnerFrontBase.dry_run')

    def real_run(self):
        """
        Боевой запуск таски, переопределяется в конкретном классе
        """
        logging.debug('PartnerFrontBase.real_run')

    def run_env_aware(self):
        """
        Выполниться в зависимости от окружения в тестовом либо реальном режиме
        """
        logging.debug('PartnerFrontBase.run_env_aware')
        if self.is_dev_mode:
            self.dry_run()
        else:
            self.real_run()

    def run_subtask_env_aware(self, task):
        """
        Выполнить запуск подзадачи в зависимости от окружения
        В продовом режиме задача ставится в очередь.
        В тестовом - сохраняется как черновик.
        :param task: sandbox.sdk2.Task  Объект подзадачи
        :return: sandbox.sdk2.Task
        """
        logging.debug('PartnerFrontBase.run_subtask_env_aware')
        if self.is_dev_mode:
            logging.debug('PartnerFrontBase.run_subtask_env_aware dry run')
            task.description += '\nDry run - left in DRAFT status'
            task.save()
        else:
            task.enqueue()

        return task

    @property
    def task_url(self):
        return get_task_link(self.id) + '/view'
