# -*- coding: utf-8 -*-

import logging
import os


class Template(object):
    """
    Базовый (абстрактный) шаблон.
    Helper для работы с ресурсами sandbox-таски.
    """
    _text = ''

    def __init__(self, text=None, replace=None):
        """
        :param text: данные
        :type text: str
        :param replace: значения для замены
        :type replace: dict
        """
        if text:
            self._text = text

            if replace:
                self.replace(replace)

    @property
    def text(self):
        """
        :return: текст
        :rtype: str
        """
        return self._text

    def set(self, text):
        """
        Устанавливает новое содержимое

        :param text: текст ресурса
        :type text: str
        :rtype: Template
        """
        self._text = text
        return self

    @staticmethod
    def replace_placeholders(text, values):
        """
        :param text: текст с шаблонами
        :type text: str
        :param values: элементы для замены
        :type values: dict
        :rtype: str
        """
        old_text = None
        while text != old_text:
            old_text = text
            for key in values:
                text = text.replace(key, str(values[key]))

        return text

    def replace(self, values):
        """
        Замена шаблонов текста

        :param values: значения для замены
        :type values: dict
        :rtype: Template
        """
        self._text = Template.replace_placeholders(self._text, values)

        return self

    def clone(self, count, values_generator=None):
        """
        Клонирует текст заданное количество раз

        :param count: кол-во копий
        :type count: int
        :param values_generator: генератор словаря замен
        :type values_generator: function(instance -> int) -> dict
        :rtype: Template
        """
        content = ''
        for i in range(0, count):
            content += Template.replace_placeholders(self._text, values_generator(i)) \
                if values_generator else self._text
        self._text = content

        return self

    def debug(self, name=None):
        """
        Выводит в логи название шаблона и текст

        :param name: название шаблона
        :type name: str
        :rtype: Template
        """
        if name:
            logging.debug('=== ' + (name if name else '') + ' ===')

        logging.debug(self.text)

        return self

    @staticmethod
    def prepare_dir(file_path):
        """
        Подготовка директории для файла

        :param file_path: путь файла
        :type file_path: str
        :return file_path
        :rtype: str
        """
        directory = os.path.dirname(file_path)
        if not os.path.exists(directory):
            os.makedirs(directory)
        return file_path

    @staticmethod
    def write(file_path, text):
        """
        Сохранение содержимого в файл

        :param file_path: путь файла
        :type file_path: str
        :param text: содержимое для записи
        :type text: str
        """
        with open(Template.prepare_dir(file_path), mode='w') as f:
            f.write(str(text))

    def save(self, path):
        """
        Сохранение текста в файл

        :param path: путь файла
        :type path: str
        :rtype: Template
        """
        Template.write(path, self._text)

        return self

    def __str__(self):
        """
        Конвертирует в строку

        :return: контент ресурса
        :rtype: str
        """
        return self._text


class StringTemplate(Template):
    """
    Ресурс из строки
    """
    pass


class FileTemplate(Template):
    """
    Ресурс из локального файла.
    """
    _file_path = None

    def __init__(self, file_path, replace=None):
        """
        :param file_path: путь файла
        :type file_path: str
        :param replace: значения для замены
        :type replace: dict
        """
        self._file_path = file_path
        with open(file_path, 'r') as f:
            content = f.read()

        super(FileTemplate, self).__init__(text=content, replace=replace)

    def save(self, path=None):
        """
        Сохранить текст в файл.

        :param path: путь файла
        :type path: str
        :return: FileTemplate
        """
        if not path:
            path = self._file_path

        return super(FileTemplate, self).save(path=path)

    def debug(self, name=None):
        """
        Выводит в логи название шаблона и текст

        :param name: название
        :type name: str
        :rtype: FileTemplate
        """
        if not name:
            name = self._file_path

        return super(FileTemplate, self).debug(name=name)


class ResourceTemplate(Template):
    """
    Ресурс sandbox.
    """
    _name = None

    def __init__(self, name, replace=None):
        """
        :param name: путь ресурса
        :type name: str
        :param replace: значения для замены
        :type replace: dict
        """
        self._name = name
        import library.python.resource as resource
        content = resource.find(name)
        super(ResourceTemplate, self).__init__(text=content, replace=replace)

    def save(self, path=None):
        """
        Сохранить ресурс в файл.

        :param path: путь файла
        :type path: str
        :return: SandboxResource
        """
        if not path:
            path = os.path.join(
                os.getcwd(),
                self._name.lstrip('.').strip('/')
            )

        return super(ResourceTemplate, self).save(path=path)

    def debug(self, name=None):
        """
        Выводит в логи название ресурса и контент

        :param name: название ресурса
        :type name: str
        :rtype: ResourceTemplate
        """
        if not name:
            name = self._name

        return super(ResourceTemplate, self).debug(name)
