# -*- coding: utf-8 -*-
import logging
import os
import yaml
import json

from sandbox import sdk2
from sandbox.projects.partner.settings import ROBOT_PARTNER_SECRET
from sandbox.projects.partner.utils.arc import Arc
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.types import misc as ctm
from sandbox.common.types import notification as ctn


class PartnerCheckDeploySpecs(sdk2.Task):
    name = 'PARTNER_CHECK_DEPLOY_SPECS'

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        arc_token = sdk2.parameters.YavSecret(
            'Arc OAuth token', description='Use arc to create branch',
            default='{}#arc_token'.format(ROBOT_PARTNER_SECRET),
            required=True
        )
        ya_token = sdk2.parameters.YavSecret(
            'Ya token', default='{}#ya_token'.format(ROBOT_PARTNER_SECRET),
            required=True
        )
        dctl_yp_token = sdk2.parameters.YavSecret(
            'Dctl_yp token', default='{}#dctl_yp_token'.format(ROBOT_PARTNER_SECRET),
            required=True
        )
        stage = sdk2.parameters.String(
            'Stage',
            choices=[
                ('Production', 'production'),
                ('Preprod', 'preprod'),
                ('Test', 'test')
            ],
            default='production',
            required=True
        )
        with sdk2.parameters.Output():
            is_equal = sdk2.parameters.Bool('Is equal')
            result = sdk2.parameters.String('Result')

    def _execute(self, command, cwd=os.getcwd(), shell=False, env=None):
        logging.debug('Try to call command: %s in path %s' % (str(command), os.getcwd()))
        subprocess = sp.Popen(command, cwd=cwd, stdout=sp.PIPE, stderr=sp.PIPE, shell=shell, env=env)
        result, err = subprocess.communicate()

        logging.info('result')
        logging.info(result)
        logging.error('err')
        logging.error(err)
        if subprocess.returncode != 0:
            raise Exception('{}\nReturn code: {}'.format(err, subprocess.returncode))

        return result, err

    def send_to_juggler(self, is_equal, stage):
        logging.debug('Отправляем эвент в juggler')
        if is_equal:
            info = "Спеки деплоя равны: https://sandbox.yandex-team.ru/task/%s" % self.id
            status = ctn.JugglerStatus.OK
        else:
            info = "Спеки деплоя разъехались, нужно синкнуть: https://sandbox.yandex-team.ru/task/%s" % self.id
            status = ctn.JugglerStatus.CRIT

        if stage == 'production':
            stage = 'prod'

        self.server.notification(
            body=info,
            recipients=["host=partner-{}&service=partner_check_deploy_specs".format(stage)],
            transport=ctn.Transport.JUGGLER,
            check_status=status
        )

    def ordered(self, obj):
        if isinstance(obj, dict):
            return sorted((k, self.ordered(v)) for k, v in obj.items())
        if isinstance(obj, list):
            return sorted(self.ordered(x) for x in obj)
        else:
            return obj

    def get_list_with_placeholders(self, stage):
        placeholders_list = {
            'annotations': [
                'stagectl_epoch',
                {
                    'stagectl_vcs': [
                        'svn_branch',
                        'svn_revision',
                        'svn_tag'
                    ]
                }
            ],
            'labels': [
                'notifications_last_timestamp',
                {
                    'du_sidecar_target_revision': {
                        'Backend':
                            [
                                'logbrokerToolsLayer',
                                'podBin'
                            ],
                        'Crons':
                            [
                                'logbrokerToolsLayer',
                                'podBin'
                            ],
                        'Hourglass':
                            [
                                'logbrokerToolsLayer',
                                'podBin'
                            ]
                    }
                }
            ],
            'spec': [
                {
                    'deploy_units': {
                        'Backend': [
                            {
                                'images_for_boxes': {
                                    'FrontendNode': 'tag',
                                    'JavaAppINTAPI': 'tag',
                                    'JavaAppJSONAPI': 'tag',
                                    'PerlApp': 'tag',
                                }
                            },
                            {
                                'pod_agent_sandbox_info': 'revision'
                            },
                            'revision'
                        ],
                        'Crons': [
                            {
                                'images_for_boxes': {
                                    'CronApp': 'tag'
                                }
                            },
                            {
                                'pod_agent_sandbox_info': 'revision'
                            },
                            'revision'
                        ],
                        'Hourglass': [
                            {
                                'images_for_boxes': {
                                    'Haproxy': 'tag',
                                    'JavaAppHourglass': 'tag'
                                }
                            },
                            {
                                'pod_agent_sandbox_info': 'revision'
                            },
                            'revision'
                        ]
                    }
                },
                'revision',
                {
                    'revision_info': 'description'
                }
            ]
        }

        if stage == 'preprod':
            placeholders_list['labels'][1]['du_sidecar_target_revision']['Database'] = 'podBin'
            placeholders_list['spec'][0]['deploy_units']['Database'] = [
                {
                    'images_for_boxes': {
                        'MySQL': 'tag',
                    }
                },
                {
                    'pod_agent_sandbox_info': 'revision'
                },
                'revision'
            ]

        return placeholders_list

    def insert_placeholders(self, placeholders, data):
        if isinstance(placeholders, dict):
            for key, value in placeholders.items():
                self.insert_placeholders(value, data[key])
        elif isinstance(placeholders, list):
            for item in placeholders:
                self.insert_placeholders(item, data)
        else:
            data[placeholders] = '%s'

    def add_placeholders(self, deploy_config, remote_deploy_config, stage):
        placeholders = self.get_list_with_placeholders(stage)
        for item in (deploy_config, remote_deploy_config):
            self.insert_placeholders(placeholders, item)

            workloads = (item['spec']['deploy_units']['Backend']['multi_cluster_replica_set']
                             ['replica_set']['pod_template_spec']['spec']
                             ['pod_agent_payload']['spec']['workloads'])

            for workload in workloads:
                if workload['box_ref'] == 'FrontendNode':
                    envs = workload['env']
                    for env in envs:
                        if env['name'] == 'ADFOX_HOST':
                            env['value']['literal_env']['value'] = '%s'

    def on_execute(self):
        logging.info('Инициализируем арк')
        arc = Arc(path='.', token=self.Parameters.arc_token.value())
        stage = self.Parameters.stage

        logging.info('Проставляем переменные окружения')
        my_env = os.environ.copy()
        my_env['DCTL_YP_TOKEN'] = self.Parameters.dctl_yp_token.value()
        my_env['YA_TOKEN'] = self.Parameters.ya_token.value()

        try:
            root = str(sdk2.task.Task.current.path())

            logging.info('Получаем конфиг из деплоя')
            result, err = self._execute(
                ['{}/ya'.format(arc.path), 'tool', 'dctl', 'get', 'stage', 'partner-{}-stage'.format(stage)],
                env=my_env
            )
            remote_deploy_stage_data = yaml.safe_load(result)
            logging.info(remote_deploy_stage_data)

            logging.info('Переходим в папку с конфигами')
            src_path = os.path.join(root, 'arc', 'arcadia', 'partner', 'perl', 'partner2', 'configs', stage,
                                    'deploy-stage.json')

            logging.info('Получаем продакшн  конфиг, лежащий в проекте')
            with open(src_path, "r") as file:
                deploy_stage_data = json.load(file)
            logging.info(deploy_stage_data)

            logging.info('Проставляем в обоих конфигах плейсхолдеры')
            self.add_placeholders(deploy_stage_data, remote_deploy_stage_data, stage)

            logging.info('Сравниваем конфиги')
            is_equal = self.ordered(deploy_stage_data) == self.ordered(remote_deploy_stage_data)
            if is_equal:
                logging.info('Конфиги равны')
                self.Parameters.is_equal = True
                self.Parameters.result = 'Configs are equal'
                self.send_to_juggler(True, stage)
            else:
                logging.info('Конфиги не равны')
                self.Parameters.is_equal = False
                self.Parameters.result = 'Configs are not equal'
                self.send_to_juggler(False, stage)
        except Exception as ex:
            self.send_to_juggler(False, stage)
            logging.info(ex)
            raise ex
        finally:
            arc.finish()
