# coding: utf-8

import logging
import os
import time
import yaml

import sandbox.sdk2 as sdk2
import sandbox.sdk2.helpers
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import DnsType
from sandbox.projects.common.arcadia import sdk
from sandbox.sandboxsdk import paths
from sandbox.sdk2 import yav

from sandbox.projects.partner.settings import STAGE_PREPROD_FIXED_PI, ROBOT_PARTNER_SECRET

DEPLOY_TIMEOUT = 10

STAGE_DICT = {
    "PREPROD": {
        "stage_name": STAGE_PREPROD_FIXED_PI,
        "deploy_units": {
            "Database": {
                "box_name": "MySQL"
            }
        }
    }
}


class RedeployStage(sdk2.Task):
    """
    Обновление стейджа партнерского интерфейса на текущую версию со сбросом базы данных
    """

    name = "PARTNER_REDEPLOY_STAGE"

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        description = "Redeploy stage with database clearing"
        stage = sdk2.parameters.String("Stage name", choices=[(s, s) for s in STAGE_DICT.keys()],
                                       required=True, default='PREPROD')
        with stage.value['PREPROD']:
            deploy_unit = sdk2.parameters.String('Deploy unit',
                                                 choices=[(s, s) for s in STAGE_DICT['PREPROD']['deploy_units'].keys()],
                                                 required=True, default='Database')

    def exec_command(self, command, logname='exec_log', cwd=None):
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logname) as process_log:
            return sandbox.sdk2.helpers.subprocess.Popen(
                ('bash', '-c', command),
                stdout=process_log.stdout,
                stderr=process_log.stdout,
                cwd=cwd
            ).wait()

    def init_env_variables(self):
        secret = yav.Secret(ROBOT_PARTNER_SECRET)
        data = secret.data()['ya_token']
        os.environ['YA_TOKEN'] = data

    def get_preprod_boxes_spec(self, yaml_spec):
        logging.info('Run get_preprod_boxes_spec')
        deploy_unit = yaml_spec['spec']['deploy_units']['Database']
        return deploy_unit['replica_set']['replica_set_template']['pod_template_spec']['spec']['pod_agent_payload']['spec']['boxes']

    def get_arcadia(self):
        logging.info('Mount arcadia')
        return sdk.mount_arc_path("arcadia-arc:/#trunk")

    def get_deploy_spec(self, arcadia, stage_name):
        logging.info('Run ya tool dctl get stage')
        with arcadia as arcadia_path:
            status = self.exec_command(
                command='{path}ya tool dctl get stage {name}'.format(
                    path=arcadia_path,
                    name=stage_name
                ),
                logname='dctl_get_stage'
            )

        if status != 0:
            raise TaskFailure('Failed to run ya tool dctl get')

        output_path = os.path.join(str(paths.get_logs_folder()), 'dctl_get_stage.out.log')
        with open(output_path) as f:
            return yaml.safe_load(f)

    def put_deploy_spec(self, arcadia, deploy_spec):
        logging.info('Dump yaml spec to tmp file')
        with open('/tmp/stage_out.yaml', 'w') as out_file:
            yaml.dump(deploy_spec, out_file, default_flow_style=False)

        logging.info('Run ya tool dctl put stage')
        with arcadia as arcadia_path:
            status = self.exec_command(
                command='{path}ya tool dctl put stage {name}'.format(
                    path=arcadia_path,
                    name='/tmp/stage_out.yaml'
                ),
                logname='dctl_put_stage'
            )

            if status != 0:
                raise TaskFailure('Failed to run ya tool dctl put')

    def get_deploy_status(self, arcadia_path, stage_name, count):
        status = self.exec_command(
            command='{path}ya tool dctl status stage {name}'.format(
                path=arcadia_path,
                name=stage_name
            ),
            logname='dctl_status_stage'
        )

        if status != 0:
            raise TaskFailure('Failed to run ya tool dctl status')

        output = os.path.join(
            str(paths.get_logs_folder()),
            'dctl_status_stage.out.{}log'.format('' if count == 0 else str(count) + ".")
        )

        with open(output) as f:
            input_data = [l for l in f.readlines() if '|' in l][1:]
            statuses = [l.split('|')[4].strip() == 'Ready' for l in input_data]
            return all(statuses)

    def wait_deploy(self, arcadia, stage_name):
        with arcadia as arcadia_path:
            for i in range(DEPLOY_TIMEOUT):
                stage_ready = self.get_deploy_status(arcadia_path, stage_name, i)

                if stage_ready:
                    break

                time.sleep(60)
            else:
                raise TaskFailure('Failed to update deploy stage')

    def update_redeploy_flag_for_box(self, boxes_spec, redeploy_box_name):
        logging.info('Update redeploy flag')
        for box in boxes_spec:
            if box['id'] == redeploy_box_name:
                logging.info('Found matching box name: %s', redeploy_box_name)
                if 'env' not in box:
                    logging.info('Creating env section')
                    box['env'] = [{'name': 'REDEPLOY_FLAG', 'value': {'literal_env': {'value': '1'}}}]
                box_envs = box['env']
                for env in box_envs:
                    if env['name'] == 'REDEPLOY_FLAG':
                        value = env['value']['literal_env']['value']
                        env['value']['literal_env']['value'] = '1' if value == '0' else '0'
                break

    def on_execute(self):
        with self.memoize_stage.main_task:
            self.init_env_variables()

            stage = STAGE_DICT[self.Parameters.stage]
            stage_name = stage['stage_name']
            deploy_unit = stage['deploy_units'][self.Parameters.deploy_unit]
            box_name = deploy_unit['box_name']

            arcadia = self.get_arcadia()
            deploy_spec = self.get_deploy_spec(arcadia, stage_name)
            boxes = self.get_preprod_boxes_spec(deploy_spec)
            self.update_redeploy_flag_for_box(boxes, box_name)

            self.put_deploy_spec(arcadia, deploy_spec)
            self.wait_deploy(arcadia, stage_name)
