# coding: utf-8

import logging
import os
import subprocess as sp

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox import sdk2

from sandbox.projects.common.environments import SandboxNodeNpmEnvironment

from sandbox.projects.partner.settings import \
    ROBOT_PARTNER_SECRET

from sandbox.projects.partner.tasks.e2e_tests.misc import HermioneReportLayer

from sandbox.projects.partner.tasks.misc.partner_front_task_base import \
    PartnerFrontTaskBase

from sandbox.projects.partner.utils.arc import Arc
from sandbox.projects.partner.utils.autotests import send_tests_result_message

from sandbox.projects.partner.tasks.misc.arc import validate_branch

from sandbox.projects.partner.tasks.misc import exec_cmd

REFERENCE_STAND = 'https://screen-ref.beta.partner.yandex.ru'
TESTING_STAND = 'https://screen-test.beta.partner.yandex.ru'

RESULT_MESSAGE_ENTITY_NAME = 'в скриншутере'


def run_shooter(pl, stand, ref_stand, limit):
    logging.debug('run_shooter')
    try:
        cmd = [
            'npm',
            'run',
            'test:screenshooter',
            '--',
            '--ref-stand', ref_stand,
            '--stand', stand,
        ]
        if limit:
            cmd = cmd + ['--limit', str(limit)]
        exec_cmd(cmd, pl)
    except sp.CalledProcessError as cpe:
        if cpe.returncode == 1:
            return False
        raise cpe
    return True


class Screenshooter(HermioneReportLayer):
    """
    Run screenshooter to compare old and new builds
    """

    name = 'PARTNER_SCREENSHOOTER'

    class Context(sdk2.Task.Context):
        json_report_filepath = None
        task_status = 'UNKNOWN'

    class Parameters(PartnerFrontTaskBase.Parameters):

        arc_ref = sdk2.parameters.String(
            'Arc ref',
            description='Branch, tar or commit hash at project\'s arc repository. '
            'Source of shooting script code, url list, and reference images',
            default='trunk',
            required=True
        )

        reference_stand = sdk2.parameters.String(
            'Reference build stand',
            description='Reference stand (current production build)',
            default=REFERENCE_STAND
        )

        testing_stand = sdk2.parameters.String(
            'Build testing stand',
            description='Stand with build to be tested',
            default=TESTING_STAND
        )

        limit = sdk2.parameters.Integer(
            'Limit number of checks',
            description='Use to quickly test the state of infrastructure',
            default=0
        )

        st_issue = sdk2.parameters.String(
            'Release ticket',
            description='PI-NNNNN',
            required=True
        )

        with sdk2.parameters.Group('Secrets'):
            arc_token = sdk2.parameters.YavSecret(
                'Arc OAuth token', description='Use arc to create release branch', default=ROBOT_PARTNER_SECRET
            )

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_XENIAL
        # https://sandbox.yandex-team.ru/resource/2612407195/view
        environments = [SandboxNodeNpmEnvironment('14.18.2')]

    def get_arc_token(self):
        logging.debug('getting arc token')
        if not self.Parameters.arc_token:
            raise Exception('Arc access token is missed')

        token = self.Parameters.arc_token.data()['arc_token']
        logging.debug('success getting arc token')
        return token

    def on_prepare(self):
        logging.debug('on_prepare')
        super(Screenshooter, self).on_prepare()
        self.prepare_persistent_dir()

        if not validate_branch(self.Parameters.arc_ref):
            raise Exception('Invalide branch name')

    def prepare_persistent_dir(self):
        logging.debug('prepare_persistent_dir')
        persistent_dir = os.path.realpath('screenshooter')
        os.mkdir(persistent_dir)
        self.Context.persistent_dir = persistent_dir
        self.Context.save()

    def on_execute(self):
        super(Screenshooter, self).on_execute()
        self.make_start_callback()
        branch = self.Parameters.arc_ref
        self.arc = Arc(path='.', token=self.get_arc_token())
        self.arc.checkout(branch)
        os.chdir('{}/adv/frontend/services/yharnam/test/autotests'.format(self.arc.path))
        logging.info('Current path is {}'.format(os.getcwd()))

        with sdk2.helpers.ProcessLog('on_execute') as pl:
            # self.load_deps('autotests_deps', pl)
            exec_cmd(['npm', 'ci'], pl)
            is_error = True
            try:
                stand = self.Parameters.testing_stand
                ref_stand = self.Parameters.reference_stand
                limit = self.Parameters.limit
                success = run_shooter(pl, stand, ref_stand, limit)
                self.Context.task_status = 'SUCCESS' if success else 'FAILURE'
                is_error = not success
            except Exception as err:
                self.Context.task_status = 'EXCEPTION'
                raise err
            finally:
                self.Context.save()
                self.store_report_data(self.Context.persistent_dir, pl)
                self.make_report(
                    branch=branch,
                    type='screenshooter'
                )
                send_tests_result_message(
                    self.send_message,
                    RESULT_MESSAGE_ENTITY_NAME,
                    self.Context.report_data or {},
                    self.id,
                    is_error
                )

    def store_report_data(self, persistent_dir, pl):
        report_path = os.path.join(os.getcwd(), 'hermione/shooter-reports/commonReport')
        if not os.path.exists(report_path):
            logging.error(
                'store_report_data report data directory "{}" does not exist'.format(
                    report_path
                ))
            return

        self.Context.json_report_filepath = os.path.join(os.getcwd(), 'hermione/shooter-reports/jsonReport.json')
        self.Context.save()

        cmd = ['rsync', '-ar', report_path + '/', persistent_dir + '/']
        exec_cmd(cmd, pl)

    def make_start_callback(self):
        message = 'Скриншот-тестирование началось. {}'.format(self.task_url)
        self.send_message(message)

    @property
    def error_callback_title(self):
        return 'Ошибка при выполнении скриншот-тестирования'

    @property
    def report_description(self):
        return 'Screenshooter report'

    @property
    def report_dir_path(self):
        return self.Context.persistent_dir

    @property
    def report_message_title(self):
        return 'Проверка скриншутером завершена'

    @property
    def template_dir_path(self):
        return os.path.dirname(os.path.abspath(__file__))
