# coding: utf-8

import logging
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.projects.partner.tasks.e2e_tests.misc import \
    configure_semaphore_by_name, \
    PartnerFrontTaskBase
from sandbox.projects.partner.tasks.deploy_release import DeployRelease
from sandbox.projects.partner.tasks.screenshooter import Screenshooter
from sandbox.projects.partner.settings import \
    SCREENSHOOTER_BETA, \
    SCREENSHOOTER_REFERENCE_STAND, \
    SCREENSHOOTER_TESTING_STAND, \
    SCREENSHOOTER_TESTING_SEMAPHORE


class ScreenshooterDeploy(PartnerFrontTaskBase):
    """
    Run screenshooter_deploy to deploy screenshooter stage
    """
    name="PARTNER_SCREENSHOOTER_DEPLOY"

    class Context(sdk2.Task.Context):
        screenshooter_task_id = 0
        screenshooter_stage_deploy_task_id = 0

    class Parameters(PartnerFrontTaskBase.Parameters):

        arc_ref = sdk2.parameters.String(
            'Arc ref',
            description='Branch, tar or commit hash at project\'s arc repository. '
            'Source of shooting script code, url list, and reference images',
            default='trunk',
            required=True
        )

        java_docker_images = sdk2.parameters.String(
            'Java docker images',
            description='String with java docker images, separated by ";"'
        )

        perl_tag = sdk2.parameters.String(
            "Perl package tag",
            description="(full version like a 2.18.2100)",
            required=True
        )

        frontend_tag = sdk2.parameters.String(
            "Frontend tag",
            required=True,
            description="For ex.: 0.91.0_a129cb95249184b57ba2e36b18cfe2325734c316",
        )

        st_issue = sdk2.parameters.String(
            'Release ticket',
            description='PI-NNNNN',
            required=True
        )

    def on_enqueue(self):
        super(ScreenshooterDeploy, self).on_enqueue()

        # step 1: Захватываем write-lock семафор стейджа 'https://screen-test.beta.partner.yandex.ru'
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[configure_semaphore_by_name(SCREENSHOOTER_TESTING_SEMAPHORE)],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        )

    def on_execute(self):
        super(ScreenshooterDeploy, self).on_execute()

        # запускаем деплой скриншутерного стейджа и ждём его завершения
        with self.memoize_stage.deploy_screenshooter_stand:
            self.make_deploy_start_callback()
            screenshooter_stage_deploy_task = DeployRelease(
                self,
                stage=SCREENSHOOTER_BETA,
                st_issue=self.Parameters.st_issue,
                frontend_tag=self.Parameters.frontend_tag,
                java_docker_images=self.Parameters.java_docker_images,
                perl_tag=self.Parameters.perl_tag,
            )
            self.Context.screenshooter_stage_deploy_task_id = screenshooter_stage_deploy_task.id
            logging.debug(
                'partner_deploy_release task created: %s' % str(self.Context.screenshooter_stage_deploy_task_id)
            )
            screenshooter_stage_deploy_task.enqueue()
            raise sdk2.WaitTask(
                [self.Context.screenshooter_stage_deploy_task_id],
                [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK],
            )

        if self.Context.screenshooter_stage_deploy_task_id != 0:
            self.check_child_task_status(self.Context.screenshooter_stage_deploy_task_id)

        # запускаем само скриншутерное тестирование и ждём его завершения
        self.Context.screenshooter_stage_deploy_task_id = 0
        with self.memoize_stage.run_screenshooter:
            self.make_testing_start_callback()

            screenshooter_task = Screenshooter(
                self,
                arc_ref=self.Parameters.arc_ref,
                reference_stand=SCREENSHOOTER_REFERENCE_STAND,
                st_issue=self.Parameters.st_issue,
                testing_stand=SCREENSHOOTER_TESTING_STAND,
                callback_url=self.Parameters.callback_url,
                callback_params=self.Parameters.callback_params
            )

            self.Context.screenshooter_task_id = screenshooter_task.id
            logging.debug('screenshooter task created: %s' % str(self.Context.screenshooter_task_id))
            screenshooter_task.enqueue()

            raise sdk2.WaitTask(
                [self.Context.screenshooter_task_id],
                [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK],
            )

        if self.Context.screenshooter_task_id != 0:
            self.check_child_task_status(self.Context.screenshooter_task_id)

    def make_deploy_start_callback(self):
        message = 'Начался деплой стейджа для скриншутерного тестирования.'
        self.send_message(message)

    def make_testing_start_callback(self):
        message = 'Начался прогон скриншутерного тестирования.'
        self.send_message(message)

    def check_child_task_status(self, child_task_id):
        task = self.find(id=child_task_id, status=ctt.Status.SUCCESS).limit(1).first()
        if task is None:
            raise Exception('Дочерняя задача завершилась с ошибкой. Подробнее в задаче с ID = %s.' % child_task_id)

    @property
    def error_callback_title(self):
        return 'Ошибка при деплое скриншутерного стенда'
