# -*- coding: utf-8 -*-

import json
import logging
import os
import subprocess
import time
from datetime import datetime

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.common.types import notification as ctn

from sandbox.projects.partner.settings import ROBOT_PARTNER_SECRET
from sandbox.projects.partner.utils.arc import Arc


class TankerSync(sdk2.Task):
    suffix = datetime.now().strftime('%Y%m%d%H%M%S')

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        arc_branch = sdk2.parameters.String(
            'Arcadia branch',
            description='users/robot-partner/tanker-sync',
            default='trunk',
            required=True,
        )
        arc_token = sdk2.parameters.YavSecret(
            'Arc OAuth token', description='Use arc to create branch',
            default='{}#arc_token'.format(ROBOT_PARTNER_SECRET),
            required=True,
        )

        debug = sdk2.parameters.Bool(
            'Debug mode',
            description='pr only will be created will not be merged',
            default_value=False,
            required=True,
        )
        with sdk2.parameters.Output():
            pr_id = sdk2.parameters.String("Pull request id")
            pr_url = sdk2.parameters.String("Pull request url")

    def get_language(self):
        raise Exception('You have to override this method')

    def update_and_add_to_stage(self, arc):
        raise Exception('You have to override this method')

    def on_execute(self):
        arc_token = self.Parameters.arc_token.value()
        arc = Arc(path='.', token=arc_token)
        try:
            arc_tanker_sync_branch = 'partner-{}-tanker-sync-{}'.format(self.get_language(), self.suffix)
            logging.debug(
                'checkout branch "{}" based on "{}"'.format(arc_tanker_sync_branch, self.Parameters.arc_branch))
            arc.checkout(branch=arc_tanker_sync_branch, create_branch=True, base_branch=self.Parameters.arc_branch,
                         track_branch=False)

            self.update_and_add_to_stage(arc)

            if arc.has_files_for_commit():
                arc.commit(description='Updated translations')
                merge_command = [arc.arc_binary, 'pr', 'create', '--json', '--auto', '--publish', '--no-code-review',
                                 '--no-edit', '--push']
                result, err = self._execute(command=merge_command, path=arc.path, allow_error_code=True)
                result_json = json.loads(result)

                self.Parameters.pr_id = result_json['id']
                self.Parameters.pr_url = result_json['url']

                if self.Parameters.debug:
                    self._execute(command=[arc.arc_binary, 'pr', 'merge', '--no-auto', self.Parameters.pr_id],
                                  path=arc.path, allow_error_code=True)
                else:
                    self._execute(command=[arc.arc_binary, 'pr', 'merge', '--now', '--force', self.Parameters.pr_id],
                                  path=arc.path, allow_error_code=True)

            else:
                logging.debug('nothing to update')

            self.send_to_juggler(is_merged=True)
        except Exception:
            self.send_to_juggler(is_merged=False)
            raise
        finally:
            arc.finish()

    def _execute_with_retries(self, command, path, retries=3, delay=5):
        for i in range(1, retries + 1):
            try:
                return self._execute(command=command, path=path)
            except Exception:
                logging.debug('ignore exception. failed {} of {}'.format(i, retries))
                if i == retries:
                    raise
                time.sleep(delay)

    def _execute(self, command, path, allow_error_code=False, env=None):
        logging.debug('Execute: {}'.format(command))
        if env is None:
            env = os.environ

        p = subprocess.Popen(
            command, cwd=path, stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=env
        )
        result, err = p.communicate()
        return_code = p.returncode

        logging.debug('return_code')
        logging.debug(return_code)
        logging.debug('result')
        logging.debug(result)
        logging.debug('err')
        logging.debug(err)

        if return_code > 0 and not allow_error_code:
            raise Exception('ErrCode: {}. ErrText: {}. Failed command {}'.format(return_code, err, command))

        return result, err

    def send_to_juggler(self, is_merged):
        logging.debug('send event to juggler')
        if is_merged:
            info = "Tanker sync sandbox task was completed: https://sandbox.yandex-team.ru/task/%s" % self.id
            status = ctn.JugglerStatus.OK
        else:
            info = "Tanker sync sandbox task wasn't completed: https://sandbox.yandex-team.ru/task/%s" % self.id
            status = ctn.JugglerStatus.CRIT
        self.server.notification(
            body=info,
            recipients=["host=partner-prod&service=check_{}_tanker_sync_task".format(self.get_language())],
            transport=ctn.Transport.JUGGLER,
            check_status=status
        )
