#!/usr/bin/env python
# coding: utf-8

import json
import logging
import os
import subprocess

from sandbox import sdk2
from sandbox.projects.common.vcs import arc
from sandbox.projects.partner.utils.startrek import extract_pi_tickets_from_text


class Arc(object):
    path = None

    def __init__(self, token, path=None):
        self.path = path if path is not None else str(sdk2.task.Task.current.path())
        self.arc_binary = arc.Arc(arc_oauth_token=token).binary_path
        os.environ["ARC_TOKEN"] = token
        self._execute(['rm', '-rf', 'arc'])
        self._execute(['mkdir', '-p', 'arc/arcadia', 'arc/store'])
        self.path = self.path + '/arc/'
        self._execute([self.arc_binary, 'mount', '-m', 'arcadia/', '-S', 'store/'])
        self.path = os.path.abspath(self.path + 'arcadia/')
        os.environ["ARC_PATH"] = self.path
        self._finished = False

    def _execute(self, command, path=None):
        if not path:
            path = self.path

        if not path:
            raise Exception('Arc path is empty')

        logging.debug('Try to call command: %s in path %s' % (str(command), path))
        result, err = subprocess.Popen(
            command, cwd=path, stdout=subprocess.PIPE, stderr=subprocess.PIPE
        ).communicate()

        logging.debug('result')
        logging.debug(result)
        logging.debug('err')
        logging.debug(err)

        return result

    def checkout(self, branch, create_branch=False, base_branch=None, track_branch=True):
        # Проверка входных параметров
        #   base_branch используется только при create_branch=True
        if not create_branch and base_branch:
            raise Exception('Arc checkout called with incorrect params')

        # При переключении на существующую ветку её надо подтянуть
        if not create_branch:
            fetch_command = [self.arc_binary, 'fetch', branch]
            self._execute(fetch_command)

        # Переключение на нужную ветку или её создание на основе базовой или текущей
        checkout_command = [self.arc_binary, 'co']

        if not track_branch:
            checkout_command.append('--no-track')

        if create_branch:
            checkout_command.append('-b')
            checkout_command.append(branch)
            # Если передана базовая ветка, нужно её подтянуть и укзаать в качестве базовой
            if base_branch:
                checkout_command.append(base_branch)
                fetch_base_command = [self.arc_binary, 'fetch', base_branch]
                self._execute(fetch_base_command)
        else:
            checkout_command.append(branch)
        self._execute(checkout_command)

    def checkout_path(self, path):
        checkout_command = [self.arc_binary, 'co', path]
        self._execute(checkout_command)

    def push(self, upstream=None):
        command = [self.arc_binary, 'push']
        if upstream is not None:
            command.extend(['-u', upstream])
        self._execute(command)

    def commit(self, description):
        command = [
            self.arc_binary,
            'commit',
            '-m',
            description
        ]
        self._execute(command)

    def create_tag(self, tag, commit_hash=None):
        cmd = [self.arc_binary, 'tag', tag]
        if commit_hash:
            cmd.append(commit_hash)
        self._execute(cmd)

    def push_tag(self, tag, path):
        cmd = [self.arc_binary, 'push', '{tag}:{path}'.format(tag=tag, path=path)]
        self._execute(cmd)

    def get_tickets_since(self, vcs_object):
        self._execute([self.arc_binary, 'fetch', '--all'])
        command = [
            self.arc_binary,
            'log',
            '--format',
            '{attr.pr.tickets}',
            '{}...HEAD'.format(vcs_object),
        ]

        result = self._execute(command)
        return extract_pi_tickets_from_text(result)

    def get_head_commit_hash(self):
        cmd = [self.arc_binary, 'rev-parse', 'HEAD']
        return self._execute(cmd)

    def has_files_for_commit(self):
        result = self._execute([self.arc_binary, 'st', '--json'])
        result_json = json.loads(result)
        if 'staged' in result_json['status']:
            return len(result_json['status']['staged']) > 0
        else:
            return False

    def export(self, path, branch, arc_path):
        self._execute([self.arc_binary, 'init', '--bare'], path=path)
        self._execute([self.arc_binary, 'export', branch, arc_path, '--to', '.'], path=path)

    def finish(self):
        if self._finished:
            return
        os.chdir('%s/../' % self.path)
        self._execute([self.arc_binary, 'unmount', 'arcadia'], path='%s/../' % self.path)
        self._finished = True
