# coding=utf-8

import json
import logging

import requests


class DnsApi:
    dns_host = 'https://dns-api.yandex.net'
    api_version = 'v2.3'

    def __init__(self, user, token):
        self.user = user
        self.token = token

    def __get_dns_primitive(self, operation, host, ip, ttl=1800):
        return {'primitives': [{
            'operation': '{}'.format(operation),
            'name': '{}'.format(host),
            'type': 'AAAA',
            'data': '{}'.format(ip),
            'ttl': ttl,
        }]}

    def __get_dns_api_url(self, uri):
        return '{host}/{v}/{uri}'.format(
            host=self.dns_host,
            v=self.api_version,
            uri=uri
        )

    def __set_dns_record(self, operation, host, ip):
        data = self.__get_dns_primitive(operation, host, ip)
        oauth_token = self.token
        api_url = self.__get_dns_api_url('{}/primitives'.format(self.user))
        return requests.put(
            api_url,
            headers={'X-Auth-Token': oauth_token},
            data=json.dumps(data)
        )

    def create_dns_record(self, host, ip):
        logging.info('Создание DNS-записи: %s => %s' % (host, ip))
        response = self.__set_dns_record('add', host, ip)
        if response.status_code != 200:
            raise Exception('Error during creating DNS mock record: %s' % response.text)

    def delete_dns_record(self, host, ip):
        logging.info('Удаление DNS-записи: %s => %s' % (host, ip))
        response = self.__set_dns_record('delete', host, ip)
        if response.status_code != 200:
            raise Exception(
                'Error during deleting DNS mock record (%s): %s, remove it manually.\n' % (host, response.text),
                'Check out https://wiki.yandex-team.ru/aviaraspinfra/dns-monkey/#udalittolkoa/aaaa/ptr'
            )
