from sandbox.projects.partner.settings import \
    ROBOT_PARTNER_YAV_SECRET
import logging
import os

from sandbox.common.errors import TaskFailure
from sandbox.sdk2 import yav

ARC_MOUNT_PATH = "/arcadia/"

ARC_PI_PATH = os.path.join(
    ARC_MOUNT_PATH,
    "partner/perl/partner2/")

ANSIBLE_RELEASE_SCRIPT_PATH = os.path.join(
    ARC_MOUNT_PATH,
    "partner/infra/partner_ansible/",
    "roles/partner2_release_robot/files/script")

ROOT_PATH = "/opt/partner2_release_robot/"

WORKING_PATH = os.path.join(
    ROOT_PATH,
    "partner2")

RELEASE_SCRIPT_PATH = os.path.join(
    ROOT_PATH,
    "bin/script")


DEFAULT_ENV_TYPE = "development"


class PerlReleaseScript:
    yav_token = None
    _branch = None

    def __init__(self, executor, branch='trunk'):
        self._branch = branch
        self._exec_command = executor
        logging.info("=== Prepare environment ===")
        self.set_env_variables()
        self.init_pi_secrets()
        self.init_root_tokens()

        self.import_gpg_keys()
        self.mount_arc_repo()
        self.update_release_script()

    def get_oauth_token(self):
        if not self.yav_token:
            secret = yav.Secret(ROBOT_PARTNER_YAV_SECRET)
            data = secret.data()['oauth_token']
            self.yav_token = data
        return self.yav_token

    def init_pi_secrets(self):
        logging.info("=== Init /etc/pi-secrets.json with yav-deploy  ===")
        self._exec_mult([
            ['01_yav_deploy', 'yav-deploy', [],
                'run yav deploy when generating /etc/pi-secrets.json']
        ])

    def set_env_variables(self):
        logging.info("=== Set up ENV variables ===")
        os.environ['YENV_TYPE'] = DEFAULT_ENV_TYPE
        os.environ['YAV_TOKEN'] = self.get_oauth_token()

    def init_root_tokens(self):
        logging.info("=== Init root tokens with yav-deploy ===")
        self._exec_mult([
            ['02_robot_yav_deploy', 'yav-deploy --file robot-partner.conf',
                [], 'run yav deploy for root tokens init']
        ])

    def import_gpg_keys(self):
        logging.info("=== Import GPG keys ===")
        self._exec_mult([
            ['03_gpg_import_public', 'gpg --import /root/robot-partner-public.key',
                [], 'import gpg public key'],
            ['04_gpg_import_private', 'gpg --import /root/robot-partner-private.key',
                [], 'import gpg private key']
        ])

    def mount_arc_repo(self):
        self._exec_mult([
            ['05_mount_arc_repo', 'arc mount {}', [ARC_MOUNT_PATH], 'mount arc'],
            ['06_link_arc_repo', 'ln -s {} {}',
                [ARC_PI_PATH, WORKING_PATH], 'link arc to old path'],
        ])

    def update_release_script(self):
        self._exec_mult([
            ['07_checkout', 'arc checkout -f {}', [self._branch],
                'checkout release script branch', ARC_MOUNT_PATH],
            ['08_copy_script', 'cp {} {}', [ANSIBLE_RELEASE_SCRIPT_PATH,
                                            RELEASE_SCRIPT_PATH], 'copy release script to old location']
        ])

    def unmount_arc_repo(self):
        self._exec_mult([
            ['09_unmount_arc_repo', 'arc unmount {}', [ARC_MOUNT_PATH], 'unmount arc'],
        ])

    def _exec_mult(self, commands_list):
        for record in commands_list:
            cwd = None
            if len(record) == 5:
                log, cmd, params, descr, cwd = record
            else:
                log, cmd, params, descr = record
            status = self._exec_command(
                command=cmd.format(*params),
                logname=log,
                cwd=cwd,
            )

            if status != 0:
                raise TaskFailure('Failed to ' + descr)

    def run(self, arguments='', logname=''):
        command = '%s %s' % (RELEASE_SCRIPT_PATH, arguments)
        logging.info("=== Run release script === %s" % command)

        try:
            status = self._exec_command(
                command=command,
                logname=logname
            )
            return status
        finally:
            self.unmount_arc_repo()
