#!/usr/bin/env python
# coding: utf-8

import argparse
import logging
import sys
import re


ST_ACTIVE_RELEASES_QUERY = 'Queue: PI AND Status: new AND Type: Release "Sort By": Weight desc, Created desc'


def exec_command(command, options):
    if command == 'active-release':
        return active_release_command(options.ignore, options.oauth)
    raise Exception('Unknown command "{}"'.format(command))


def active_release_command(ignore, token):
    active_release = find_active_release_issue(ignore, token)
    if active_release:
        print(active_release)


def find_active_release_issue(ignore, token):
    client = get_st_client(token)

    query = ST_ACTIVE_RELEASES_QUERY
    issues = client.issues.find(query)

    num_active_releases = len(issues)
    if num_active_releases > 2:
        raise Exception(
            'Too many active release issues (#{})'.format(num_active_releases)
        )

    issues = map(lambda issue: issue.key, list(issues))
    if ignore:
        issues = filter(lambda issue: issue != ignore, issues)

    if len(issues) == 1:
        return issues[0]

    if len(issues) == 0:
        logging.info('No active release issues found')
        return None

    raise Exception('Too many active release issues: {}'.format(', '.join(issues)))


def get_st_client(token):
    import startrek_client
    api_version = startrek_client.settings.VERSION_V2
    useragent = 'Sandboxed'

    return startrek_client.Startrek(
        api_version=api_version,
        token=token,
        useragent=useragent
    )


def parse_args(argv):
    parser = argparse.ArgumentParser(description='Communicate with Startrek')

    parser.add_argument(
        'command',
        type=str,
        choices=['active-release']
    )

    parser.add_argument(
        '--ignore',
        type=str,
        help='Startrek issue ID to ignore in list'
    )

    parser.add_argument(
        '--oauth',
        type=str,
        required=True,
        help='OAuth token'
    )

    return parser.parse_args()


def extract_pi_tickets_from_text(text):
    logging.info("try to get some tickets from " + text)
    pattern = re.compile("(?:^|[^A-Z])(PI-[0-9]+)", re.MULTILINE)
    existing_tickets = pattern.findall(text)
    return existing_tickets

if __name__ == '__main__':
    args = parse_args(sys.argv)
    exec_command(args.command, args)
