import os
import shutil

from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk import environments, svn
from sandbox import common

import sandbox.common.types.client as ctc


CONFIG_TEMPLATE = '''
[distutils]
index-servers =
    yandex

[yandex]
repository: https://pypi.yandex-team.ru/simple/
username: {access_key}
password: {secret_key}

'''

DARWIN_TAG = (
    'macosx_10_10_x86_64'
)
DARWIN_ARM_TAG = (
    'macosx_11_0_arm64'
)
LINUX_TAG = 'manylinux1_x86_64'
WIN_TAG = 'win_amd64'

TAGS = {
    'darwin': DARWIN_TAG,
    'default-darwin-arm64': DARWIN_ARM_TAG,
    'linux': LINUX_TAG,
    'win': WIN_TAG,
}
SUPPORTED_PYTHONS = ['cp37', 'cp38', 'cp39', 'cp310']

# this MUST be a subset of SUPPORTED_PYTHONS
DARWIN_ARM_SUPPORTED_PYTHONS = ['cp38', 'cp39', 'cp310']


def to_arcadia_style(version):
    return version[2] + '.' + version[3:]


def get_library_name_for_platform(platform):
    if platform == 'win':
        return 'tvmauth_pymodule.pyd'
    else:
        return 'tvmauth_pymodule.so'


class BuildTvmAuthPypi(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_TRUSTY
        environments = (
            environments.SvnEnvironment(),
        )

    class Parameters(sdk2.Task.Parameters):

        path = sdk2.parameters.String('Arcadia path to tvmauth')
        target_platform = sdk2.parameters.String('Target platform (all, %s)' % ', '.join(TAGS.keys()))
        target_python = sdk2.parameters.String('Target python (all, %s)' % ', '.join(SUPPORTED_PYTHONS))

        vault_access_key_record = sdk2.parameters.Vault('Pypi access-key record in yav')
        vault_secret_key_record = sdk2.parameters.Vault('Pypi secret-key record in yav')

    class PutVaultDataToHome(object):
        def __init__(self, access_key, secret_key):
            self.restore_original_pypirc = False
            self.original_pypirc_backup = None
            self.config_path = os.path.expanduser('~/.pypirc')

            self.access_key = access_key
            self.secret_key = secret_key

        def __enter__(self):
            if os.path.exists(self.config_path):
                self.restore_original_pypirc = True
                self.original_pypirc_backup = self.config_path + '_backup'
                os.rename(self.config_path, self.original_pypirc_backup)

            with open(self.config_path, 'w') as f:
                f.write(
                    CONFIG_TEMPLATE.format(
                        access_key=self.access_key,
                        secret_key=self.secret_key
                    )
                )

        def __exit__(self, *args):
            os.remove(self.config_path)
            if self.restore_original_pypirc:
                os.rename(self.original_pypirc_backup, self.config_path)

    def export(self, path, target):
        svn.Arcadia.export(os.path.join("arcadia:/arc/trunk/arcadia", path), target)

    def on_execute(self):
        # params
        if self.Parameters.target_platform not in {'all'}.union(TAGS.keys()):
            raise common.errors.TaskError("Target Platform parameter must be: all, %s" % ', '.join(TAGS.keys()))

        if self.Parameters.target_python not in SUPPORTED_PYTHONS and self.Parameters.target_python != 'all':
            raise common.errors.TaskError(
                "Target Python parameter must be: 'all', %s" % ', '.join(map(lambda x: "'%s'" % x, SUPPORTED_PYTHONS))
            )

        if self.Parameters.target_platform == 'default-darwin-arm64' \
                and self.Parameters.target_python not in {'all'}.union(DARWIN_ARM_SUPPORTED_PYTHONS):
            raise common.errors.TaskError("Doesn't support %s for Darwin arm64" % self.Parameters.target_python)

        if self.Parameters.target_platform == 'all':
            target_platforms = list(TAGS.keys())
        else:
            target_platforms = [self.Parameters.target_platform]

        if self.Parameters.target_python == 'all':
            target_pythons = list(SUPPORTED_PYTHONS)
        else:
            target_pythons = [self.Parameters.target_python]

        access_key = sdk2.Vault.data(self.Parameters.vault_access_key_record)
        secret_key = sdk2.Vault.data(self.Parameters.vault_secret_key_record)

        # prepare env
        self.export(self.Parameters.path, 'package')

        # build
        for target_platform in target_platforms:
            for target_python in target_pythons:
                if target_platform == 'default-darwin-arm64' and target_python not in DARWIN_ARM_SUPPORTED_PYTHONS:
                    continue

                # prepare package
                tmp_dir_name = './tmp.dir'
                if os.path.exists(tmp_dir_name):
                    shutil.rmtree(tmp_dir_name)
                os.mkdir(tmp_dir_name)

                shutil.copytree(
                    os.path.join('package', 'tvmauth'),
                    os.path.join(tmp_dir_name, 'tvmauth'),
                )
                shutil.copyfile(
                    os.path.join('package', 'setup_py3.py'),
                    os.path.join(tmp_dir_name, 'setup.py'),
                )

                self.build_so(
                    target_platform=target_platform,
                    target_python=target_python,
                    out_dir=os.path.join(tmp_dir_name, 'tvmauth'),
                )

                self.build_wheel(
                    target_platform=target_platform,
                    target_python=target_python,
                    package_dir=tmp_dir_name,
                    access_key=access_key,
                    secret_key=secret_key,
                )

    def build_so(self, target_platform, target_python, out_dir):
        build_results_dir = './build_results_dir'
        if os.path.exists(build_results_dir):
            shutil.rmtree(build_results_dir)

        with arcadiasdk.mount_arc_path(sdk2.svn.Arcadia.ARCADIA_TRUNK_URL) as aarcadia:
            arcadiasdk.do_build(
                build_system=sdk_constants.SEMI_DISTBUILD_BUILD_SYSTEM,
                source_root=aarcadia,
                build_type='relwithdebinfo',
                targets=[os.path.join(self.Parameters.path, 'so')],
                results_dir=build_results_dir,
                target_platform=target_platform,
                clear_build=False,
                def_flags={'USE_SYSTEM_PYTHON': to_arcadia_style(target_python)},
            )

        library_name = get_library_name_for_platform(target_platform)
        shutil.copyfile(
            os.path.join(build_results_dir, self.Parameters.path, 'so', library_name),
            os.path.join(out_dir, library_name),
        )

    def build_wheel(self, target_platform, target_python, package_dir, access_key, secret_key):
        os.chdir(package_dir)

        with environments.VirtualEnvironment() as venv:
            with sdk2.helpers.ProcessLog(self, logger='exec') as pl:
                with BuildTvmAuthPypi.PutVaultDataToHome(access_key=access_key, secret_key=secret_key):
                    venv.pip('wheel==0.30.0')
                    sp.call(
                        [
                            'python', 'setup.py',
                            'bdist_wheel',
                            '-p', TAGS[target_platform],
                            '--python-tag', target_python,
                            'upload', '-r', 'yandex',
                        ],
                        shell=False,
                        stdout=pl.stdout,
                        stderr=sp.STDOUT
                    )

        os.chdir('../')
