import os

from sandbox import sdk2, common
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs import svn
from sandbox.sandboxsdk import environments
from shutil import copyfile


CONFIG_TEMPLATE = '''
[distutils]
index-servers =
    yandex

[yandex]
repository: https://pypi.yandex-team.ru/simple/
username: {access_key}
password: {secret_key}

'''


class BuildVaultClientPypi(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        vault_user = sdk2.parameters.String('Vault owner', default='PASSPORT')
        vault_access_key_record = sdk2.parameters.String('Pypi access-key record in vault', default='passport-pypi-access-key')
        vault_secret_key_record = sdk2.parameters.String('Pypi secret-key record in vault', default='passport-pypi-secret-key')
        skip_upload = sdk2.parameters.Bool('Skip to upload a package to the PyPi repository', default=False)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.SvnEnvironment(),
        )

    class PutVaultDataToHome(object):
        def __init__(self, access_key, secret_key):
            self.restore_original_pypirc = False
            self.original_pypirc_backup = None
            self.config_path = os.path.expanduser('~/.pypirc')

            self.access_key = access_key
            self.secret_key = secret_key

        def __enter__(self):
            if os.path.exists(self.config_path):
                self.restore_original_pypirc = True
                self.original_pypirc_backup = self.config_path + '_backup'
                os.rename(self.config_path, self.original_pypirc_backup)

            with open(self.config_path, 'w') as f:
                f.write(
                    CONFIG_TEMPLATE.format(
                        access_key=self.access_key,
                        secret_key=self.secret_key,
                    )
                )

        def __exit__(self, *args):
            os.remove(self.config_path)
            if self.restore_original_pypirc:
                os.rename(self.original_pypirc_backup, self.config_path)

    def export(self, path, target):
        svn.Arcadia.export(os.path.join("arcadia:/arc/trunk/arcadia", path), target)

    def on_execute(self):
        access_key = sdk2.Vault.data(self.Parameters.vault_user, self.Parameters.vault_access_key_record)
        secret_key = sdk2.Vault.data(self.Parameters.vault_user, self.Parameters.vault_secret_key_record)

        self.export('certs', 'certs')
        self.export('passport/python/vault/cli/pypi/', 'package')

        self.export('library/python/vault_client/vault_client', 'package/vault_client')
        copyfile('certs/cacert.pem', 'package/vault_client/YandexInternalRootCA.crt')

        self.export('passport/backend/vault/cli/yav/vault_client_cli', 'package/vault_client_cli')
        self.export('passport/backend/vault/cli/yav_deploy/vault_client_deploy', 'package/vault_client_deploy')

        self.export('passport', 'package/passport')

        os.chdir('package')
        with environments.VirtualEnvironment() as venv:
            with sdk2.helpers.ProcessLog(self, logger='exec') as pl:
                with BuildVaultClientPypi.PutVaultDataToHome(access_key=access_key, secret_key=secret_key):
                    venv.pip('wheel==0.30.0')

                    command = ['python', 'setup.py', 'bdist_wheel', '--universal']
                    if not self.Parameters.skip_upload:
                        command.extend(['upload', '-r', 'yandex'])

                    returncode = sp.call(
                        command,
                        shell=False,
                        stdout=pl.stdout,
                        stderr=sp.STDOUT,
                    )
                    if returncode != 0:
                        raise common.errors.TaskError('Error executing setup.py ({})'.format(returncode))

        os.chdir('../')
