import datetime
import sys

import yt.wrapper


# YTADMINREQ-28913, https://yt.yandex-team.ru/docs/description/dynamic_tables/dynamic_tables_mapreduce#konvertaciya-staticheskoj-tablicy-v-dinamicheskuyu
IO_SPEC = {
    "table_writer": {
        "block_size": 256 * 2**10,
        "desired_chunk_size": 100 * 2**20,
    },
}

ATTRIBUTES_TO_PRESERVE = [
    "schema",
    "expiration_time",
    "optimize_for",
    "atomicity",
    "compression_codec",
    "erasure_codec",
    "min_data_versions",
    "max_data_versions",
    "min_data_ttl",
    "max_data_ttl",
    "merge_rows_on_flush",
    "desired_tablet_count",
    "chunk_writer",
]


def log(msg):
    sys.stderr.write('%s: %s\n' % (datetime.datetime.now(), msg))
    sys.stderr.flush()


def get_attributes(yt_client, table):
    attributes = {}
    for attribute in ATTRIBUTES_TO_PRESERVE:
        if yt_client.has_attribute(table, attribute):
            attributes[attribute] = yt_client.get_attribute(table, attribute)

    return attributes


def prepare_attributes(attributes):
    attributes['compression_codec'] = "brotli_8"
    attributes['desired_tablet_count'] = 10
    return attributes


def run(date, oauth_token, retries):
    yt_client = yt.wrapper.YtClient(proxy='hahn', token=oauth_token)

    # prepare
    table = '//home/passport-rt/production/lbcbck/' + date.strftime('%Y-%m-%d')
    log(table)

    tmp_table = table + '_tmp'
    attributes = None

    def call():
        run_impl(yt_client, table, tmp_table, attributes)

    for idx in range(retries - 1):
        try:
            if attributes is None:
                attributes = prepare_attributes(get_attributes(yt_client, table))
            call()
            return
        except Exception as e:
            log('Exception: %s' % e)

    if attributes is None:
        attributes = prepare_attributes(get_attributes(yt_client, table))
    call()


def run_impl(yt_client, table, tmp_table, attributes):
    # run yt
    log("create table...")
    yt_client.create("table", path=tmp_table, attributes=attributes, ignore_existing=True)
    log("create table... OK")

    log("unmount_table...")
    yt_client.unmount_table(table, sync=True)
    log("unmount_table... OK")

    log("run_merge table...")
    yt_client.run_merge(
        source_table=table,
        destination_table=tmp_table,
        mode='sorted',
        spec={'force_transform': True, 'job_io': IO_SPEC},
    )
    log("run_merge table... OK")

    log("move table...")
    yt_client.move(
        source_path=tmp_table,
        destination_path=table,
        force=True,
        preserve_expiration_time=True,
        preserve_account=True,
    )
    log("move table... OK")

    log("alter_table...")
    yt_client.alter_table(table, dynamic=True)
    log("alter_table... OK")

    log("reshard_table...")
    yt_client.reshard_table(table, tablet_count=attributes['desired_tablet_count'], sync=True)
    log("reshard_table... OK")

    log("mount_table...")
    yt_client.mount_table(table, sync=True)
    log("mount_table... OK")

    log("Finished!")
