# coding: utf-8
import datetime
import logging

from sandbox import sdk2


TIME_FORMAT_PER_DAY = '%Y-%m-%d'


class PassportLogfellerMarkExternalLogsBin(sdk2.Resource):
    """
    binary file, source: arcadia/logfeller/python/logfeller/bin/standalone/mark_external_logs
    """

    executable = True


class PassportLogfellerDropRows(sdk2.Task):
    """
    Удаляет лишние строки из blackbox-log или oauth-log
    PASSP-38104
    """

    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group("Yav settings") as yav_settings:
            yql_token = sdk2.parameters.YavSecret(
                "OAuth YQL token",
                default="sec-01g5pkn8d17p5kp0zk0qhfqhyz#yql_token",
                required=True,
            )
            nirvana_token = sdk2.parameters.YavSecret(
                "OAuth Nirvana token",
                default="sec-01g5pkn8d17p5kp0zk0qhfqhyz#nirvana_token",
                required=True,
            )

        with sdk2.parameters.Group("Common settings") as common_settings:
            retries = sdk2.parameters.Integer(
                "retries",
                default=3,
                required=True,
            )
            logtype = sdk2.parameters.String(
                "logtype",
                default="blackbox-log",
                choices=[("blackbox log", "blackbox-log"), ("oauth log", "oauth-log")],
                required=True,
            )
            offset = sdk2.parameters.Integer(
                "offset (0 to disable) (conflicts with 'date')",
                default=-190,
            )
            date = sdk2.parameters.String(
                "date (%s) (conflicts with 'offset')" % TIME_FORMAT_PER_DAY,
                default='',
            )

    def on_execute(self):
        yql_token = self.Parameters.yql_token.data()[self.Parameters.yql_token.default_key]
        assert len(yql_token) > 0

        nirvana_token = self.Parameters.nirvana_token.data()[self.Parameters.nirvana_token.default_key]
        assert len(nirvana_token) > 0

        retries = self.Parameters.retries
        assert self.Parameters.retries >= 1

        logtype = self.Parameters.logtype

        table = self._get_table()
        logging.info("run for table %s" % table)

        mark_ext_resource = (
            sdk2.Resource.find(
                resource_type=PassportLogfellerMarkExternalLogsBin,
            )
            .order(-sdk2.Resource.id)
            .first()
        )
        mark_ext_bin_path = str(sdk2.ResourceData(mark_ext_resource).path)

        from sandbox.projects.passport.logfeller_drop_rows.lib.drop_rows import run

        run(
            logtype=logtype,
            table=table,
            yql_token=yql_token,
            nirvana_token=nirvana_token,
            retries=retries,
            mark_ext_bin_path=mark_ext_bin_path,
        )

    def _get_table(self):
        offset = self.Parameters.offset or 0
        strdate = self.Parameters.date or ''

        if offset == 0:
            assert strdate != ''
            date = datetime.datetime.strptime(strdate, TIME_FORMAT_PER_DAY)
        else:
            assert strdate == ''
            assert offset < 0
            date = datetime.datetime.now() + offset * datetime.timedelta(days=1)

        return '/'.join(("1d", date.date().strftime(TIME_FORMAT_PER_DAY)))
