import datetime
import subprocess
import sys
import time

from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient


TMP_FOLDER = '//home/passport/production/tmp/logfeller'

DEFAULT_YQL_PREPROC = [
    'PRAGMA yt.DataSizePerJob = "10G"',
    'PRAGMA yt.TmpFolder = "{tmp_folder}"',
    'PRAGMA yt.BinaryTmpFolder = "{tmp_folder}/binary_tmp"',
    'PRAGMA yt.QueryCacheTtl = "1d"',
    'PRAGMA yt.BinaryExpirationInterval = "1d"',
]

BLACKBOX_LOG_SELECT_QUERY_TEMPLATE = """
INSERT INTO `{dst}`
SELECT
    *
FROM `{src}`
WHERE NOT (
    `action` LIKE "sescheck" OR
    (
        `action` LIKE "auth" AND
        `type` LIKE "oauthcheck"
    )
);
"""

OAUTH_LOG_SELECT_QUERY_TEMPLATE = """
INSERT INTO `{dst}`
SELECT
    *
FROM `{src}`
WHERE NOT (
    `mode` LIKE "verify_token"
);
"""

LOG_TYPES = {
    "blackbox-log": {
        "type": "blackbox-log",
        "query": BLACKBOX_LOG_SELECT_QUERY_TEMPLATE,
    },
    "oauth-log": {
        "type": "oauth-log",
        "query": OAUTH_LOG_SELECT_QUERY_TEMPLATE,
    },
}


ATTRIBUTES_TO_PRESERVE = [
    "schema",
    "optimize_for",
    "atomicity",
    "compression_codec",
    "erasure_codec",
    "replication_factor",
]


def log(msg):
    sys.stderr.write('%s: %s\n' % (datetime.datetime.now(), msg))
    sys.stderr.flush()


def run(logtype, table, yql_token, nirvana_token, retries, mark_ext_bin_path):
    context = Context(logtype, table, yql_token, nirvana_token, mark_ext_bin_path)

    for _ in range(retries - 1):
        try:
            context.try_run()
            return
        except Exception as e:
            log('Exception: %s. Sleep for 5 seconds' % e)
            time.sleep(5)

    context.try_run()


class Context:
    def __init__(self, logtype, table, yql_token, nirvana_token, mark_ext_bin_path):
        self.log_settings = LOG_TYPES[logtype]

        self.yql_token = yql_token
        self.nirvana_token = nirvana_token

        self.yt_client = YtClient(proxy='hahn', token=self.yql_token)
        self.yql_client = YqlClient(db='hahn', token=self.yql_token)

        self.log_path = '/'.join(('//logs', logtype, table))
        self.tmp_path = '/'.join((TMP_FOLDER, logtype, table))

        self.mark_ext_command = [mark_ext_bin_path, "-c", "hahn", "-l", logtype, "-d", self.log_path, "-f"]

        self.attributes = None
        self.select_completed = False
        self.move_completed = False

    def try_run(self):
        log("Run for '%s' with tmp table: '%s'" % (self.log_path, self.tmp_path))

        self.__prepare_attributes()
        self.__select()
        self.__move()
        self.__mark()

        log("Finished!")

    def __prepare_attributes(self):
        if self.attributes is not None:
            return

        log("prepare attributes...")
        attributes = {}
        for attribute in ATTRIBUTES_TO_PRESERVE:
            if self.yt_client.has_attribute(self.log_path, attribute):
                attributes[attribute] = self.yt_client.get_attribute(self.log_path, attribute)
        log("prepare attributes... OK")

        self.attributes = attributes

    def __select(self):
        if self.select_completed:
            return

        log("create tmp table...")
        self.yt_client.create("table", path=self.tmp_path, attributes=self.attributes, recursive=True, force=True)
        log("create tmp table... OK")

        log("run select query...")
        request = self.yql_client.query(
            ';\n'.join(DEFAULT_YQL_PREPROC + [self.log_settings["query"]]).format(
                src=self.log_path,
                dst=self.tmp_path,
                tmp_folder=TMP_FOLDER,
            ),
            syntax_version=1,
        )
        request.run()
        query_result = request.get_results()
        assert query_result.is_success, "%s.%s" % (query_result.status, ' - '.join(query_result.errors))
        log("run select query... OK")

        self.select_completed = True

    def __move(self):
        if self.move_completed:
            return

        log("move table...")
        self.yt_client.move(
            source_path=self.tmp_path,
            destination_path=self.log_path,
            force=True,
        )
        log("move table... OK")

        self.move_completed = True

    def __mark(self):
        env = {
            "YT_TOKEN": self.nirvana_token,
            "OAUTH_TOKEN": self.nirvana_token,
            "LOGFELLER_CONFIGS": "nirvactor:hahn",
            "WORKFLOW_MANAGER_SUBSYSTEM": "BOTH",
        }
        subprocess.check_call(self.mark_ext_command, stdout=sys.stderr, stderr=sys.stderr, env=env)
