# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import link_builder
from sandbox.common.types.task import Status
from sandbox.common.types import resource


class TestGroup(object):
    PassportBackend = 'passport_backend'
    Other = 'other'


class TestGroupParameter(sdk2.parameters.String):
    name = 'Test group'
    description = 'Test group describes test environment'
    default_value = 'passport_backend'
    choices = [
        ('passport_backend', TestGroup.PassportBackend),
        ('other', TestGroup.Other)
    ]


class Env(object):
    Development = 'development'
    Testing = 'testing'
    Rc = 'rc'
    Production = 'production'
    IntranetTesting = 'intranet_testing'
    IntranetRc = 'intranet_rc'
    IntranetProduction = 'intranet_production'


ENV_TO_SECRET_ID = {
    Env.Testing: 'sec-01ewabcg6j4jyd129mt11t4r5d',
    Env.Production: 'sec-01ewaz478n30zjpnknv6dsay4f',
    Env.IntranetTesting: 'sec-01ewaz5zsas0jbc2p5djtrc2sz',
    Env.IntranetProduction: 'sec-01ewaz85aye7za71za8h9zyfdr',
}
ENV_TO_SECRET_ID[Env.Development] = ENV_TO_SECRET_ID[Env.Testing]
ENV_TO_SECRET_ID[Env.Rc] = ENV_TO_SECRET_ID[Env.Production]
ENV_TO_SECRET_ID[Env.IntranetRc] = ENV_TO_SECRET_ID[Env.IntranetProduction]


CI_SECRET_ID = 'sec-01ekz37bg804cbcsqeftv8v8ec'
AUTOTESTS_GROUP_NAME = 'Autotests\' params'


def get_sandbox_task_url(task_id):
    return 'https://sandbox.yandex-team.ru/task/%s/view' % task_id


class EnvParameter(sdk2.parameters.String):
    name = 'Environment'
    description = 'Environment where tests will be run'
    default_value = 'testing'
    choices = [
        ('development', Env.Development),
        ('testing', Env.Testing),
        ('rc', Env.Rc),
        ('production', Env.Production),
        ('intranet_testing', Env.IntranetTesting),
        ('intranet_rc', Env.IntranetRc),
        ('intranet_production', Env.IntranetProduction)
    ]


class PassportAutotests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        push_tasks_resource = True

        with sdk2.parameters.Group(AUTOTESTS_GROUP_NAME):
            test_group = TestGroupParameter()
            env = EnvParameter()
            targets_prefix = sdk2.parameters.String(
                label='Projects path prefix',
                default='passport/backend/qa/autotests',
            )
            projects = sdk2.parameters.List(
                label='Projects',
                description='List of projects to run tests for (if omitted, all the tests will be run)',
            )
            send_report_to_st_issue = sdk2.parameters.String(
                label='Send report to ST issue',
                description='Issue key; tests results will be posted as comment',
            )

        with sdk2.parameters.Output():
            are_tests_ok = sdk2.parameters.Bool('Are tests ok?')
            allure_report_url = sdk2.parameters.String('Allure report url')

    def run_subtask(self):
        task_class = sdk2.Task['YA_MAKE_2']
        secret_id = None

        if self.Parameters.test_group == TestGroup.PassportBackend:
            task_class = sdk2.Task['PASSPORT_AUTOTESTS_YA_MAKE_2']
            secret_id = ENV_TO_SECRET_ID[self.Parameters.env]

        targets_prefix = self.Parameters.targets_prefix
        if self.Parameters.projects:
            targets = ';'.join([
                '%s/%s' % (targets_prefix, project)
                for project in self.Parameters.projects
            ])
        else:
            targets = targets_prefix

        subtask = task_class(
            self,
            description='Run passport autotests',
            targets=targets,
            test=True,
            allure_report=True,
            allure_report_ttl=365,
            junit_report=True,
            use_aapi_fuse=True,
            use_arc_instead_of_aapi=True,
            ya_yt_token_yav_secret='%s#ci.token' % CI_SECRET_ID,
            sandbox_tags='IPV6',
            env_vars='ENV="%s"' % self.Parameters.env,
            secret_id=secret_id,
            # Должно быть не более https://a.yandex-team.ru/arc/trunk/arcadia/passport/backend/qa_proxy/cmd/main.go?rev=r9261342#L36
            test_threads=40,
        )
        subtask.enqueue()
        self.Context.subtask_id = subtask.id
        self.Context.save()
        raise sdk2.WaitTask(subtask.id, Status.Group.FINISH + Status.Group.BREAK)

    def parse_junit_report(self, junit_report_url):
        from junitparser import JUnitXml
        import requests

        junit_report = requests.get(junit_report_url).content
        junit_xml = JUnitXml.fromstring(junit_report)

        total_tests = sum([suite.tests for suite in junit_xml], 0)
        failed_tests = sum([suite.failures for suite in junit_xml], 0)
        skipped_tests = sum([suite.skipped for suite in junit_xml], 0)

        self.Context.tests_stats = {
            'total': total_tests,
            'failed': failed_tests,
            'skipped': skipped_tests,
        }
        self.Context.save()

    @sdk2.report(title='Tests result', label='tests_result')
    def report(self):
        title_style = 'style="line-height: 2; font-size: 20px"'
        link_style = 'style="font-size: 16px"'

        if not self.Context.tests_stats:
            return '<p %s><b>Тесты ещё не завершились</b></p>' % title_style
        else:
            total_tests = self.Context.tests_stats['total']
            failed_tests = self.Context.tests_stats['failed']
            skipped_tests = self.Context.tests_stats['skipped']

            if failed_tests:
                message = '<font color="red">Сломалось %s тестов из %s</font>' % (failed_tests, total_tests)
            else:
                message = '<font color="green">Прошло %s тестов, пропущено %s</font>' % (
                    total_tests - skipped_tests,
                    skipped_tests,
                )
            return '<p %s><b>%s</b></p> <p %s><a href="%s" target="_self">Allure-отчёт</a></p>' % (
                title_style,
                message,
                link_style,
                self.Parameters.allure_report_url,
            )

    def send_report_to_st_issue(self, issue_key, allure_report_url):
        total_tests = self.Context.tests_stats['total']
        failed_tests = self.Context.tests_stats['failed']
        skipped_tests = self.Context.tests_stats['skipped']

        if failed_tests:
            tests_status = u'!!(red)**не OK**!! (упало %s тестов из %s)' % (failed_tests, total_tests)
        else:
            tests_status = u'!!(green)**OK**!! (прошло %s тестов, пропущено %s)' % (
                total_tests - skipped_tests,
                skipped_tests,
            )
        msg = u'((%s Запуск)) python-автотестов в %s: %s\n\n((%s Allure-отчёт))' % (
            get_sandbox_task_url(self.id),
            self.Parameters.env,
            tests_status,
            allure_report_url,
        )

        self.post_comment_to_st(issue_key, msg)

    @staticmethod
    def post_comment_to_st(issue_key, comment):
        from startrek_client import Startrek

        secret_data = sdk2.yav.Secret(CI_SECRET_ID).data()
        oauth_token = secret_data['startrek.token']
        startrek_client = Startrek(
            useragent='PASSPORT_AUTOTESTS sandbox task',
            base_url='https://st-api.yandex-team.ru',
            token=oauth_token,
        )
        startrek_client.issues[issue_key].comments.create(text=comment, params={'isAddToFollowers': False})

    def check_subtask_result(self):
        subtask = sdk2.Task.find(id=self.Context.subtask_id).first()
        allure_report_resource = sdk2.Resource['ALLURE_REPORT'].find(task=subtask).first()
        if not allure_report_resource:
            raise errors.TaskError('No allure report found (seems like subtask has failed)')
        allure_report_url = '%s/index.html' % allure_report_resource.http_proxy
        self.Parameters.allure_report_url = allure_report_url

        msg = (
            'Subtask {subtask} status: {task_status}\n'
            '<a href="{allure_report_url}">Allure report</a>'
        ).format(
            subtask=link_builder.task_link(subtask.id, plain=False),
            task_status=subtask.status,
            allure_report_url=allure_report_url,
        )
        self.set_info(info=msg, do_escape=False)

        junit_report_resource = sdk2.Resource['TASK_LOGS'].find(task=subtask).first()
        if not junit_report_resource:
            raise errors.TaskError('No JUnit report found')
        junit_report_url = '%s/junit_report.xml' % junit_report_resource.http_proxy
        self.parse_junit_report(junit_report_url)

        st_issue_key = self.Parameters.send_report_to_st_issue
        if st_issue_key:
            self.send_report_to_st_issue(
                issue_key=st_issue_key,
                allure_report_url=allure_report_url,
            )

        # Если subtask.status == Status.SUCCESS - значит, все тесты прошли.
        # Иначе - сломалась часть тестов или задача целиком.
        are_tests_ok = subtask.status == Status.SUCCESS
        self.Parameters.are_tests_ok = are_tests_ok

        if not are_tests_ok:
            raise errors.TaskError('Some tests have failed')

    def on_create(self):
        if not self.Requirements.tasks_resource:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                owner='PASSPORT',
                state=resource.State.READY,
                attrs={'task_type': 'PASSPORT_AUTOTESTS', 'auto_deploy': True},
            ).first()

    def on_execute(self):
        with self.memoize_stage.run_subtask:
            self.run_subtask()

        self.check_subtask_result()
