# -*- coding: utf-8 -*-
import logging
from time import time

import requests
from sandbox import sandboxsdk
from sandbox.projects.common.build.YaPackage import YaPackage


log = logging.getLogger(__name__)


TASK_TIMEOUT = 600
WAIT_FOR = 10


class WaitForReviewParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'wait_for_review'
    description = 'Review id to be merged before build'
    required = False
    default_value = None


class ArcanumTokenParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'arcanum_token_name'
    description = 'OAuth token for Arcanum (task owner must have permissions to access it)'
    required = False
    default_value = 'passport-robot-arcanum-token'


class PassportYaPackage(YaPackage, object):
    # Осторожно: SDK1-задача!
    """
    Like YaPackage, but allows postponing execution until PR with changelog is merged.
    """
    type = 'PASSPORT_YA_PACKAGE'

    input_parameters = YaPackage.input_parameters + [
        ArcanumTokenParameter,
        WaitForReviewParameter,
    ]

    def is_review_merged(self, review_id, oauth_token):
        try:
            resp = requests.get(
                'https://a.yandex-team.ru/api/v1/pullrequest/%s' % review_id,
                headers={
                    'Authorization': 'OAuth %s' % oauth_token,
                },
            )
        except requests.RequestException as e:
            log.error('Error while accessing Arcanum api: %s %s', e.__class__.__name__, e)
            return

        if resp.status_code != 200:
            log.error('Arcanum api responded with status %s: %s', resp.status_code, resp.content)
            return

        rv = resp.json()
        if 'state' not in rv:
            log.error('Got bad answer from Arcanum api: %s', rv)
            return

        return rv['state'] == 'submitted'

    def on_execute(self):
        if 'started_at' not in self.ctx:
            self.ctx['started_at'] = int(time())

        review_id = self.ctx.get('wait_for_review')
        if review_id:
            if time() > self.ctx['started_at'] + TASK_TIMEOUT:
                error_message = 'Failed to start build: review https://a.yandex-team.ru/review/%s hasn\'t been merged yet' % review_id
                raise RuntimeError(error_message)

            token = self.get_vault_data(self.owner, self.ctx.get('arcanum_token_name'))
            if not self.is_review_merged(review_id=review_id, oauth_token=token):
                log.debug('Review not merged yet. Waiting for %s seconds...', WAIT_FOR)
                self.wait_time(WAIT_FOR)

        return super(PassportYaPackage, self).on_execute()
