# -*- coding: utf-8 -*-
import logging
from time import time

import requests
from sandbox import sdk2
from sandbox.projects.common.build.YaPackage2 import YaPackage2


log = logging.getLogger(__name__)


TASK_TIMEOUT = 600
WAIT_FOR = 10


class PassportYaPackage2(YaPackage2):
    """
    Like YaPackage2, but allows postponing execution until PR with changelog is merged.
    """

    class Parameters(YaPackage2.Parameters):
        wait_for_review = sdk2.parameters.String('Review id to be merged before build')
        arcanum_token_name = sdk2.parameters.String(
            'OAuth token for Arcanum (task owner must have permissions to access it)',
            default='passport-robot-arcanum-token',
        )

    def is_review_merged(self, review_id, oauth_token):
        try:
            resp = requests.get(
                'https://a.yandex-team.ru/api/v1/pullrequest/%s' % review_id,
                headers={
                    'Authorization': 'OAuth %s' % oauth_token,
                },
            )
        except requests.RequestException as e:
            log.error('Error while accessing Arcanum api: %s %s', e.__class__.__name__, e)
            return

        if resp.status_code != 200:
            log.error('Arcanum api responded with status %s: %s', resp.status_code, resp.content)
            return

        rv = resp.json()
        if 'state' not in rv:
            log.error('Got bad answer from Arcanum api: %s', rv)
            return

        return rv['state'] == 'submitted'

    def on_execute(self):
        if 'started_at' not in self.Context:
            self.Context.started_at = int(time())

        review_id = self.Parameters.wait_for_review
        if review_id:
            if time() > self.Context.started_at + TASK_TIMEOUT:
                error_message = 'Failed to start build: review https://a.yandex-team.ru/review/%s hasn\'t been merged yet' % review_id
                raise RuntimeError(error_message)

            token = sdk2.Vault.data(self.owner, self.Parameters.arcanum_token_name)
            if not self.is_review_merged(review_id=review_id, oauth_token=token):
                log.debug('Review not merged yet. Waiting for %s seconds...', WAIT_FOR)
                raise sdk2.WaitTime(WAIT_FOR)

        return super(PassportYaPackage2, self).on_execute()
