# -*- coding: utf-8 -*-
import errno
from json import loads
import logging
import os
import re

import requests
from sandbox import (
    common,
    sdk2,
)
from sandbox.common import errors
from sandbox.common.types import (
    notification as ctn,
    task as ctt,
)
from sandbox.projects.common.geosearch.startrek import StartrekClient
from sandbox.projects.passport.regression_shooting.generate_ammo import (
    AmmoType,
    PassportGenerateAmmo,
    try_get_freshest_ammo,
)
from sandbox.projects.passport.regression_shooting.send_report import (
    create_report,
    PassportSendShootingReport,
)
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi


log = logging.getLogger(__name__)


TANKS_IVA = [
    'tank-i1.passport.yandex.net:8083',
]

DEVIATION_THRESHOLD = 0.1


def read_file(key):
    # FIXME(white): this could be (and should be) replaced by common.fs.read_file()
    # FIXME(white): once we deploy the new client package
    if common.system.inside_the_binary():
        from library.python import resource
        key = os.path.join("sandbox/projects/passport/regression_shooting", key)
        resource_data = resource.find(key)
        if resource_data is not None:
            return resource_data
        raise IOError(errno.ENOENT, "There's no resource {} in binary".format(key))

    base_dir = os.path.dirname(os.path.abspath(__file__))
    config_path = os.path.join(base_dir, key)
    with open(config_path, 'r') as fileobj:
        return fileobj.read()


def read_tank_config(config_name):
    return read_file('configs/%s.tank.yaml' % config_name)


def read_monitoring_config():
    return read_file('configs/monitoring.xml')


@common.utils.singleton
def AVAILABLE_SHOOT_TYPES():
    return {
        'frontend_auth_const': {
            'desc': 'Shoot on auth (frontend) - const',
            'config_content': read_tank_config('frontend_auth_const'),
            'ammo_type': AmmoType.PassportFrontendAuth.value,
            'ammo_count': 300 * 10 * 60,
            'type': 'const',
        },
        'frontend_auth_line': {
            'desc': 'Shoot on auth (frontend) - line',
            'config_content': read_tank_config('frontend_auth_line'),
            'ammo_type': AmmoType.PassportFrontendAuth.value,
            'ammo_count': 1000 * 20 * 60,
            'type': 'line',
        },
        'backend_register_const': {
            'desc': 'Shoot on register (backend) - const',
            'config_content': read_tank_config('backend_register_const'),
            'ammo_type': AmmoType.PassportBackendRegister.value,
            'ammo_count': 500 * 10 * 60,
            'type': 'const',
        },
        'backend_register_line': {
            'desc': 'Shoot on register (backend) - line',
            'config_content': read_tank_config('backend_register_line'),
            'ammo_type': AmmoType.PassportBackendRegister.value,
            'ammo_count': 1000 * 20 * 60,
            'type': 'line',
        },
        'blackbox_checkip_const': {
            'desc': 'Shoot on blackbox method=checkip - const',
            'config_content': read_tank_config('blackbox_checkip_const'),
            'ammo_type': AmmoType.BlackboxCheckIp.value,
            'ammo_count': 0,
            'type': 'const',
            'reuse_ammo': True,
            'retry_count': 5,
            'stddev': 1.3,
        },
        'blackbox_checkip_line': {
            'desc': 'Shoot on blackbox method=checkip - line',
            'config_content': read_tank_config('blackbox_checkip_line'),
            'ammo_type': AmmoType.BlackboxCheckIp.value,
            'ammo_count': 0,
            'type': 'line',
            'reuse_ammo': True,
        },
        'blackbox_userinfo_simple_const': {
            'desc': 'Shoot on blackbox method=userinfo, simple request - const',
            'config_content': read_tank_config('blackbox_userinfo_simple_const'),
            'ammo_type': AmmoType.BlackboxUserinfoSimple.value,
            'ammo_count': 0,
            'type': 'const',
            'reuse_ammo': True,
            'retry_count': 5,
            'stddev': 1.5,
        },
        'blackbox_userinfo_simple_line': {
            'desc': 'Shoot on blackbox method=userinfo, simple request - line',
            'config_content': read_tank_config('blackbox_userinfo_simple_line'),
            'ammo_type': AmmoType.BlackboxUserinfoSimple.value,
            'ammo_count': 0,
            'type': 'line',
            'reuse_ammo': True,
        },
        'blackbox_userinfo_mixed_json_const': {
            'desc': 'Shoot on blackbox method=userinfo, mixed request, format=json - const',
            'config_content': read_tank_config('blackbox_userinfo_mixed_json_const'),
            'ammo_type': AmmoType.BlackboxUserinfoMixedJson.value,
            'ammo_count': 0,
            'type': 'const',
            'reuse_ammo': True,
        },
        'blackbox_userinfo_mixed_json_line': {
            'desc': 'Shoot on blackbox method=userinfo, mixed request, format=json - line',
            'config_content': read_tank_config('blackbox_userinfo_mixed_json_line'),
            'ammo_type': AmmoType.BlackboxUserinfoMixedJson.value,
            'ammo_count': 0,
            'type': 'line',
            'reuse_ammo': True,
        },
        'blackbox_userinfo_mixed_xml_const': {
            'desc': 'Shoot on blackbox method=userinfo, mixed request, format=xml - const',
            'config_content': read_tank_config('blackbox_userinfo_mixed_xml_const'),
            'ammo_type': AmmoType.BlackboxUserinfoMixedXml.value,
            'ammo_count': 0,
            'type': 'const',
            'reuse_ammo': True,
        },
        'blackbox_userinfo_mixed_xml_line': {
            'desc': 'Shoot on blackbox method=userinfo, mixed request, format=xml - line',
            'config_content': read_tank_config('blackbox_userinfo_mixed_xml_line'),
            'ammo_type': AmmoType.BlackboxUserinfoMixedXml.value,
            'ammo_count': 0,
            'type': 'line',
            'reuse_ammo': True,
        },
        'blackbox_login_mixed_const': {
            'desc': 'Shoot on blackbox method=login, mixed request - const',
            'config_content': read_tank_config('blackbox_login_mixed_const'),
            'ammo_type': AmmoType.BlackboxLoginMixed.value,
            'ammo_count': 0,
            'type': 'const',
            'reuse_ammo': True,
        },
        'blackbox_login_mixed_line': {
            'desc': 'Shoot on blackbox method=login, mixed request - line',
            'config_content': read_tank_config('blackbox_login_mixed_line'),
            'ammo_type': AmmoType.BlackboxLoginMixed.value,
            'ammo_count': 0,
            'type': 'line',
            'reuse_ammo': True,
        },
        'blackbox_sessionid_mixed_const': {
            'desc': 'Shoot on blackbox method=sessionid, mixed request - const',
            'config_content': read_tank_config('blackbox_sessionid_mixed_const'),
            'ammo_type': AmmoType.BlackboxSessionidMixed.value,
            'ammo_count': 0,
            'type': 'const',
        },
        'blackbox_sessionid_mixed_line': {
            'desc': 'Shoot on blackbox method=sessionid, mixed request - line',
            'config_content': read_tank_config('blackbox_sessionid_mixed_line'),
            'ammo_type': AmmoType.BlackboxSessionidMixed.value,
            'ammo_count': 0,
            'type': 'line',
        },
        'blackbox_oauth_mixed_const': {
            'desc': 'Shoot on blackbox method=oauth, mixed request - const',
            'config_content': read_tank_config('blackbox_oauth_mixed_const'),
            'ammo_type': AmmoType.BlackboxOAuthMixed.value,
            'ammo_count': 0,
            'type': 'const',
        },
        'blackbox_oauth_mixed_line': {
            'desc': 'Shoot on blackbox method=oauth, mixed request - line',
            'config_content': read_tank_config('blackbox_oauth_mixed_line'),
            'ammo_type': AmmoType.BlackboxOAuthMixed.value,
            'ammo_count': 0,
            'type': 'line',
        },
        'blackbox_oauth_uniform_const': {
            'desc': 'Shoot on blackbox method=oauth, single request - const',
            'config_content': read_tank_config('blackbox_oauth_uniform_const'),
            'ammo_type': AmmoType.BlackboxOAuthUniform.value,
            'ammo_count': 0,
            'type': 'const',
            'retry_count': 5,
            'stddev': 12.0,
        },
        'blackbox_login_uniform_const': {
            'desc': 'Shoot on blackbox method=login, single request - const',
            'config_content': read_tank_config('blackbox_login_uniform_const'),
            'ammo_type': AmmoType.BlackboxLoginUniform.value,
            'ammo_count': 0,
            'type': 'const',
            'reuse_ammo': True,
            'retry_count': 5,
            'stddev': 4.0,
        },
        'blackbox_sessionid_uniform_const': {
            'desc': 'Shoot on blackbox method=sessionid, single request - const',
            'config_content': read_tank_config('blackbox_sessionid_uniform_const'),
            'ammo_type': AmmoType.BlackboxSessionidUniform.value,
            'ammo_count': 0,
            'type': 'const',
            'retry_count': 5,
            'stddev': 10.0,
        },
        'oauth_gt_password_new_normal_const': {
            'desc': 'Shoot on OAuth gt=password (issue normal tokens with deleting old ones) - const',
            'config_content': read_tank_config('oauth_gt_password_new_normal_const'),
            'ammo_type': AmmoType.OAuthGTPasswordNewNormal.value,
            'ammo_count': 500 * 10 * 60,
            'type': 'const',
            'reuse_ammo': True,
        },
        'oauth_gt_password_new_normal_line': {
            'desc': 'Shoot on OAuth gt=password (issue normal tokens with deleting old ones) - line',
            'config_content': read_tank_config('oauth_gt_password_new_normal_line'),
            'ammo_type': AmmoType.OAuthGTPasswordNewNormal.value,
            'ammo_count': 500 * 20 * 60,
            'type': 'line',
            'reuse_ammo': True,
        },
        'oauth_gt_xtoken_update_normal_const': {
            'desc': 'Shoot on OAuth gt=x-token (issue normal tokens with reuse and update) - const',
            'config_content': read_tank_config('oauth_gt_xtoken_update_normal_const'),
            'ammo_type': AmmoType.OAuthGTXTokenUpdateNormal.value,
            'ammo_count': 1000 * 10 * 60,
            'type': 'const',
            'reuse_ammo': True,
        },
        'oauth_gt_xtoken_update_normal_line': {
            'desc': 'Shoot on OAuth gt=x-token (issue normal tokens with reuse and update) - line',
            'config_content': read_tank_config('oauth_gt_xtoken_update_normal_line'),
            'ammo_type': AmmoType.OAuthGTXTokenUpdateNormal.value,
            'ammo_count': 500 * 20 * 60,
            'type': 'line',
            'reuse_ammo': True,
        },
        'oauth_gt_xtoken_new_stateless_const': {
            'desc': 'Shoot on OAuth gt=x-token (issue stateless tokens) - const',
            'config_content': read_tank_config('oauth_gt_xtoken_new_stateless_const'),
            'ammo_type': AmmoType.OAuthGTXTokenNewStateless.value,
            'ammo_count': 1500 * 10 * 60,
            'type': 'const',
            'reuse_ammo': True,
        },
        'oauth_gt_xtoken_new_stateless_line': {
            'desc': 'Shoot on OAuth gt=x-token (issue stateless tokens) - line',
            'config_content': read_tank_config('oauth_gt_xtoken_new_stateless_line'),
            'ammo_type': AmmoType.OAuthGTXTokenNewStateless.value,
            'ammo_count': 750 * 20 * 60,
            'type': 'line',
            'reuse_ammo': True,
        },
        'vault_tokens_heavy_line': {
            'desc': 'Shoot on Vault (tokens) - heavy line',
            'config_content': read_tank_config('vault_tokens_heavy_line'),
            'ammo_type': AmmoType.VaultTokensHeavy.value,
            'ammo_count': 0,
            'type': 'line',
            'reuse_ammo': True,
        },
        'vault_tokens_line': {
            'desc': 'Shoot on Vault (tokens) - line',
            'config_content': read_tank_config('vault_tokens_line'),
            'ammo_type': AmmoType.VaultTokens.value,
            'ammo_count': 0,
            'type': 'line',
            'reuse_ammo': True,
        },
        'vault_tokens_mixed_line': {
            'desc': 'Shoot on Vault (tokens) - mixed line',
            'config_content': read_tank_config('vault_tokens_mixed_line'),
            'ammo_type': AmmoType.VaultTokensMixed.value,
            'ammo_count': 0,
            'type': 'line',
            'reuse_ammo': True,
        },
    }


def get_freshest_ammo(shoot_type, ammo_type):
    ammo = try_get_freshest_ammo(shoot_type, ammo_type)
    if not ammo:
        raise errors.TaskError('No ammo (type %s) found for %s', ammo_type, shoot_type)
    return ammo


class PassportRegressionShooting(sdk2.Task):
    """ Shooting at passport projects """

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Task parameters'):
            send_results_to_ticket = sdk2.parameters.Bool(
                'Send results to ST ticket',
                description='Whether the results will be sent to ST as comments',
                default=False,
            )
            with send_results_to_ticket.value[True]:
                ticket = sdk2.parameters.String(
                    'ST ticket for shooting',
                    description='The shoot will be performed in this ticket. The report will be sent as a comment',
                    default='PASSP-19689',
                    required=True,
                )
                token = sdk2.parameters.String(
                    'ST token for comment',
                    description='Task owner must have permissions to access this token in SB Vault.',
                    default='passport-robot-startrek-token',
                )
            send_results_to_email = sdk2.parameters.Bool(
                'Send results via email',
                description='Whether the results will be sent to email',
                default=False,
            )
            with send_results_to_email.value[True]:
                emails_or_logins = sdk2.parameters.List(
                    'Emails or logins',
                    required=True,
                    default=['passport-stats'],
                )
            aggregate_reports = sdk2.parameters.Bool(
                'Aggregate reports',
                description='Send one letter/comment for all the shoot types',
                default=False,
            )
            with aggregate_reports.value[True]:
                report_title = sdk2.parameters.String(
                    'Aggregated report title',
                    default='Shooting aggregated report',
                )

        with sdk2.parameters.CheckGroup('Shoot types') as shoot_types:
            for shoot_type, config in sorted(AVAILABLE_SHOOT_TYPES().iteritems()):
                setattr(
                    shoot_types.values,
                    shoot_type,
                    shoot_types.Value('[%s] %s' % (shoot_type, config['desc']), checked=False),
                )

        force_recreate_ammo = sdk2.parameters.Bool(
            'Force recreate ammo',
            description='Create new ammo even if old one exists and is valid',
            default=False,
        )

    class Context(sdk2.Task.Context):
        shoot_data = {}
        generate_ammo_tasks = []
        report_tasks = []

    def start_generating_ammo(self, shoot_type, ammo_type, ammo_count, try_reuse_ammo=False):
        ammo_generation_subtask = PassportGenerateAmmo(
            self,
            description='Generating ammo for %s' % shoot_type,
            shoot_type=shoot_type,
            ammo_type=ammo_type,
            ammo_count=ammo_count,
            try_reuse_ammo=try_reuse_ammo,
        ).enqueue()
        self.Context.generate_ammo_tasks.append(ammo_generation_subtask.id)

    def start_shooting(self, shoot_type):
        log.info('Preparing to shoot at %s', shoot_type)
        shoot = self.Context.shoot_data[shoot_type]
        subtask_shoot = ShootViaTankapi(
            self,
            description='Shooting at %s' % shoot_type,
            ammo_source='resource',
            ammo_resource=shoot['ammo'],
            config_source='file',
            config_content=AVAILABLE_SHOOT_TYPES()[shoot_type]['config_content'],
            tanks=TANKS_IVA,
            logs_to_save=['tank*.log'],  # not downloading answer.log: it can be too large
            use_monitoring=True,
            monitoring_source='file',
            monitoring_content=read_monitoring_config(),
        ).enqueue()
        log.info('Subtask (id=%s) with shooting %s is started', subtask_shoot.id, shoot_type)
        shoot['subtask'] = subtask_shoot.id
        raise sdk2.WaitTask(
            [subtask_shoot.id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
            timeout=14400,
        )

    def _get_shoot_id_from_lunapark_link(self, lunapark_link):
        if not lunapark_link:
            return
        digit_groups = re.findall(r'\d+', lunapark_link)
        if digit_groups:
            return digit_groups[0]

    def _shooting_accepted(self, lunapark_link, stddev):
        if not stddev:
            log.info('Warning: I feel bug in your config, you want retries but didn\'t set stddev', lunapark_link)
            return True

        shoot_id = self._get_shoot_id_from_lunapark_link(lunapark_link)

        if not shoot_id:
            log.info('Warning: lunapark job failed, could\'t parse lunapark link %s', lunapark_link)
            return False

        r = requests.get('https://lunapark.yandex-team.ru/api/job/%s/aggregates.json?main_only=1' % shoot_id)
        doc = loads(r.text)[0]
        dev = doc['expect_stddev']

        # shooting is accepted if its deviation is not more than stddev + 10%
        status = dev < stddev * (1 + DEVIATION_THRESHOLD)
        log.info('Lunapark job %s deviation is %f, status: %s', shoot_id, dev, 'accepted' if status else 'failed')

        # if the job is rejected let's add comment to job description
        # and clean the job component - it will stay in history but not show on regression chart
        if not status:
            r = requests.get('https://lunapark.yandex-team.ru/api/job/%s/summary.json' % shoot_id)
            doc = loads(r.text)[0]
            desc = doc['dsc']
            if desc:
                desc += '\n'
            desc += 'Rejected, stddev=%.2f' % dev

            requests.post('https://lunapark.yandex-team.ru/api/job/%s/edit.json' % shoot_id, json={'component': '', 'description': desc})

        return status

    def start_reporting(self, shoot_type):
        shoot = self.Context.shoot_data[shoot_type]
        log.info('Start reporting for %s, shoot %s (task %s)', shoot_type, shoot['lunapark_link'], shoot['subtask'])

        shoot_id = self._get_shoot_id_from_lunapark_link(shoot['lunapark_link'])
        if not shoot_id:
            log.info('Failed to parse lunapark link %s for %s shoot', shoot['lunapark_link'], shoot_type)
            return

        subtask_report = PassportSendShootingReport(
            self,
            description='Creating report for %s' % shoot_type,
            shoot_id=shoot_id,
            report_type=AVAILABLE_SHOOT_TYPES()[shoot_type]['type'],
            report_title=AVAILABLE_SHOOT_TYPES()[shoot_type]['desc'],
            send_comment=self.Parameters.send_results_to_ticket,
            ticket_id=self.Parameters.ticket,
            st_token_name=self.Parameters.token,
            send_letter=self.Parameters.send_results_to_email,
            mail_recipients=self.Parameters.emails_or_logins,
        ).enqueue()
        self.Context.report_tasks.append(subtask_report.id)

    def send_aggregated_report(self, shoot_types):
        failed_subreports, ok_subreports = [], []
        for shoot_type in shoot_types:
            subtitle = AVAILABLE_SHOOT_TYPES()[shoot_type]['desc']

            shoot = self.Context.shoot_data[shoot_type]
            shoot_id = self._get_shoot_id_from_lunapark_link(shoot['lunapark_link'])
            if not shoot_id:
                failed_subreports.append((subtitle, 'Failed to parse lunapark link %s' % shoot['lunapark_link']))
                continue

            report = create_report(shoot_id=shoot_id, report_type=AVAILABLE_SHOOT_TYPES()[shoot_type]['type'])
            report_message = report.message
            if 'try_number' in shoot:
                report_message += 'Accepted result from try %s\n' % shoot['try_number']

            is_ok = report.check_resolution()
            if is_ok:
                ok_subreports.append((subtitle, report_message))
            else:
                failed_subreports.append((subtitle, report_message))

        if failed_subreports:
            title = '[%s/%s failed] <FAIL> %s' % (
                len(failed_subreports),
                len(failed_subreports) + len(ok_subreports),
                self.Parameters.report_title,
            )
        else:
            title = '[%s complete] <OK> %s' % (
                len(ok_subreports),
                self.Parameters.report_title,
            )

        message = ''
        if failed_subreports:
            message += '\n\n'.join([
                '    ====== Failed shoots ======',
                ''.join('  === %s ===\n%s\n' % subreport for subreport in failed_subreports),
                '',  # to separate sections
            ])
        if ok_subreports:
            message += '\n\n'.join([
                '    ====== Successful shoots ======',
                ''.join('  === %s ===\n%s\n' % subreport for subreport in ok_subreports),
            ])

        if self.Parameters.send_results_to_ticket:
            st_client = StartrekClient(
                oauth_token=sdk2.Vault.data(self.Parameters.token),
            )
            comment_text = '%s\n\n%s' % (title, message)
            st_client.add_comment(self.Parameters.ticket, comment_text)

        if self.Parameters.send_results_to_email:
            self.server.notification(
                subject=title,
                body=message,
                recipients=self.Parameters.emails_or_logins,
                transport=ctn.Transport.EMAIL,
            )

    def on_execute(self):
        # Step 0: Select ammo types
        for shoot_type in self.Parameters.shoot_types:
            self.Context.shoot_data.setdefault(
                shoot_type,
                {
                    'lunapark_link': '',
                    'ammo': None,
                    'subtask': None,
                },
            )

        # Step 1: Generate ammo
        with self.memoize_stage.generate_ammo(commit_on_entrance=False):
            log.info('Starting to generate ammo')
            for shoot_type in self.Context.shoot_data:
                shoot_params = AVAILABLE_SHOOT_TYPES()[shoot_type]
                self.start_generating_ammo(
                    shoot_type=shoot_type,
                    ammo_type=shoot_params['ammo_type'],
                    ammo_count=shoot_params['ammo_count'],
                    try_reuse_ammo=shoot_params.get('reuse_ammo', False) and not self.Parameters.force_recreate_ammo,
                )
            if self.Context.generate_ammo_tasks:
                raise sdk2.WaitTask(
                    self.Context.generate_ammo_tasks,
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                    wait_all=True,
                )

        for shoot_type, shoot_data in self.Context.shoot_data.iteritems():
            log.info('Getting ammo for type ' + shoot_type)
            shoot_data['ammo'] = get_freshest_ammo(
                shoot_type=shoot_type,
                ammo_type=AVAILABLE_SHOOT_TYPES()[shoot_type]['ammo_type'],
            ).id
            log.info('Ammo with %s urls is %s', shoot_type, shoot_data['ammo'])

        # Step 2: Run shooting
        for shoot_type, shoot_data in self.Context.shoot_data.iteritems():
            attempts = AVAILABLE_SHOOT_TYPES()[shoot_type].get('retry_count', 1)
            with self.memoize_stage['shoot_{}'.format(shoot_type)](max_runs=attempts + 1, commit_on_entrance=False) as st:
                log.debug('Entering task shoot_%s, run count is %d', shoot_type, st.runs)
                if shoot_data['subtask']:
                    shoot_data['lunapark_link'] = sdk2.Task[shoot_data['subtask']].Parameters.lunapark_link

                    # if this shooting needs retries and it is not already successful
                    if attempts > 1 and 'try_number' not in shoot_data:
                        stddev = AVAILABLE_SHOOT_TYPES()[shoot_type].get('stddev')

                        # retry if we have more attempts and previous shooting failed
                        if st.runs < attempts and not self._shooting_accepted(shoot_data['lunapark_link'], stddev):
                            shoot_data['subtask'] = None
                            log.info('Restart shooting for type %s, attempt %d', shoot_type, st.runs + 1)
                            self.start_shooting(shoot_type)
                        else:
                            # if we run out of retries or shooting successful, remember it
                            shoot_data['try_number'] = st.runs
                else:
                    log.info('Start shooting for type ' + shoot_type)
                    self.start_shooting(shoot_type)

        # Step 3: Make a report
        with self.memoize_stage.reporting(commit_on_entrance=False):
            if self.Parameters.aggregate_reports:
                self.send_aggregated_report(shoot_types=self.Context.shoot_data.keys())
            else:
                for shoot_type in self.Context.shoot_data:
                    self.start_reporting(shoot_type)
                if self.Context.report_tasks:
                    raise sdk2.WaitTask(
                        self.Context.report_tasks,
                        ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                        wait_all=True,
                    )
