# -*- coding: utf-8 -*-
from datetime import (
    datetime,
    timedelta,
)
from enum import Enum
from itertools import (
    cycle,
    islice,
    product,
)
import json
import logging
import os
from random import sample
import sys
from time import time
from urllib import urlencode
from urlparse import urljoin

import requests
from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.sandboxsdk import svn


log = logging.getLogger(__name__)


DEFAULT_PASSWORD = 'simple123456'

# templates for phantom ammo
REQ_TEMPLATE = (
    "%s %s HTTP/1.1\r\n"
    "%s\r\n"
    "\r\n"
)
REQ_TEMPLATE_WITH_BODY = (
    "%s %s HTTP/1.1\r\n"
    "%s\r\n"
    "Content-Length: %d\r\n"
    "\r\n"
    "%s\r\n"
)

AMMO_TTL = timedelta(days=30)


class AmmoType(Enum):
    PassportBackendRegister = 'passport_backend_register'
    PassportFrontendAuth = 'passport_frontend_auth'
    BlackboxCheckIp = 'blackbox_checkip'
    BlackboxUserinfoSimple = 'blackbox_userinfo_simple'
    BlackboxUserinfoMixedJson = 'blackbox_userinfo_mixed_json'
    BlackboxUserinfoMixedXml = 'blackbox_userinfo_mixed_xml'
    BlackboxLoginMixed = 'blackbox_login_mixed'
    BlackboxSessionidMixed = 'blackbox_sessionid_mixed'
    BlackboxOAuthMixed = 'blackbox_oauth_mixed'
    BlackboxOAuthUniform = 'blackbox_oauth_uniform'
    BlackboxLoginUniform = 'blackbox_login_uniform'
    BlackboxSessionidUniform = 'blackbox_sessionid_uniform'
    OAuthGTPasswordNewNormal = 'oauth_gt_password_new_normal'
    OAuthGTXTokenUpdateNormal = 'oauth_gt_xtoken_update_normal'
    OAuthGTXTokenNewStateless = 'oauth_gt_xtoken_new_stateless'
    VaultTokens = 'vault_tokens'
    VaultTokensHeavy = 'vault_tokens_heavy'
    VaultTokensMixed = 'vault_tokens_mixed'


class PassportAmmo(sdk2.Resource):
    """ Ammo for shooting on passport-api """
    releasable = True
    shoot_type = sdk2.parameters.String(
        'Type of scenario ammo is used for',
        required=True,
    )
    ammo_type = sdk2.parameters.String(
        'Type of ammo',
        required=True,
    )


def try_get_freshest_ammo(shoot_type, ammo_type, max_age=AMMO_TTL):
    resource = PassportAmmo.find(
        state=ctr.State.READY,
        attrs=dict(shoot_type=shoot_type, ammo_type=ammo_type),
        created='%s..%s' % (
            (datetime.now() - max_age).isoformat(),
            datetime.now().isoformat(),
        ),
    ).order(-sdk2.Resource.id).first()
    if not resource:
        return
    else:
        sdk2.ResourceData(resource)
        return resource


class RequestError(Exception):
    pass


class PassportApiBuilder(object):
    def __init__(self, passport_api_host='http://passport-load-internal.yandex.ru', consumer='passport'):
        super(PassportApiBuilder, self).__init__()
        self.passport_api_host = passport_api_host
        self.consumer = consumer

    def _bundle_api_request(self, path, method='GET', params=None, data=None, headers=None):
        params = params or {}
        params.setdefault('consumer', self.consumer)
        try:
            resp = requests.request(
                url=urljoin(self.passport_api_host, path),
                method=method,
                params=params,
                data=data,
                headers=headers,
            )
            if resp.status_code != 200:
                raise RequestError('Bad response status: %s' % resp.status_code)
            rv = json.loads(resp.content)
        except (requests.RequestException, ValueError) as e:
            raise RequestError(e)

        if rv.get('status') != 'ok':
            errors = rv.get('errors', [])
            raise RequestError(errors or rv)
        return rv

    def create_user(self, login, password=DEFAULT_PASSWORD, **kwargs):
        rv = self._bundle_api_request(
            path='/1/bundle/account/register/',
            method='POST',
            data=dict(
                login=login,
                password=password,
                **kwargs
            ),
            headers={
                'Ya-Consumer-Client-Ip': '127.0.0.1',
                'Ya-Client-User-Agent': 'sandbox-tank',
            },
        )
        return {
            'uid': rv['uid'],
            'login': login,
            'password': password,
        }

    def generate_login(self):
        return 'tank-%.3f' % time()

    def generate_logins(self, count):
        login_template = 'tank-%.3f-%%d' % time()
        return [login_template % i for i in xrange(count)]

    def create_users(self, count):
        users = []
        for login in self.generate_logins(count):
            try:
                user = self.create_user(login=login)
                users.append(user)
            except Exception as e:
                log.warning('Failed to create user %s: %s', login, e)
        if not users:
            raise RuntimeError('Failed to create any users')
        return users


class OAuthBuilder(object):
    def __init__(self, oauth_host='http://oauth-load-internal.yandex.ru'):
        super(OAuthBuilder, self).__init__()
        self.oauth_host = oauth_host

    def _token_request(self, data):
        try:
            resp = requests.request(
                url=urljoin(self.oauth_host, '/token'),
                method='POST',
                data=data,
            )
            rv = json.loads(resp.content)
        except (requests.RequestException, ValueError) as e:
            raise RequestError(e)

        if 'access_token' not in rv:
            raise RequestError(rv)
        return rv

    def issue_token(self, client_id, client_secret, login, password):
        rv = self._token_request({
            'grant_type': 'password',
            'client_id': client_id,
            'client_secret': client_secret,
            'username': login,
            'password': password,
        })
        return rv['access_token']

    def fill_users_with_tokens(self, users, client_id, client_secret):
        for user in users:
            user['token'] = self.issue_token(
                client_id=client_id,
                client_secret=client_secret,
                login=user['login'],
                password=user['password'],
            )
        return users


BLACKBOX_TEST_IP_LIST = [
    '1.1.1.1', '2.2.2.2',  # v4 external
    '77.88.19.12', '5.255.255.50',  # v4 internal
    '2a00:1148:db00:0:b0b0::1', '2a00:1450:4010:c0a::71',  # v6 external
    '2a02:6b8:0:40c:e866:52e1:a92e:d951', '2a02:6b8:a::a'  # v6 internal
]

BLACKBOX_TEST_UIDS = range(2000000, 2000042) + range(1130000000000001, 1130000000000011)
BLACKBOX_TEST_LOGINS = [
    'kostik', 'pupkin-vasily', 'sv@dnn.ru', 'kraus1@rambler.ru', 'bb-soap-no-sid2', 'bb-test-bare-sid2',
    'bb-test-all-different-soaps', 'sid2-alias@yandex.com', 'sid16-alias@narod.ru', 'bb-test-have-foreign-soap',
    'bb-test-no-sid2-have-foreign', 'bb-test-gs', 'test-gs@galatasaray.net', 'malice-gs', 'malice',
    u'malice1@яндекс.рф', 'malice@xn--d1acpjx3f.xn--p1ai', 'malice-gs@galatasaray.net', 'malice2', 'ko-shaki',
    'bbtst_nopassword', 'bbtst_disabled', 'junit-test', '79161603189', 'junit-test@yandex.com.tr', 'dot-user',
    '79161111111', 'dot-user-gs@galatasaray.com', 'bbtst_staff', 'bbtst_disabled_staff', 'disabled_yateam',
    'bbtst_changerequired', '79162222222', 'malice@yandex.ru', 'bbtst_nomail', 'bbtst_nomail@galatasaray.net',
    'bbtst_nomail@kino.poisk.ru', 'bbtst_expired_password', 'addr-master', '79032426488', 'addr-default',
    'addr-default-native', 'addr-external-default', 'addr-broken-default', 'addr-default-phone', '70001112233',
    'addr-default-phone-denorm', '79637778899', 'alias-master', 'phne-deadbeef', '78001010101',
    'alias-narodmail@narod.ru', 'alias-narod@narod.ru', 'alias-lite@mail.ru', 'uid-deadbeef', 'need-pwd-create',
    'need-pwd-change', '79032426489', 'killed1', 'killed1@domain.tst', 'fire@most.ua', 'fire@infojet.net.ua',
    'aacoii@most.ua', 'bvizsu@most.ua', 'dqobho@most.ua', 'ffnnov@most.ua', 'hafajn@most.ua', 'irrqci@most.ua',
    'kgkmxd@most.ua', 'lyaecr@most.ua', 'nszhjd@most.ua', 'pnrmry@most.ua', 'rionqy@most.ua', 'tbkyxf@most.ua',
    'uyntzq@most.ua', 'wtrwxy@most.ua', 'ymvqoq@most.ua', 'zzqtks@most.ua', 'test43@most.ua', 'tester@most.ua',
    'mememe@most.ua', 'test44@most.ua', 'aacoii@infojet.net.ua', 'bvizsu@infojet.net.ua',
    'dqobho@infojet.net.ua', 'ffnnov@infojet.net.ua', 'hafajn@infojet.net.ua', 'irrqci@infojet.net.ua',
    'kgkmxd@infojet.net.ua', 'lyaecr@infojet.net.ua', 'nszhjd@infojet.net.ua', 'pnrmry@infojet.net.ua',
    'rionqy@infojet.net.ua', 'tbkyxf@infojet.net.ua', 'uyntzq@infojet.net.ua', 'wtrwxy@infojet.net.ua',
    'ymvqoq@infojet.net.ua', 'zzqtks@infojet.net.ua', 'test43@infojet.net.ua', 'tester@infojet.net.ua',
    'mememe@infojet.net.ua', 'test44@infojet.net.ua', 'robbitter-1343380562@imap1.yandex.ru',
    'blocked-pdd-u@imap1.yandex.ru', 'qwe@denis-anikin.ru', 'qwe@denis-anikin4.ru',
    'robbitter-4480829663@tst1.ts', 'dot.user@denis-anikin.ru', 'dot.user@denis-anikin4.ru',
]


class BlackboxRequestGenerator(object):
    def __init__(self):
        super(BlackboxRequestGenerator, self).__init__()

    def generate_checkip(self):
        return ['method=checkip&nets=yandexusers&ip=%s' % ip for ip in BLACKBOX_TEST_IP_LIST]

    def generate_userinfo_simple(self):
        return ['method=userinfo&userip=1.2.3.4&uid=%d' % uid for uid in BLACKBOX_TEST_UIDS]

    def generate_additional_args(self, request):
        personal_data = ['', '&regname=yes', '&regname=yes&aliases=all&suids=all']
        emails_data = ['', '&emails=getall', '&getemails=all&email_attributes=all']
        phones_data = ['', '&getphones=all&phone_attributes=103,104,105,106,107,108']
        attrs_data = ['', '&attributes=1,24,34', '&attributes=131,14,10,38,39,48,1013,1008,1007,1003,1009,1015,13,66,67,110,111']
        dbfields_data = ['', '&dbfields=account_info.birth_date.uid,subscription.login.669,subscription.login.2,'
                         ',account_info.sex.uid,account_info.fio.uid,account_info.email.uid,account_info.birth_date.uid,subscription.host_id.36,userinfo.lang.uid,subscription.suid.669']

        return [request + ''.join(t) for t in product(personal_data, emails_data, phones_data, attrs_data, dbfields_data)]

    def generate_userinfo_mixed(self):
        # userinfo by login (single user)
        request_base = ['method=userinfo&userip=1.2.3.5&login=%s' % login for login in BLACKBOX_TEST_LOGINS]

        # userinfo by uid (single user)
        request_base.extend(['method=userinfo&userip=1.2.3.6&uid=%d' % uid for uid in BLACKBOX_TEST_UIDS])

        # userinfo by uids (5 users in each request)
        for _ in xrange(10):
            request_base.append('method=userinfo&userip=1.2.3.5&uid=%s' % ','.join(str(uid) for uid in sample(BLACKBOX_TEST_UIDS, 5)))

        # userinfo by uids (10 users in each request)
        for _ in xrange(5):
            request_base.append('method=userinfo&userip=1.2.3.10&uid=%s' % ','.join(str(uid) for uid in sample(BLACKBOX_TEST_UIDS, 10)))

        return sum([self.generate_additional_args(r) for r in request_base], [])

    def generate_login_mixed(self):
        request_base = ['method=login&userip=1.2.10.%d&login=%s&password=qwerty' % (num, login) for num, login in enumerate(BLACKBOX_TEST_LOGINS, 1)]

        request_base.extend(['method=login&userip=1.2.11.%d&uid=%d&password=qwerty' % (num, uid) for num, uid in enumerate(BLACKBOX_TEST_UIDS, 1)])

        return sum([self.generate_additional_args(r) for r in request_base], [])

    def call_blackbox_method(self, name, params):
        try:
            resp = requests.get("http://pass-load.sezam.yandex.net/blackbox?method=%s&userip=1.1.1.1&%s&format=json" % (name, params))

            if resp.status_code != 200:
                raise RequestError('Bad response status: %s' % resp.status_code)

            rv = json.loads(resp.content)
        except (requests.RequestException, ValueError) as e:
            raise RequestError(e)

        if "exception" in rv:
            raise RequestError(rv)

        return rv

    def add_to_session(self, uid, session=None):
        if session:
            r = self.call_blackbox_method("editsession", "op=add&sessionid=%s&host=yandex.ru&uid=%s" % (session, uid))
        else:
            r = self.call_blackbox_method("createsession", "uid=%s&ttl=5&is_stress&keyspace=yandex.ru" % uid)

        if "new-session" not in r:
            raise RequestError(r)

        return r["new-session"]["value"]

    def create_session(self, uids):
        session = None
        for uid in uids:
            session = self.add_to_session(uid, session=session)

        return session

    def generate_sessionid_mixed(self):
        # sessionid with single user
        request_base = ['method=sessionid&userip=1.2.20.1&sessionid=%s&host=yandex.ru&multisession=yes&full_info=yes&authid=yes' %
                        self.create_session([uid]) for uid in BLACKBOX_TEST_UIDS]

        # sessionid with 5 users each
        for _ in xrange(10):
            request_base.append('method=sessionid&userip=1.2.20.5&sessionid=%s&host=yandex.ru&multisession=yes&full_info=yes&authid=yes' %
                                self.create_session(sample(BLACKBOX_TEST_UIDS, 5)))

        # sessionid with 10 users each
        for _ in xrange(5):
            request_base.append('method=sessionid&userip=1.2.20.10&sessionid=%s&host=yandex.ru&multisession=yes&full_info=yes&authid=yes' %
                                self.create_session(sample(BLACKBOX_TEST_UIDS, 10)))

        return sum([self.generate_additional_args(r) for r in request_base], [])

    def create_oauth_token(self, data):
        try:
            resp = requests.request(
                url='http://oauth-load-internal.yandex.ru/token',
                method='POST',
                data=data,
            )
            rv = json.loads(resp.content)
        except (requests.RequestException, ValueError) as e:
            raise RequestError(e)

        if 'access_token' not in rv:
            raise RequestError(rv)

        return rv['access_token']

    def create_client_token(self, client_id, client_secret):
        return self.create_oauth_token({
            'grant_type': 'client_credentials',
            'client_id': client_id,
            'client_secret': client_secret
        })

    def create_tokens(self, login, client_id, client_secret):
        tokens = []

        try:
            db_token = self.create_oauth_token({
                'grant_type': 'password',
                'client_id': client_id,
                'client_secret': client_secret,
                'username': login,
                'password': 'qwerty'
            })

            st_token = self.create_oauth_token({
                'grant_type': 'password',
                'client_id': client_id,
                'client_secret': client_secret,
                'username': login,
                'password': 'qwerty',
                'x_stateless': 'true'
            })

            st_by_xtoken = self.create_oauth_token({
                'grant_type': 'x-token',
                'client_id': client_id,
                'client_secret': client_secret,
                'access_token': db_token,
                'x_stateless': 'true'
            })

            tokens.extend([db_token, st_token, st_by_xtoken])
        except Exception as e:
            log.debug('Failed to generate token for login %s: %s' % (login, str(e)))

        return tokens

    def generate_oauth_mixed(self, client_id, client_secret):
        request_base = ['method=oauth&userip=1.2.3.0&oauth_token=%s' % self.create_client_token(client_id, client_secret)]

        for login in BLACKBOX_TEST_LOGINS:
            tokens = self.create_tokens(login, client_id, client_secret)
            request_base.extend(['method=oauth&userip=1.2.3.1&oauth_token=%s' % token for token in tokens])
            request_base.extend(['method=oauth&userip=1.2.3.1&oauth_token=%s&oauth_token_attributes=all&oauth_client_attributes=all' % token for token in tokens])

        return sum([self.generate_additional_args(r) for r in request_base], [])

    def generate_oauth_uniform(self, client_id, client_secret):

        token = self.create_oauth_token({
            'grant_type': 'password',
            'client_id': client_id,
            'client_secret': client_secret,
            'device_id': 'a43500eb0d9b1d68a664e2ab374b8b45',
            'x_meta': 'OAuth token meta info, which is a string that may be a bit long',
            'username': 'bbtst_staff',
            'password': 'qwerty'
        })

        request = 'method=oauth&userip=1.2.3.4&oauth_token=' + token
        request += '&oauth_token_attributes=all&oauth_client_attributes=all'
        request += '&regname=yes&aliases=all&suids=all'
        request += '&emails=getall&getemails=all&email_attributes=all&getphones=all&phone_attributes=103,104,105,106,107,108'
        request += '&attributes=1,24,34,36,27,28,29,30,1001,131,14,10,38,39,48,1013,1008,1007,1003,1009,1015,13,66,67,110,111'
        request += '&dbfields=account_info.birth_date.uid,subscription.login.669,subscription.login.2,'
        ',account_info.sex.uid,account_info.fio.uid,account_info.email.uid,account_info.birth_date.uid,subscription.host_id.36,userinfo.lang.uid,subscription.suid.669'

        return [request]

    def generate_login_uniform(self):

        request = 'method=login&userip=1.2.3.4&login=bbtst_staff&password=qwerty'
        request += '&regname=yes&aliases=all&suids=all'
        request += '&emails=getall&getemails=all&email_attributes=all&getphones=all&phone_attributes=103,104,105,106,107,108'
        request += '&attributes=1,24,34,36,27,28,29,30,1001,131,14,10,38,39,48,1013,1008,1007,1003,1009,1015,13,66,67,110,111'
        request += '&dbfields=account_info.birth_date.uid,subscription.login.669,subscription.login.2,'
        ',account_info.sex.uid,account_info.fio.uid,account_info.email.uid,account_info.birth_date.uid,subscription.host_id.36,userinfo.lang.uid,subscription.suid.669'

        return [request]

    def generate_sessionid_uniform(self):

        request = 'method=sessionid&userip=1.2.3.4&host=yandex.ru&sessionid=%s' % self.create_session([2000017])
        request += '&multisession=yes&full_info=yes&authid=yes'
        request += '&regname=yes&aliases=all&suids=all'
        request += '&emails=getall&getemails=all&email_attributes=all&getphones=all&phone_attributes=103,104,105,106,107,108'
        request += '&attributes=1,24,34,36,27,28,29,30,1001,131,14,10,38,39,48,1013,1008,1007,1003,1009,1015,13,66,67,110,111'
        request += '&dbfields=account_info.birth_date.uid,subscription.login.669,subscription.login.2,'
        ',account_info.sex.uid,account_info.fio.uid,account_info.email.uid,account_info.birth_date.uid,subscription.host_id.36,userinfo.lang.uid,subscription.suid.669'

        return [request]


class PassportGenerateAmmo(sdk2.Task):
    """ Generates ammo for shooting """

    class Parameters(sdk2.Task.Parameters):
        shoot_type = sdk2.parameters.String(
            'Тип стрельбы',
            required=True,
        )
        with sdk2.parameters.RadioGroup('Тип патронов') as ammo_type:
            for ammo_enum_type in AmmoType:
                setattr(
                    ammo_type.values,
                    ammo_enum_type.value,
                    ammo_type.Value(ammo_enum_type.value),
                )
        ammo_count = sdk2.parameters.Integer(
            'Размер ленты',
            default=1000,
            required=True,
        )
        try_reuse_ammo = sdk2.parameters.Bool(
            'Переиспользовать существующую ленту (при наличии)',
            default=False,
        )
        oauth_credentials_vault_key = sdk2.parameters.String(
            'Vault record name with oauth client_id and secret',
            description='Vault record name with oauth client_id and secret',
            default='passport-oauth-load-client-credentials',
            required=False,
        )

    def make_bullet(self, method, url, post_args, extra_headers=None):
        """ https://yandextank.readthedocs.io/en/latest/tutorial.html#request-style """
        headers = {
            'User-Agent': 'sandbox-tank',
        }
        headers.update(extra_headers or {})
        if post_args:
            headers.setdefault('Content-Type', 'application/x-www-form-urlencoded')
        serialized_headers = '\r\n'.join(['%s: %s' % (key, value) for key, value in headers.iteritems()])

        if not post_args:
            req = REQ_TEMPLATE % (method, url, serialized_headers)
        else:
            if 'json' in headers.get('Content-Type'):
                body = json.dumps(post_args)
            else:
                body = urlencode(post_args)
            req = REQ_TEMPLATE_WITH_BODY % (method, url, serialized_headers, len(body), body)

        return '%d\n%s' % (len(req), req)

    def make_ammo(self, ammo_type, count):
        oauth_client_id, _, oauth_client_secret = sdk2.Vault.data(self.Parameters.oauth_credentials_vault_key).partition(':')

        bullets = []
        if ammo_type == AmmoType.PassportFrontendAuth.value:
            max_user_count = 100
            users = PassportApiBuilder().create_users(min(max_user_count, count / 10))
            for user in islice(cycle(users), 0, count):
                bullets.append(self.make_bullet(
                    method='POST',
                    url='/auth',
                    post_args={
                        'login': user['login'],
                        'passwd': user['password'],
                    },
                    extra_headers={
                        'Host': 'passport-load.yandex.ru',
                        'Connection': 'Keep-Alive',
                    },
                ))
        elif ammo_type == AmmoType.PassportBackendRegister.value:
            for login in PassportApiBuilder().generate_logins(count):
                bullets.append(self.make_bullet(
                    method='POST',
                    url='/1/bundle/account/register/?consumer=passport',
                    post_args={
                        'login': login,
                        'password': DEFAULT_PASSWORD,
                    },
                    extra_headers={
                        'Host': 'passport-load-internal.yandex.ru',
                        'Connection': 'Keep-Alive',
                        'Ya-Consumer-Client-Ip': '127.0.0.1',
                        'Ya-Client-User-Agent': 'sandbox-tank',
                    },
                ))
        elif ammo_type == AmmoType.BlackboxCheckIp.value:
            for req in BlackboxRequestGenerator().generate_checkip():
                bullets.append('/blackbox?%s\n' % req)
        elif ammo_type == AmmoType.BlackboxUserinfoSimple.value:
            for req in BlackboxRequestGenerator().generate_userinfo_simple():
                bullets.append('/blackbox?%s\n' % req)
        elif ammo_type == AmmoType.BlackboxUserinfoMixedJson.value:
            for req in BlackboxRequestGenerator().generate_userinfo_mixed():
                bullets.append('/blackbox?%s&format=json\n' % req)
        elif ammo_type == AmmoType.BlackboxUserinfoMixedXml.value:
            for req in BlackboxRequestGenerator().generate_userinfo_mixed():
                bullets.append('/blackbox?%s&format=xml\n' % req)
        elif ammo_type == AmmoType.BlackboxLoginMixed.value:
            for req in BlackboxRequestGenerator().generate_login_mixed():
                bullets.append('/blackbox?%s&format=json\n' % req)
                bullets.append('/blackbox?%s&format=xml\n' % req)
        elif ammo_type == AmmoType.BlackboxSessionidMixed.value:
            for req in BlackboxRequestGenerator().generate_sessionid_mixed():
                bullets.append('/blackbox?%s&format=json\n' % req)
                bullets.append('/blackbox?%s&format=xml\n' % req)
        elif ammo_type == AmmoType.BlackboxOAuthMixed.value:
            for req in BlackboxRequestGenerator().generate_oauth_mixed(oauth_client_id, oauth_client_secret):
                bullets.append('/blackbox?%s&format=json\n' % req)
                bullets.append('/blackbox?%s&format=xml\n' % req)
        elif ammo_type == AmmoType.BlackboxOAuthUniform.value:
            for req in BlackboxRequestGenerator().generate_oauth_uniform(oauth_client_id, oauth_client_secret):
                bullets.append('/blackbox?%s&format=json\n' % req)
        elif ammo_type == AmmoType.BlackboxLoginUniform.value:
            for req in BlackboxRequestGenerator().generate_login_uniform():
                bullets.append('/blackbox?%s&format=json\n' % req)
        elif ammo_type == AmmoType.BlackboxSessionidUniform.value:
            for req in BlackboxRequestGenerator().generate_sessionid_uniform():
                bullets.append('/blackbox?%s&format=xml\n' % req)
        elif ammo_type in (
            AmmoType.OAuthGTPasswordNewNormal.value,
            AmmoType.OAuthGTXTokenUpdateNormal.value,
            AmmoType.OAuthGTXTokenNewStateless.value,
        ):
            max_user_count = 100
            users = PassportApiBuilder().create_users(min(max_user_count, count / 10))
            OAuthBuilder().fill_users_with_tokens(users, oauth_client_id, oauth_client_secret)

            args = dict(
                client_id=oauth_client_id,
                client_secret=oauth_client_secret,
            )

            for bullet_id, user in enumerate(islice(cycle(users), 0, count)):
                if ammo_type == AmmoType.OAuthGTPasswordNewNormal.value:
                    args.update(
                        grant_type='password',
                        username=user['login'],
                        password=user['password'],
                    )
                else:
                    args.update(
                        grant_type='x-token',
                        access_token=user['token'],
                    )

                if ammo_type == AmmoType.OAuthGTPasswordNewNormal.value:
                    args.update(device_id='device_%s' % bullet_id)
                elif ammo_type == AmmoType.OAuthGTXTokenUpdateNormal.value:
                    args.update(x_meta='meta_%s' % bullet_id)
                elif ammo_type == AmmoType.OAuthGTXTokenNewStateless.value:
                    args.update(x_stateless='true')

                bullets.append(self.make_bullet(
                    method='POST',
                    url='/token',
                    post_args=args,
                    extra_headers={
                        'Host': 'oauth-load-internal.yandex.ru',
                        'Connection': 'Keep-Alive',
                    },
                ))
        elif ammo_type in (
            AmmoType.VaultTokens.value,
            AmmoType.VaultTokensHeavy.value,
            AmmoType.VaultTokensMixed.value,
        ):
            sys.path.extend([
                svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/library/python/vault_client/"),
                svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/contrib/python/"),
            ])
            from vault_client import VaultClient
            passport_builder = PassportApiBuilder()
            login = passport_builder.generate_login()
            passport_builder.create_user(login)
            oauth_token = OAuthBuilder().issue_token(
                oauth_client_id, oauth_client_secret, login, DEFAULT_PASSWORD,
            )
            vault_client = VaultClient(
                host='http://vault-load.passport.yandex.net:9500',
                check_status=False,
                authorization=oauth_token
            )
            light_secret = os.urandom(24 / 2).encode('hex')
            heavy_secret = os.urandom(2048 / 2).encode('hex')
            secret_uuid = vault_client.create_secret('load-test-secret')
            secret_light_version_uuid = vault_client.create_secret_version(secret_uuid, {'key': light_secret})
            secret_heavy_version_uuid = vault_client.create_secret_version(secret_uuid, {'key': heavy_secret})
            token, token_uuid = vault_client.create_token(secret_uuid)

            if ammo_type in (
                AmmoType.VaultTokens.value,
                AmmoType.VaultTokensMixed.value,
            ):
                bullets.append(self.make_bullet(
                    method='POST',
                    url='/1/tokens/',
                    post_args={
                        'tokenized_requests': [{
                            'token': token,
                            'secret_version': secret_light_version_uuid,
                        }],
                    },
                    extra_headers={
                        'Host': 'vault-load.passport.yandex.net',
                        'Connection': 'Keep-Alive',
                        'Content-Type': 'application/json',
                    },
                ))
            if ammo_type in (
                AmmoType.VaultTokensHeavy.value,
                AmmoType.VaultTokensMixed.value,
            ):
                bullets.append(self.make_bullet(
                    method='POST',
                    url='/1/tokens/',
                    post_args={
                        'tokenized_requests': [{
                            'token': token,
                            'secret_version': secret_heavy_version_uuid,
                        }],
                    },
                    extra_headers={
                        'Host': 'vault-load.passport.yandex.net',
                        'Connection': 'Keep-Alive',
                        'Content-Type': 'application/json',
                    },
                ))
        else:
            raise NotImplementedError('Don\'t know how to make ammo for %s' % ammo_type)

        return ''.join(bullets)

    def on_execute(self):
        shoot_type = self.Parameters.shoot_type
        ammo_type = self.Parameters.ammo_type
        if self.Parameters.try_reuse_ammo:
            resource = try_get_freshest_ammo(shoot_type, ammo_type)
            if resource:
                log.debug('Found existing ammo (resource %s), not generating new one', resource.id)
                return

        ammo = self.make_ammo(ammo_type=ammo_type, count=self.Parameters.ammo_count)
        resource = PassportAmmo(self, '%s urls' % shoot_type, '%s.ammo.txt' % ammo_type)
        resource.shoot_type = shoot_type
        resource.ammo_type = ammo_type
        resource_data = sdk2.ResourceData(resource)
        resource_data.path.write_bytes(ammo.encode('utf8'))
