# -*- coding: utf-8 -*-


from sandbox import sdk2
import sandbox.common as common


from datetime import datetime, timedelta
import logging

from sandbox.sandboxsdk import environments

INFO_DELAY_HOURS = 8
ERROR_DELAY_HOURS = 12
RAW_DELAY_HOURS = 5

INFO_EVENT_TYPE = 'info'
ERROR_EVENT_TYPE = 'error'
RAW_EVENT_TYPE = 'raw'

EVENT_TYPES = [RAW_EVENT_TYPE, INFO_EVENT_TYPE, ERROR_EVENT_TYPE]

DELAY_FROM_EVENT = {
    INFO_EVENT_TYPE: INFO_DELAY_HOURS,
    ERROR_EVENT_TYPE: ERROR_DELAY_HOURS,
    RAW_EVENT_TYPE: RAW_DELAY_HOURS
}

LOG_PATH_PREFIX = '//home/payplatform/logs'
RAW_LOG_PATH_PREFIX = '//logs'
RAW_LOG_PATH_SUFFIX = '30min'
LOG_NAMES = ['paysys-trust-stable-yb-trust-payments',
             'paysys-trust-stable-yb-trust-payments-ng',
             'paysys-trust-stable-yb-trust-payments-xg',
             'paysys-trust-stable-yb-trust-worker',
             'paysys-trust-stable-yb-trust-worker-ng',
             'paysys-trust-stable-yb-trust-worker-xg',
             'paysys-trust-stable-yb-trust-paysys',
             'paysys-trust-stable-yb-trust-paysys-bindings',
             'paysys-trust-stable-yb-trust-paysys-callbacks',
             'paysys-trust-stable-yb-trust-paysys-mobile',
             'paysys-trust-stable-yb-trust-paysys-web',
]

RECIPIENTS_ERRORS = ['payplatform-robot-alerts@yandex-team.ru']
RECIPIENTS_INFO = ['payplatform-robot-info@yandex-team.ru']

YT_TABLE_URL_PREFIX = 'https://yt.yandex-team.ru/hahn/navigation?path={path}'
YT_PROXY = 'hahn'
YT_SECRET_NAME = 'log-enrichment-monitoring-yt-token'

TIME_FORMAT = '%Y-%m-%dT%H:%M:%S'


class NotifyOnEnrichmentDelay(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt', version='0.9.17'),
        ]

    class Parameters(sdk2.Parameters):
        pass

    class Context(sdk2.Context):
        pass

    @staticmethod
    def get_table_link(table_path):
        return YT_TABLE_URL_PREFIX.format(path=table_path)

    @staticmethod
    def get_empty_table_notification_name(raw=False):
        return 'EMPTY RAW TABLE' if raw else 'EMPTY TABLE'

    @staticmethod
    def get_no_table_notification_name(hours, raw=False):
        return 'NO RAW TABLE IN {x} HOURS'.format(x=hours) if raw else 'NO TABLE IN {x} HOURS'.format(x=hours)

    def do_notify_on_email(self, subject, text, info=False):
        self.server.notification(
            subject=subject,
            body=text,
            recipients=RECIPIENTS_ERRORS if not info else RECIPIENTS_INFO,
            transport=common.types.notification.Transport.EMAIL
        )

    def send_raw_no_table_email_err(self, table):
        text = 'Внимание! Необогащенная таблица {table} не появилась за {hours} часа!\n{link}'.format(
            table=table, hours=RAW_DELAY_HOURS, link=self.get_table_link(table)
        )
        self.do_notify_on_email(subject=self.get_no_table_notification_name(RAW_DELAY_HOURS, raw=True), text=text)

    def send_no_table_email_err(self, table):
        text = 'Внимание! Обогащенная таблица {table} не появилась за {hours} часов!\n{link}'.format(
            table=table, hours=ERROR_DELAY_HOURS, link=self.get_table_link(table)
        )
        self.do_notify_on_email(subject=self.get_no_table_notification_name(ERROR_DELAY_HOURS), text=text)

    def send_no_table_email_info(self, table):
        text = 'Внимание! Обогащенная таблица {table} не появилась за {hours} часов!\n{link}'.format(
            table=table, hours=INFO_DELAY_HOURS, link=self.get_table_link(table)
        )
        self.do_notify_on_email(subject=self.get_no_table_notification_name(INFO_DELAY_HOURS), text=text, info=True)

    def send_raw_empty_table_email_err(self, table):
        text = 'Внимание! Необогащенная таблица {table} пуста\n{link}'.format(table=table, link=self.get_table_link(table))
        self.do_notify_on_email(subject=self.get_empty_table_notification_name(raw=True), text=text)

    def send_empty_table_email_err(self, table):
        text = 'Внимание! Обогащенная таблица {table} пуста!\n{link}'.format(table=table, link=self.get_table_link(table))
        self.do_notify_on_email(subject=self.get_empty_table_notification_name(), text=text)

    def check_delay_and_fullness(self, event_type):
        checking_time = self.get_check_time(event_type)
        logging.info('Notification type = {ev_type}. Checking time: {ch_time}'.format(ev_type=event_type, ch_time=checking_time))

        raw_flag = True if event_type == RAW_EVENT_TYPE else False
        checking_tables = self.get_tables_for_check(checking_time, raw=raw_flag)
        logging.info('Notification type = {ev_type}. Checking tables: {ch_tables}'.format(ev_type=event_type, ch_tables=checking_tables))

        if event_type == RAW_EVENT_TYPE:
            for _, raw_table in checking_tables:
                if not self.yt_client.exists(raw_table):
                    self.send_raw_no_table_email_err(raw_table)
                elif self.yt_client.row_count(raw_table) == 0:
                    self.send_raw_empty_table_email_err(raw_table)
        elif event_type == INFO_EVENT_TYPE:
            for table, raw_table in checking_tables:
                if self.yt_client.exists(raw_table) and not self.yt_client.exists(table):
                    self.send_no_table_email_info(table)
        elif event_type == ERROR_EVENT_TYPE:
            for table, raw_table in checking_tables:
                if self.yt_client.exists(raw_table) and not self.yt_client.exists(table):
                    self.send_no_table_email_err(table)
                elif self.yt_client.exists(raw_table) \
                        and self.yt_client.exists(table) \
                        and self.yt_client.row_count(raw_table) > 0 \
                        and self.yt_client.row_count(table) == 0:
                    self.send_empty_table_email_err(table)

    @staticmethod
    def get_check_time(event_type):
        result = (datetime.today() - timedelta(hours=DELAY_FROM_EVENT[event_type])).replace(second=0, microsecond=0)
        if result.minute > 30:
            result = result.replace(minute=30)
        elif result.minute < 30:
            result = result.replace(minute=0)

        return result

    @staticmethod
    def get_yt_client():
        import yt.wrapper as yt
        proxy = YT_PROXY
        token = sdk2.Vault.data(YT_SECRET_NAME)
        return yt.YtClient(proxy=proxy, token=token)

    @staticmethod
    def get_tables_for_check(check_datetime, raw=False):
        from yt.wrapper import ypath_join
        table_name = check_datetime.strftime('%Y-%m-%dT%H:%M:%S')
        return [(ypath_join(LOG_PATH_PREFIX, log_name, table_name) if not raw else None,
                 ypath_join(RAW_LOG_PATH_PREFIX, log_name, RAW_LOG_PATH_SUFFIX, table_name))
                for log_name in LOG_NAMES]

    def on_execute(self):
        logging.info('Task: Start')
        self.yt_client = self.get_yt_client()

        for event_type in EVENT_TYPES:
            self.check_delay_and_fullness(event_type)

        logging.info('Task: End')
