import logging
import requests
from sandbox import sdk2
from sandbox import common
from sandbox.projects.common import link_builder
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt


PARENT_TASK_PARAM_NAME = """Parent task
(NOTICE: By default the latest stable release will be picked,
WARNING: Please keep in mind that by default children tasks are not shown)
"""


class PaysysDockerDummyResource(sdk2.resource.AbstractResource):
    releasable = True
    releasers = [
        "PAYSYSADMIN",
        "PAYSYS",
    ]
    release_subscribers = ["PAYSYSADMIN"]


class PaysysDockerRegistryManifestResource(sdk2.resource.AbstractResource):
    releasable = True
    releasers = [
        "PAYSYSADMIN",
        "PAYSYS",
        "OEBSADMIN"
    ]
    release_subscribers = ["PAYSYSADMIN"]


class PaysysDockerParameters(sdk2.Parameters):
    description = "Build docker images recursively"

    parent_task = None

    vcs_commit = sdk2.parameters.String(
        "VCS commit", default=None,
    )
    vcs_branch = sdk2.parameters.String(
        "VCS branch (for arcadia master will be replaced with trunk/arcadia)",
        default="master",
    )
    registry_extra_tags = sdk2.parameters.List(
        "Registry extra tags",
    )


class PaysysDocker(sdk2.Task):
    _docker_root_image = False
    _docker_build_args = {}

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 2048
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(PaysysDockerParameters):
        docker_registry_login = sdk2.parameters.String(
            "Registry login", default="", required=True,
        )
        docker_token_vault_name = sdk2.parameters.String(
            "Registry token vault name",
            default="",
            required=True,
        )
        docker_token_vault_owner = sdk2.parameters.String(
            "Registry token vault owner", default="", required=True,
        )
        docker_ssh_vault_name = sdk2.parameters.String(
            "SSH vault name", default="", required=True,
        )
        docker_ssh_vault_owner = sdk2.parameters.String(
            "SSH vault owner", default="", required=True,
        )

    def _set_task_tags(self):
        self.hint(self._docker_registry_path.replace("/", "_"))

    def _gen_registry_tags(self):
        registry_tag = self.id
        self.Context.built_image = "registry.yandex.net/{}/{}:{}".format(
            self._docker_registry_path,
            self._docker_app_name,
            registry_tag,
        )

        self.Context.registry_tags = \
            [registry_tag] + self.Parameters.registry_extra_tags

        self.Context.save()

    def _save_manifest_from_registry(self):
        vault = sdk2.task.VaultItem(
            self.Parameters.docker_token_vault_owner,
            self.Parameters.docker_token_vault_name,
        )
        registry_oauth_token = vault.data()

        registry_location = self.Context.built_image.split("/", 1)[0]
        registry_image, registry_tag = \
            self.Context.built_image.split("/", 1)[1].split(":")

        manifest_result = requests.get(
            "https://{}/v2/{}/manifests/{}".format(
                registry_location,
                registry_image,
                registry_tag,
            ),
            headers={
                "Authorization": "OAuth {}".format(registry_oauth_token),
                "Accept": "application/vnd.docker.distribution.manifest.v2+json",
            },
        )

        manifest_result.raise_for_status()

        resource = sdk2.ResourceData(
            PaysysDockerRegistryManifestResource(
                self, "registry_manifest", "manifest.json"
            ),
        )

        resource.path.write_text(manifest_result.text)

        resource.ready()

    def _set_registry_tag(self, new_registry_tag):
        vault = sdk2.task.VaultItem(
            self.Parameters.docker_token_vault_owner,
            self.Parameters.docker_token_vault_name,
        )
        registry_oauth_token = vault.data()

        registry_location = self.Context.built_image.split("/", 1)[0]
        registry_image, registry_tag = \
            self.Context.built_image.split("/", 1)[1].split(":")

        logging.info(
            "Setting registry tag for {}/{} to {}".format(
                registry_location,
                registry_image,
                new_registry_tag,
            )
        )

        resource = sdk2.ResourceData(
            PaysysDockerRegistryManifestResource(
                self, "registry_manifest", "manifest.json"
            ),
        )

        manifest_result = requests.put(
            "https://{}/v2/{}/manifests/{}".format(
                registry_location,
                registry_image,
                new_registry_tag,
            ),
            headers={
                "Authorization": "OAuth {}".format(registry_oauth_token),
                "Content-type": "application/vnd.docker.distribution.manifest.v2+json",
            },
            data=resource.path.read_text(),
        )

        manifest_result.raise_for_status()

        self.set_info(
            "Set registry tag: {}/{}:{}".format(
                registry_location,
                registry_image,
                new_registry_tag,
            ),
        )

    def _create_build_task(self):
        # This method can be overriden in child class.
        pass

    def _update_context_from_build_task(self, build_task):
        # Does nothing by default. No need to override it without a good reason
        pass

    def on_execute(self):
        with self.memoize_stage.build_docker(commit_on_entrance=False):
            if self._parent_task_type is not None:
                if self.Parameters.parent_task is not None:
                    self._parent_task = self.Parameters.parent_task
                else:
                    self._parent_task = sdk2.Task.find(
                        task_type=self._parent_task_type,
                        release=ctt.ReleaseStatus.STABLE,
                        children=True,
                        hidden=True,
                    ).limit(0).first()

                    if not isinstance(self._parent_task, sdk2.Task):
                        raise Exception(
                            "Parent task is {}. You must have released to stable task to automatically pick it".format(self._parent_task)
                        )

                    self.set_info("Automatically selected parent task {}".format(self._parent_task))

                self.Context._parent_task = self._parent_task.id

                self.Context.save()

                self.set_info(
                    "Parent image task: {}".format(
                        link_builder.task_link(self._parent_task.id)
                    ),
                    do_escape=False
                )
            else:
                self.set_info("This is the root image. No parent needed.")

            self._set_task_tags()

            self._gen_registry_tags()

            build_task = self._create_build_task()

            build_task.enqueue()

            self.Context.build_task_id = build_task.id

        with self.memoize_stage.wait_for_build_to_end:
            raise sdk2.WaitTask(
                [self.Context.build_task_id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True
            )

        with self.memoize_stage.run_children_builds_on_success(commit_on_entrance=False):
            build_task = self.find(id=self.Context.build_task_id).first()
            if build_task.status not in ctt.Status.Group.SUCCEED:
                raise common.errors.TaskFailure(
                    "Subtask has failed with status {}".format(
                        build_task.status,
                    )
                )

            self._save_manifest_from_registry()
            for tag in self.Context.registry_tags:
                self._set_registry_tag(tag)

            self._update_context_from_build_task(build_task)

            self.set_info(
                "Built from commit {} at branch {}".format(
                    self.Context._vcs_commit.strip(),
                    self.Context._vcs_branch.strip(),
                )
            )

            for child_class in self._children_classes:
                child = child_class(
                    self,
                    description="Child of {}".format(self.id),
                    owner=self.owner,
                    parent_task=self,
                    registry_extra_tags=[
                        tag for tag in self.Context.registry_tags if str(tag) != str(self.id)
                    ],
                )
                child.enqueue()

            subtasks = self.find()
            self.Context.subtasks_ids = [subtask.id for subtask in subtasks]

        with self.memoize_stage.wait_for_children_to_finish:
            if len(list(subtasks)):
                raise sdk2.WaitTask(
                    list(subtasks),
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                    wait_all=True
                )

        with self.memoize_stage.check_status_of_run_children_tasks(commit_on_entrance=False):
            for subtask_id in self.Context.subtasks_ids:
                subtask = self.find(id=subtask_id).first()
                if subtask.status not in ctt.Status.Group.SUCCEED:
                    raise common.errors.TaskFailure(
                        "Subtask is failed with status {}".format(
                            subtask.status
                        )
                    )

    def on_release(self, parameters_):
        logging.debug("Release parameters: %r", parameters_)
        if self._parent_task_type is not None:
            parent_task = sdk2.Task.find(
                id=[self.Context._parent_task],
                children=True,
                hidden=True,
            ).limit(0).first()
            if not parent_task:
                raise Exception("Parent task is not found")
            if parent_task.status != ctt.Status.RELEASED:
                raise Exception(
                    "Parent task status is {}, must be released".format(
                        ctt.Status.RELEASED
                    )
                )
        self._set_registry_tag(parameters_["release_status"])
        self._send_release_info_to_email(parameters_)
        self.mark_released_resources(parameters_["release_status"])
