import os
import json
import logging
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.paysys.lib.docker.main import PaysysDocker


class PaysysDockerFromArcadia(PaysysDocker):
    def _update_dockerfile_from_tag(self, dockerfile_path):
        dockerfile_lines = []
        with open(dockerfile_path) as dockerfile:
            dockerfile_lines = dockerfile.readlines()

        # Remove ARG since it is incompatible with version of docker used in
        # sandbox LXC image
        if dockerfile_lines[0].startswith("ARG"):
            dockerfile_lines.pop(0)

        # Update parent image revision
        if self._parent_task_type is not None:
            for line in range(len(dockerfile_lines)):
                # Replace FROM line to the parent image tag
                if dockerfile_lines[line].startswith("FROM"):
                    dockerfile_lines[line] = "FROM {}\n".format(
                        self._parent_task.Context.built_image,
                    )
                # Replace build args since ya package tool doesn't support
                # them by itself
                for key, value in self._docker_build_args.items():
                    dockerfile_lines[line] = \
                        dockerfile_lines[line].replace("${" + key + "}", value)

        with open(dockerfile_path, "w") as dockerfile:
            dockerfile.writelines(dockerfile_lines)

    def _create_patch_from_package(self):
        arcadia_commit = "HEAD" \
            if self.Parameters.vcs_commit is None \
            else self.Parameters.vcs_commit
        arcadia_branch = "trunk/arcadia" \
            if self.Parameters.vcs_branch == "master" \
            else self.Parameters.vcs_branch

        self.Context._vcs_branch = arcadia_branch
        self.Context.save()

        self._arcadia_base_url = "arcadia:/arc/" + arcadia_branch
        self._package_dir = os.path.dirname(self._docker_package_file)

        arc_info = Arcadia.info(
            "{}/{}@{}".format(
                self._arcadia_base_url,
                self._package_dir,
                arcadia_commit,
            )
        )

        logging.info("Got info from arcadia: {}".format(arc_info))

        self.Context._vcs_commit = arc_info["commit_revision"]
        self.Context.save()

        self._arc_url = "{}/{}".format(
            self._arcadia_base_url,
            self._package_dir,
        )

        logging.info("Set arcadia url to {}".format(self._arc_url))

        Arcadia.checkout(
            "{}/{}@{}".format(
                self._arcadia_base_url,
                self._package_dir,
                self.Context._vcs_commit,
            ),
            self._package_dir,
        )

        package = None

        with open(self._docker_package_file, "r") as package_file:
            package = json.load(package_file)

        package["meta"]["version"] = str(self.id)

        # Sometimes we need to change app name to another, for example while
        # building multiple images from one package.json
        if self._docker_app_name:
            package["meta"]["name"] = self._docker_app_name

        dockerfile_path = None

        for data in package["data"]:
            if data["source"]["type"] == "RELATIVE" and \
               data["source"]["path"].startswith("Dockerfile"):
                dockerfile_path = self._package_dir + "/" + data["source"]["path"]

        if dockerfile_path is None:
            raise Exception(
                "Dockerfile not found. Perhaps it is located in another dir and it is not supported by current version of task. You add this functionality though."
            )

        with open(self._docker_package_file, "w") as package_file:
            json.dump(package, package_file, indent=4)

        self._update_dockerfile_from_tag(
            dockerfile_path
        )

        arc_diff = Arcadia.diff(self._package_dir)

        logging.info("Got arcadia diff: {}".format(arc_diff))

        return arc_diff

    def _create_build_task(self):
        arc_diff = self._create_patch_from_package()

        build_task_kwargs = {
            "checkout_arcadia_from_url": self._arcadia_base_url + "@" + self.Context._vcs_commit,
            "do_not_remove_resources": False,
            "build_type": "release",
            "packages": self._docker_package_file,
            "package_type": "docker",
            "docker_image_repository": self._docker_registry_path,
            "docker_save_image": False,
            "docker_push_image": True,
            "docker_user": self.Parameters.docker_registry_login,
            "docker_token_vault_name": self.Parameters.docker_token_vault_name,
            "run_tests": False,
            "arcadia_patch": arc_diff,
            "use_aapi_fuse": True,
        }

        requirements = {"disk_space": 10240}  # 10GB

        build_task = sdk2.Task["YA_PACKAGE"](
            self,
            requirements,
            description="Child of {}".format(self.id),
            owner=self.owner,
            **build_task_kwargs
        )

        return build_task
